#!/usr/bin/env python3
#
# @package      hubzero-submit-distributor
# @file         distributor.py
# @copyright    Copyright (c) 2004-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2004-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
"""Distribute computation jobs to remote resources

   Generate a pair of script files for remote job submission.
   The first file is the batch submission script, the second is a shell script
   that wraps the application and generates time usage data.

   RUN AS FOLLOWS:
     distributor -j <jobId> -v <venue> -i <inputfile> -o <outputfile>
                 -n <nCpus> -N <ppn> -w <wallTime> -e <environment variable>
                 -m <manager>

   <jobId>                 is the job identifier
   <executable>            is the MPI enabled executable to be run.
   <inputfile>             is a file to be sent to the remote site
   <outputfile>            is a file to be retrieved from the remote site
   <nCpus>                 the number of processors to use
   <ppn>                   the processors/node to use
   <wallTime>              wall time limit for remote process
   <environment variable>  variable=value
   <venue>                 remote venue
   <manager>               multi-processor control manager

                                                  | cpu time    |
                                                  | from time() |
                                                  |             |
                                      |  waiting  | real(wall)  |
                                      |   time    |    time     |
                                      |           |             |
     |------------|---------|---------|-----------|-------------|----------|----
  simulate     condor     site      files     simulation    simulation   files
   button        job    selected    staged     started        ended      staged
  pressed    submitted               in                                   out

   """
import os
import sys
import logging
from logging.handlers import SocketHandler,SysLogHandler

from hubzero.submit.JobDistributor import JobDistributor

LOGDIRECTORY      = os.path.join(os.sep,'var','log','submit','distributor')
APPLICATIONLOGGER = logging.getLogger('')

CONFIGURATIONDIRECTORY       = os.path.join(os.sep,'etc','submit')
DISTRIBUTORCONFIGURATIONFILE = 'distributor.conf'
DAEMONSCONFIGURATIONFILE     = 'daemons.conf'
INFOSCONFIGURATIONFILE       = 'infos.conf'

def openLogger():
   class EmptyFilter(logging.Filter):
      """
      This is a filter which rejects empty messages

      """

      def filter(self,record):
         if record.getMessage() == "":
            emptyRecord = True
         else:
            emptyRecord = False

         return(not emptyRecord)

   APPLICATIONLOGGER.setLevel(logging.DEBUG)

   logSyslogHandler = SysLogHandler(address="/dev/log",facility=SysLogHandler.LOG_LOCAL3)
   emptyFilter = EmptyFilter()
   logSyslogHandler.addFilter(emptyFilter)
   logFormatter = logging.Formatter('%(asctime)s %(message)s','%s [%a %b %d %H:%M:%S %Y]')
   logSyslogHandler.setFormatter(logFormatter)
   APPLICATIONLOGGER.addHandler(logSyslogHandler)


def distributor(configurationDirectory,
                distributorConfigurationFile,daemonsConfigurationFile,infosConfigurationFile):
   jobDistributor = JobDistributor(configurationDirectory,
                                   distributorConfigurationFile,daemonsConfigurationFile,infosConfigurationFile)
   if jobDistributor.configure():
      exitCode = jobDistributor.processCommandArguments()
      if not exitCode:
         exitCode = jobDistributor.buildJobs()
         if not exitCode:
            jobDistributor.registerJobs()
            jobDistributor.runJobs()
      exitCode = jobDistributor.reportExitCondition()
   else:
      exitCode = 1

   sys.exit(exitCode)


if __name__ == '__main__':
   openLogger()

   distributor(CONFIGURATIONDIRECTORY,
               DISTRIBUTORCONFIGURATIONFILE,DAEMONSCONFIGURATIONFILE,INFOSCONFIGURATIONFILE)


