# @package      hubzero-submit-distributor
# @file         VenueMechanismCore.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import sys
import os
import select
import subprocess
import shlex
import string
import time
import datetime
import re
import shutil
import fnmatch
import copy
import logging
from errno import EPIPE

from hubzero.submit.LogMessage   import getLogIDMessage as getLogMessage
from hubzero.submit.JobStatistic import JobStatistic

HARVESTINFORMATION = ".__harvest_information"

class VenueMechanismCore:
   def __init__(self,
                session,
                hubUserName,
                hubUserId,
                submitterClass,
                distributorPid,
                timeHistoryLogs,
                disableJobMonitoring,
                siteInfo,
                tapis2SiteInfo,
                tapis3SiteInfo,
                fileMoverInfo,
                toolFilesInfo,
                dockerImageInfo,
                submissionScriptsInfo,
                managerInfo,
                remoteMonitors,
                isMultiCoreRequest,
                nCpus,
                nNodes,
                ppn,
                event):
      self.logger                = logging.getLogger(__name__)
      self.session               = session
      self.hubUserName           = hubUserName
      self.hubUserId             = hubUserId
      self.submitterClass        = submitterClass
      self.distributorPid        = distributorPid
      self.timeHistoryLogs       = copy.copy(timeHistoryLogs)
      self.disableJobMonitoring  = disableJobMonitoring
      self.siteInfo              = copy.copy(siteInfo)
      self.tapis2SiteInfo        = copy.copy(tapis2SiteInfo)
      self.tapis3SiteInfo        = copy.copy(tapis3SiteInfo)
      self.fileMoverInfo         = copy.copy(fileMoverInfo)
      self.toolFilesInfo         = copy.copy(toolFilesInfo)
      self.dockerImageInfo       = copy.copy(dockerImageInfo)
      self.submissionScriptsInfo = copy.copy(submissionScriptsInfo)
      self.managerInfo           = copy.copy(managerInfo)
      self.remoteJobMonitor      = remoteMonitors['job']
      self.remoteTunnelMonitor   = remoteMonitors['tunnel']
      self.remoteCloudMonitor    = remoteMonitors['cloud']
      self.remoteIdentityManager = remoteMonitors['identity']

      self.childPid   = 0
      self.bufferSize = 4096

      self.remoteAppScript = None
      self.remoteBatch     = None

      self.venueMechanism    = ''
      self.remoteBatchSystem = ''
      self.enteredCommand    = ""
      self.destination       = ""
      self.venue             = ""

      self.isMultiCoreRequest = isMultiCoreRequest
      self.nCpus              = nCpus
      self.nNodes             = nNodes
      self.ppn                = ppn
      self.event              = event

      self.preStageJobInput = ""
      self.copyJobInput     = ""

      self.jobStaged     = False
      self.jobSubmitted  = False
      self.jobStatistics = {}
      self.jobIndex      = 0
      if self.isMultiCoreRequest:
         self.jobStatistics[self.jobIndex] = JobStatistic(self.nCpus)
      else:
         self.jobStatistics[self.jobIndex] = JobStatistic(1)

      self.scriptsCreated    = False
      self.filesSent         = False
      self.filesStashed      = False
      self.jobPostProcessed  = False
      self.filesRetrieved    = False
      self.filesCleanedup    = False
      self.scriptsKilled     = False
      self.instanceToken     = self.__generateToken(32)
      self.logger.log(logging.DEBUG,getLogMessage("VenueMechanismCore: token %s generated" % (self.instanceToken)))
      self.wsJobId           = os.getenv('WS_JOBID',"")
      self.runName           = ""
      self.localJobId        = ""
      self.instanceId        = "-1"
      self.instanceDirectory = ""
      self.stageInTarFile    = ""
      self.auxiliaryTarFile  = ""

      self.batchLogPath               = ""
      self.appScriptPath              = ""
      self.batchScriptPath            = ""
      self.nodeFilePath               = ""
      self.toolInputTemplateFilePath  = ""
      self.toolOutputTemplateFilePath = ""
      self.remoteJobIdNumber          = ""

      self.instancesInitialJobStatus = []


   @staticmethod
   def __generateToken(requestedTokenLength,
                       characterSet=""):
      """Generate a cleartext token using urandom, with even probability of each character.
      this is done by discarding random bytes that don't fit into an even multiple of the number
      of allowed characters. Based on maxwell middleware funcction.
      """
      if not characterSet:
         characterSet = string.ascii_letters + string.digits
      nCharacterSet = len(characterSet)
      # get numbers in a batch, it's faster than one by one
      randomCharacters = os.urandom(requestedTokenLength*2)
      tokenLength      = 0 # how many random characters we've generated so far
      iRandomCharacter = 0 # index to next unused random number in array
      token = ""
      maximumCharacterOrdinal = (256 // nCharacterSet) * nCharacterSet
      if sys.version_info > (3,):
         while tokenLength < requestedTokenLength:
            # reject results that would skew the freq. distribution of characters
            if randomCharacters[iRandomCharacter] < maximumCharacterOrdinal:
               token += characterSet[randomCharacters[iRandomCharacter] % nCharacterSet]
               tokenLength += 1
            iRandomCharacter += 1
            if iRandomCharacter >= requestedTokenLength*2:
               # we've run out of random numbers, get some more
               randomCharacters = os.urandom(requestedTokenLength*2)
               iRandomCharacter = 0
      else:
         while tokenLength < requestedTokenLength:
            # reject results that would skew the freq. distribution of characters
            if ord(randomCharacters[iRandomCharacter]) < maximumCharacterOrdinal:
               token += characterSet[ord(randomCharacters[iRandomCharacter]) % nCharacterSet]
               tokenLength += 1
            iRandomCharacter += 1
            if iRandomCharacter >= requestedTokenLength*2:
               # we've run out of random numbers, get some more
               randomCharacters = os.urandom(requestedTokenLength*2)
               iRandomCharacter = 0

      return(token)


   def _writeToStdout(self,
                      message):
      try:
         sys.stdout.write(message)
         sys.stdout.flush()
      except IOError as e:
         if not e.args[0] in [EPIPE]:
            self.logger.log(logging.ERROR,getLogMessage("Can't write to stdout: %s" % (message)))


   def _writeToStderr(self,
                      message):
      try:
         sys.stderr.write(message)
         sys.stderr.flush()
      except IOError as e:
         if not e.args[0] in [EPIPE]:
            self.logger.log(logging.ERROR,getLogMessage("Can't write to stderr: %s" % (message)))


   def executeCommand(self,
                      command,
                      streamOutput=False):
      child = subprocess.Popen(command,shell=True,bufsize=self.bufferSize,
                               stdout=subprocess.PIPE,
                               stderr=subprocess.PIPE,
                               close_fds=True)
      self.childPid = child.pid
      childout      = child.stdout
      childoutFd    = childout.fileno()
      childerr      = child.stderr
      childerrFd    = childerr.fileno()

      outEOF = False
      errEOF = False

      outData = []
      errData = []

      while True:
         toCheck = []
         if not outEOF:
            toCheck.append(childoutFd)
         if not errEOF:
            toCheck.append(childerrFd)
         try:
            ready = select.select(toCheck,[],[]) # wait for input
         except select.error:
            ready = {}
            ready[0] = []
         if childoutFd in ready[0]:
            outChunk = os.read(childoutFd,self.bufferSize).decode('utf-8')
            if outChunk == '':
               outEOF = True
            outData.append(outChunk)
            if streamOutput:
               self._writeToStdout(outChunk)

         if childerrFd in ready[0]:
            errChunk = os.read(childerrFd,self.bufferSize).decode('utf-8')
            if errChunk == '':
               errEOF = True
            errData.append(errChunk)
            if streamOutput:
               self._writeToStderr(errChunk)

         if outEOF and errEOF:
            break

      pid,err = os.waitpid(self.childPid,0)
      self.childPid = 0
      if err != 0:
         if   os.WIFSIGNALED(err):
            self.logger.log(logging.ERROR,getLogMessage("%s failed w/ signal %d" % (command,os.WTERMSIG(err))))
         else:
            if os.WIFEXITED(err):
               err = os.WEXITSTATUS(err)
            self.logger.log(logging.ERROR,getLogMessage("%s failed w/ exit code %d" % (command,err)))
         if not streamOutput:
            self.logger.log(logging.ERROR,getLogMessage("%s" % ("".join(errData))))

      return(err,"".join(outData),"".join(errData))


   def executeSSHCommand(self,
                         sshCommand,
                         tunnelDesignator,
                         streamOutput=False):

      minimumDelay = 1       #  1 2 4 8 16 32 64 128 256
      maximumDelay = 256
      updateFrequency = 1
      maximumDelayTime = 900

      delayTime = 0
      sleepTime = minimumDelay
      nDelays = 0

      if tunnelDesignator != "":
         self.remoteTunnelMonitor.incrementTunnelUse(tunnelDesignator)
      exitStatus,stdOutput,stdError = self.executeCommand(sshCommand,streamOutput)

      while exitStatus and (stdError.count("You don't exist, go away!") > 0):
         nDelays += 1
         time.sleep(sleepTime)
         delayTime += sleepTime
         if nDelays == updateFrequency:
            nDelays = 0
            sleepTime *= 2
            if sleepTime > maximumDelay:
               sleepTime = maximumDelay

         exitStatus,stdOutput,stdError = self.executeCommand(sshCommand,streamOutput)

         if delayTime >= maximumDelayTime:
            break

      if tunnelDesignator != "":
         self.remoteTunnelMonitor.decrementTunnelUse(tunnelDesignator)

      return(exitStatus,stdOutput,stdError)


   def executeLaunchCommand(self,
                            launchCommand,
                            streamOutput=False):

      minimumDelay = 1       #  1 2 4 8 16 32 64 128 256
      maximumDelay = 256
      updateFrequency = 1
      maximumDelayTime = 900

      delayTime = 0
      sleepTime = minimumDelay
      nDelays = 0
      exitStatus,stdOutput,stdError = self.executeCommand(launchCommand,streamOutput)

      while exitStatus and ((stdError.count("qsub: cannot connect to server") > 0) or (stdError.count("ldap-nss.c") > 0)):
         nDelays += 1
         time.sleep(sleepTime)
         delayTime += sleepTime
         if nDelays == updateFrequency:
            nDelays = 0
            sleepTime *= 2
            if sleepTime > maximumDelay:
               sleepTime = maximumDelay

         exitStatus,stdOutput,stdError = self.executeCommand(launchCommand,streamOutput)

         if delayTime >= maximumDelayTime:
            break

      return(exitStatus,stdOutput,stdError)


   def updateVenue(self,
                   executionVenue):
      if self.remoteBatchSystem == 'FACTORY':
         self.jobStatistics[self.jobIndex]['venue'] = executionVenue
      if self.remoteBatchSystem == 'PEGASUS':
         self.jobStatistics[self.jobIndex]['venue'] = executionVenue


   def getVenue(self):
      executionVenue = self.jobStatistics[self.jobIndex]['venue']

      return(executionVenue)


   def copyJobInputs(self):
      if   self.fileMoverInfo['fileMoverType'] == 'tapis2':
         self.filesStashed = True
         self.jobStatistics[self.jobIndex]['venue'] = self.venue

         command = self.copyJobInput + " " + self.instanceDirectory + " " + \
                                             self.fileMoverInfo['tapisStorageSystem'] + " " + \
                                             self.instanceToken + " " + \
                                             os.path.join(self.instanceDirectory,self.timeHistoryLogs['timestampInputStaged'])
         self.logger.log(logging.INFO,getLogMessage("command = " + command))
         tStart = time.time()
         exitStatus,stdOutput,stdError = self.executeCommand(command)
         tFinish = time.time()
         self.logger.log(logging.INFO,getLogMessage(stdOutput))
#        self.logger.log(logging.DEBUG,getLogMessage("copyJobInputs: Tapis2 copy took %f secs" % (tFinish-tStart)))

         if exitStatus:
            self.filesStashed = False
            self.jobStatistics[self.jobIndex]['exitCode'] = 11
#           if stdOutput != "":
#              if self.instanceId != "0":
#                 stdFile = os.path.join(self.instanceDirectory,"%s_%s.stdout" % (self.runName,self.instanceId))
#              else:
#                 stdFile = os.path.join(self.instanceDirectory,"%s.stdout" % (self.runName))
#              try:
#                 fpStd = open(stdFile,'a')
#                 try:
#                    fpStd.write(stdOutput)
#                 except (IOError,OSError):
#                    self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
#                 finally:
#                    fpStd.close()
#              except (IOError,OSError):
#                 self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))
   
            if stdError != "":
               if self.instanceId != "0":
                  stdFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.runName,self.instanceId))
               else:
                  stdFile = os.path.join(self.instanceDirectory,"%s.stderr" % (self.runName))
               try:
                  fpStd = open(stdFile,'a')
                  try:
                     fpStd.write(stdError)
                  except (IOError,OSError):
                     self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
                  finally:
                     fpStd.close()
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))
            self.removeJobRegistration()
         else:
            self.jobStatistics[self.jobIndex]['exitCode'] = 0
      elif self.fileMoverInfo['fileMoverType'] == 'tapis3':
         self.filesStashed = True
         self.jobStatistics[self.jobIndex]['venue'] = self.venue

         command = self.copyJobInput + " " + self.instanceDirectory + " " + \
                                             self.fileMoverInfo['tapisStorageSystem'] + " " + \
                                             self.instanceToken + " " + \
                                             os.path.join(self.instanceDirectory,self.timeHistoryLogs['timestampInputStaged'])
         self.logger.log(logging.INFO,getLogMessage("command = " + command))
         tStart = time.time()
         exitStatus,stdOutput,stdError = self.executeCommand(command)
         tFinish = time.time()
         self.logger.log(logging.INFO,getLogMessage(stdOutput))
#        self.logger.log(logging.DEBUG,getLogMessage("copyJobInputs: Tapis3 copy took %f secs" % (tFinish-tStart)))

         if exitStatus:
            self.filesStashed = False
            self.jobStatistics[self.jobIndex]['exitCode'] = 11
#           if stdOutput != "":
#              if self.instanceId != "0":
#                 stdFile = os.path.join(self.instanceDirectory,"%s_%s.stdout" % (self.runName,self.instanceId))
#              else:
#                 stdFile = os.path.join(self.instanceDirectory,"%s.stdout" % (self.runName))
#              try:
#                 fpStd = open(stdFile,'a')
#                 try:
#                    fpStd.write(stdOutput)
#                 except (IOError,OSError):
#                    self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
#                 finally:
#                    fpStd.close()
#              except (IOError,OSError):
#                 self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))
   
            if stdError != "":
               if self.instanceId != "0":
                  stdFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.runName,self.instanceId))
               else:
                  stdFile = os.path.join(self.instanceDirectory,"%s.stderr" % (self.runName))
               try:
                  fpStd = open(stdFile,'a')
                  try:
                     fpStd.write(stdError)
                  except (IOError,OSError):
                     self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
                  finally:
                     fpStd.close()
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))
            self.removeJobRegistration()
         else:
            self.jobStatistics[self.jobIndex]['exitCode'] = 0
      else:
         self.filesStashed = False

      return(self.filesStashed)


   def preStageJobInputs(self):
      if   self.fileMoverInfo['fileMoverType'] == 'tapis2':
         self.filesStashed = True
         self.jobStatistics[self.jobIndex]['venue'] = self.venue

         command = self.preStageJobInput + " " + self.instanceDirectory + " " + \
                                                 self.fileMoverInfo['preStageDirectory'] + " " + \
                                                 self.instanceToken + " " + \
                                                 os.path.join(self.instanceDirectory,self.timeHistoryLogs['timestampInputStaged'])
         self.logger.log(logging.INFO,getLogMessage("command = " + command))
         tStart = time.time()
         exitStatus,stdOutput,stdError = self.executeCommand(command)
         tFinish = time.time()
         self.logger.log(logging.INFO,getLogMessage(stdOutput))
#        self.logger.log(logging.DEBUG,getLogMessage("preStageJobInputs: Tapis2 prestage took %f secs" % (tFinish-tStart)))

         if exitStatus:
            self.filesStashed = False
            self.jobStatistics[self.jobIndex]['exitCode'] = 11
#           if stdOutput != "":
#              if self.instanceId != "0":
#                 stdFile = os.path.join(self.instanceDirectory,"%s_%s.stdout" % (self.runName,self.instanceId))
#              else:
#                 stdFile = os.path.join(self.instanceDirectory,"%s.stdout" % (self.runName))
#              try:
#                 fpStd = open(stdFile,'a')
#                 try:
#                    fpStd.write(stdOutput)
#                 except (IOError,OSError):
#                    self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
#                 finally:
#                    fpStd.close()
#              except (IOError,OSError):
#                 self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))
   
            if stdError != "":
               if self.instanceId != "0":
                  stdFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.runName,self.instanceId))
               else:
                  stdFile = os.path.join(self.instanceDirectory,"%s.stderr" % (self.runName))
               try:
                  fpStd = open(stdFile,'a')
                  try:
                     fpStd.write(stdError)
                  except (IOError,OSError):
                     self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
                  finally:
                     fpStd.close()
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))
            self.removeJobRegistration()
         else:
            self.jobStatistics[self.jobIndex]['exitCode'] = 0
      elif self.fileMoverInfo['fileMoverType'] == 'tapis3':
         self.filesStashed = True
         self.jobStatistics[self.jobIndex]['venue'] = self.venue

         command = self.preStageJobInput + " " + self.instanceDirectory + " " + \
                                                 self.fileMoverInfo['preStageDirectory'] + " " + \
                                                 self.instanceToken + " " + \
                                                 os.path.join(self.instanceDirectory,self.timeHistoryLogs['timestampInputStaged'])
         self.logger.log(logging.INFO,getLogMessage("command = " + command))
         tStart = time.time()
         exitStatus,stdOutput,stdError = self.executeCommand(command)
         tFinish = time.time()
         self.logger.log(logging.INFO,getLogMessage(stdOutput))
#        self.logger.log(logging.DEBUG,getLogMessage("preStageJobInputs: Tapis3 prestage took %f secs" % (tFinish-tStart)))

         if exitStatus:
            self.filesStashed = False
            self.jobStatistics[self.jobIndex]['exitCode'] = 11
#           if stdOutput != "":
#              if self.instanceId != "0":
#                 stdFile = os.path.join(self.instanceDirectory,"%s_%s.stdout" % (self.runName,self.instanceId))
#              else:
#                 stdFile = os.path.join(self.instanceDirectory,"%s.stdout" % (self.runName))
#              try:
#                 fpStd = open(stdFile,'a')
#                 try:
#                    fpStd.write(stdOutput)
#                 except (IOError,OSError):
#                    self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
#                 finally:
#                    fpStd.close()
#              except (IOError,OSError):
#                 self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))
   
            if stdError != "":
               if self.instanceId != "0":
                  stdFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.runName,self.instanceId))
               else:
                  stdFile = os.path.join(self.instanceDirectory,"%s.stderr" % (self.runName))
               try:
                  fpStd = open(stdFile,'a')
                  try:
                     fpStd.write(stdError)
                  except (IOError,OSError):
                     self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
                  finally:
                     fpStd.close()
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))
            self.removeJobRegistration()
         else:
            self.jobStatistics[self.jobIndex]['exitCode'] = 0
      elif self.fileMoverInfo['fileMoverType'] == 'ssh':
         self.filesStashed = True
         self.jobStatistics[self.jobIndex]['venue'] = self.venue

         command = self.preStageJobInput + " " + self.instanceDirectory + " " + \
                                                 self.fileMoverInfo['preStageDirectory'] + " " + \
                                                 self.instanceToken + " " + \
                                                 os.path.join(self.instanceDirectory,self.timeHistoryLogs['timestampInputStaged'])
         self.logger.log(logging.INFO,getLogMessage("command = " + command))
         tStart = time.time()
         exitStatus,stdOutput,stdError = self.executeCommand(command)
         tFinish = time.time()
         self.logger.log(logging.INFO,getLogMessage(stdOutput))
         self.logger.log(logging.DEBUG,getLogMessage("preStageJobInputs: SSH prestage took %f secs" % (tFinish-tStart)))

         if exitStatus:
            self.filesStashed = False
            self.jobStatistics[self.jobIndex]['exitCode'] = 11
#           if stdOutput != "":
#              if self.instanceId != "0":
#                 stdFile = os.path.join(self.instanceDirectory,"%s_%s.stdout" % (self.runName,self.instanceId))
#              else:
#                 stdFile = os.path.join(self.instanceDirectory,"%s.stdout" % (self.runName))
#              try:
#                 fpStd = open(stdFile,'a')
#                 try:
#                    fpStd.write(stdOutput)
#                 except (IOError,OSError):
#                    self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
#                 finally:
#                    fpStd.close()
#              except (IOError,OSError):
#                 self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))
   
            if stdError != "":
               if self.instanceId != "0":
                  stdFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.runName,self.instanceId))
               else:
                  stdFile = os.path.join(self.instanceDirectory,"%s.stderr" % (self.runName))
               try:
                  fpStd = open(stdFile,'a')
                  try:
                     fpStd.write(stdError)
                  except (IOError,OSError):
                     self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
                  finally:
                     fpStd.close()
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))
            self.removeJobRegistration()
         else:
            self.jobStatistics[self.jobIndex]['exitCode'] = 0
      else:
         self.filesStashed = False

      return(self.filesStashed)

 
   def stashJobInputs(self):
#     self.logger.log(logging.DEBUG,getLogMessage("stashJobInputs: fileMoverType: %s" % (self.fileMoverInfo['fileMoverType'])))
      if self.fileMoverInfo['preStageDirectory']:
         filesStashed = self.preStageJobInputs()
      else:
         filesStashed = self.copyJobInputs()
      if not filesStashed:
         self.logger.log(logging.ERROR,getLogMessage("stashJobInputs: failed"))

      return(filesStashed)


   def stageJob(self):
      exitStatus = 0

#     self.logger.log(logging.DEBUG,getLogMessage("stageJob: fileMoverType: %s" % (self.fileMoverInfo['fileMoverType'])))
      if   self.fileMoverInfo['fileMoverType'] == 'tapis2':
         self.jobStaged = True
         self.jobStatistics[self.jobIndex]['exitCode'] = 2
         self.remoteJobMonitor.postJobStaging(self.session,self.instanceToken,self.wsJobId,
                                              self.siteInfo['siteName'],self.siteInfo['identityManagers'],
                                              self.siteMonitorDesignator,
                                              self.fileMoverInfo['preStageDirectory'],self.fileMoverInfo['tapisStorageSystem'],
                                              str(self.hubUserId),
                                              self.enteredCommand,
                                              self.scratchDirectory,
                                              self.localJobId,self.instanceId,self.destination,
                                              self.runName,self.nCpus,self.distributorPid,
                                              self.jobStatistics[self.jobIndex])
         self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus
      elif self.fileMoverInfo['fileMoverType'] == 'tapis3':
         self.jobStaged = True
         self.jobStatistics[self.jobIndex]['exitCode'] = 2
         self.remoteJobMonitor.postJobStaging(self.session,self.instanceToken,self.wsJobId,
                                              self.siteInfo['siteName'],self.siteInfo['identityManagers'],
                                              self.siteMonitorDesignator,
                                              self.fileMoverInfo['preStageDirectory'],self.fileMoverInfo['tapisStorageSystem'],
                                              str(self.hubUserId),
                                              self.enteredCommand,
                                              self.scratchDirectory,
                                              self.localJobId,self.instanceId,self.destination,
                                              self.runName,self.nCpus,self.distributorPid,
                                              self.jobStatistics[self.jobIndex])
         self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus
      elif self.fileMoverInfo['fileMoverType'] == 'ssh':
         self.jobStaged = True
         self.jobStatistics[self.jobIndex]['exitCode'] = 2
         self.remoteJobMonitor.postJobStaging(self.session,self.instanceToken,self.wsJobId,
                                              self.siteInfo['siteName'],self.siteInfo['identityManagers'],
                                              self.siteMonitorDesignator,
                                              self.fileMoverInfo['preStageDirectory'],"None",
                                              str(self.hubUserId),
                                              self.enteredCommand,
                                              self.scratchDirectory,
                                              self.localJobId,self.instanceId,self.destination,
                                              self.runName,self.nCpus,self.distributorPid,
                                              self.jobStatistics[self.jobIndex])
         self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus

      if not self.jobStaged:
         self.logger.log(logging.ERROR,getLogMessage("stageJob: failed"))

      return(self.jobStaged)


   def recordJobStatisticTime(self,
                              statistic,
                              timeFileBasename):
      timePath = os.path.join(self.instanceDirectory,timeFileBasename)
      if os.path.exists(timePath):
         jobIndex = int(self.instanceId)

         if not jobIndex in self.jobStatistics:
            self.jobStatistics[jobIndex] = JobStatistic(self.nCpus)
            self.logger.log(logging.DEBUG,getLogMessage("recordJobStatisticTime: added missing jobIndex %d" % (jobIndex)))

         self.jobStatistics[jobIndex].recordTime(statistic,timePath)
      else:
         self.logger.log(logging.ERROR,getLogMessage(timeFileBasename + " not present in " + self.instanceDirectory))


   def recordJobStatisticTimer(self,
                               timeFileBasename):
      timerPath = os.path.join(self.instanceDirectory,timeFileBasename)
      if os.path.exists(timerPath):
         jobIndex = int(self.instanceId)

         if not jobIndex in self.jobStatistics:
            self.jobStatistics[jobIndex] = JobStatistic(self.nCpus)
            self.logger.log(logging.DEBUG,getLogMessage("recordJobStatisticTimer: added missing jobIndex %d" % (jobIndex)))

         self.jobStatistics[jobIndex].recordTimer(timerPath)
      else:
         if self.jobSubmitted:
            self.logger.log(logging.ERROR,getLogMessage(timeFileBasename + " not present in " + self.instanceDirectory))

# files of this form are generated by non parametric dax execution
      reFiles = re.compile(timeFileBasename + "_[0-9]+$")
      dirFiles = os.listdir(self.instanceDirectory)
      matchingFiles = list(filter(reFiles.search,dirFiles))
      if len(matchingFiles) > 0:
         maximumJobIndex = max(self.jobStatistics.keys())
         jobIndex = maximumJobIndex
         for matchingFile in matchingFiles:
            jobIndex += 1
            self.jobStatistics[jobIndex] = JobStatistic(1)
            timerPath = os.path.join(self.instanceDirectory,matchingFile)
            self.jobStatistics[jobIndex].recordTimer(timerPath)


   def recordJobStatistics(self):
      self.recordJobStatisticTimer(self.timeHistoryLogs['timeResults'])

      if self.jobSubmitted:
         self.recordJobStatisticTime('transferCompleteTime',self.timeHistoryLogs['timestampTransferred'])
         self.recordJobStatisticTime('jobStartedTime',self.timeHistoryLogs['timestampStart'])
         self.recordJobStatisticTime('jobFinshedTime',self.timeHistoryLogs['timestampFinish'])

      for jobIndex in self.jobStatistics:
         self.jobStatistics[jobIndex].setWaitingTime()
         self.jobStatistics[jobIndex].setElapsedRunTime()


   def wasJobSuccessful(self):
      success = False
      for jobIndex in self.jobStatistics:
         if self.jobStatistics[jobIndex]['exitCode'] == 0:
            success = True
            break

      return(success)


   def removeJobStatisticsFiles(self):
      for timeFileBasename in self.timeHistoryLogs['timeResults'], \
                              self.timeHistoryLogs['timestampTransferred'], \
                              self.timeHistoryLogs['timestampStart'], \
                              self.timeHistoryLogs['timestampFinish']:
         reFiles = re.compile(timeFileBasename + "(_[0-9]+)?$")
         dirFiles = os.listdir(self.instanceDirectory)
         matchingFiles = list(filter(reFiles.search,dirFiles))
         if len(matchingFiles) > 0:
            for matchingFile in matchingFiles:
               try:
                  os.remove(os.path.join(self.instanceDirectory,matchingFile))
               except:
                  pass


   def stageFilesToInstanceDirectory(self,
                                     stageAuxiliaryTarFile=False):
      if self.instanceDirectory != "":
         try:
            currentWorkingDirectory = os.getcwd()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("stageFilesToInstanceDirectory:os.getcwd(): No such file or directory"))
         else:
            if self.instanceDirectory != currentWorkingDirectory:
               if os.path.isdir(self.instanceDirectory):
                  if self.stageInTarFile != "":
                     stageInTarFile = self.stageInTarFile
                     srcPath = os.path.join(currentWorkingDirectory,stageInTarFile)
                     if os.path.exists(srcPath):
                        dstPath = os.path.join(self.instanceDirectory,stageInTarFile)
                        os.rename(srcPath,dstPath)
                     else:
                        self.logger.log(logging.DEBUG,getLogMessage("stageFilesToInstanceDirectory: missing stageInTarFile"))

                  if stageAuxiliaryTarFile and self.auxiliaryTarFile != "":
                     auxiliaryTarFile = self.auxiliaryTarFile
                     srcPath = os.path.join(currentWorkingDirectory,auxiliaryTarFile)
                     if os.path.exists(srcPath):
                        dstPath = os.path.join(self.instanceDirectory,auxiliaryTarFile)
                        os.rename(srcPath,dstPath)
                     else:
                        self.logger.log(logging.DEBUG,getLogMessage("stageFilesToInstanceDirectory: missing auxiliaryTarFile"))


   def registerJob(self):
      self.remoteJobMonitor.postJobRegistration(self.session,self.instanceToken,self.wsJobId,self.siteInfo['siteName'],
                                                self.runName,self.localJobId,self.instanceId,
                                                str(self.hubUserId),self.submitterClass,
                                                self.distributorPid)


   def isJobReleased(self):
      jobReleased = self.remoteJobMonitor.isJobReleased(self.localJobId,self.instanceId)

      return(jobReleased)


   def removeJobRegistration(self):
      self.remoteJobMonitor.deleteJobRegistration(self.localJobId,self.instanceId)


   def removeInstanceDirectory(self):
      try:
         currentWorkingDirectory = os.getcwd()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("removeInstanceDirectory:os.getcwd(): No such file or directory"))
      else:
         if self.instanceDirectory != currentWorkingDirectory:
            shutil.rmtree(self.instanceDirectory,True)
            jobDirectory = os.path.dirname(self.instanceDirectory)
            if jobDirectory != currentWorkingDirectory:
               try:
                  os.rmdir(jobDirectory)
               except:
                  pass


   def moveTree(self,
                src,
                dst,
                symlinks=False):
      if os.path.isdir(src):
         if os.path.exists(dst):
            if not os.path.isdir(dst):
               self.logger.log(logging.ERROR,getLogMessage("moveTree: %s must be a directory" % (dst)))
               return
         else:
            os.mkdir(dst)
         names = os.listdir(src)
         for name in names:
            srcPath = os.path.join(src,name)
            dstPath = os.path.join(dst,name)
            try:
               if symlinks and os.path.islink(srcPath):
                  linkto = os.readlink(srcPath)
                  os.symlink(linkto,dstPath)
               elif os.path.isdir(srcPath):
                  self.moveTree(srcPath,dstPath,symlinks)
               else:
                  os.rename(srcPath,dstPath)
            except (IOError,os.error) as why:
               self.logger.log(logging.ERROR,getLogMessage("moveTree: Can't move %s to %s: %s" % (srcPath,dstPath,str(why))))
      else:
         self.logger.log(logging.ERROR,getLogMessage("moveTree: %s must be a directory" % (src)))


   def recoverFiles(self,
                    ingoreInstanceFiles):
      try:
         currentWorkingDirectory = os.getcwd()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("recoverFiles:os.getcwd(): No such file or directory"))
      else:
         if os.path.exists(self.instanceDirectory):
            if os.path.isdir(self.instanceDirectory):
               if self.instanceDirectory != currentWorkingDirectory:
                  dirFiles = os.listdir(self.instanceDirectory)
                  if ingoreInstanceFiles:
                     ignoreFiles = fnmatch.filter(dirFiles,"*%s_%s*" % (self.runName,self.instanceId))
                     for ignoreFile in ignoreFiles:
                        dirFiles.remove(ignoreFile)

                  for dirFile in dirFiles:
                     srcFile = os.path.join(self.instanceDirectory,dirFile)
                     dstFile = os.path.join(currentWorkingDirectory,dirFile)
                     try:
                        if os.path.isdir(srcFile):
                           self.moveTree(srcFile,dstFile)
                        else:
                           os.rename(srcFile,dstFile)
                     except:
                        self.logger.log(logging.ERROR,getLogMessage("%s recovery failed" % (srcFile)))


   def recoverStdFiles(self):
      try:
         currentWorkingDirectory = os.getcwd()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("recoverStdFiles:os.getcwd(): No such file or directory"))
      else:
         if os.path.isdir(self.instanceDirectory):
            if self.instanceDirectory != currentWorkingDirectory:
               for fileType in 'stderr','stdout','FAILURE':
                  dirFiles = os.listdir(self.instanceDirectory)
                  matchingFiles = fnmatch.filter(dirFiles,"*%s_%s.%s" % (self.runName,self.instanceId,fileType))
                  if len(matchingFiles) > 0:
                     matchingFiles.sort(reverse=True)
                     fileToAppend = "%s.%s" % (self.runName,fileType)
                     try:
                        fpAppend = open(fileToAppend,'ab')
                     except (IOError,OSError):
                        self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (fileToAppend)))
                     else:
                        for matchingFile in matchingFiles:
                           fileToRecover = os.path.join(self.instanceDirectory,matchingFile)
                           try:
                              fpRecover = open(fileToRecover,'rb')
                           except (IOError,OSError):
                              self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (fileToRecover)))
                           else:
                              try:
                                 shutil.copyfileobj(fpRecover,fpAppend)
                              except:
                                 self.logger.log(logging.ERROR,getLogMessage("%s could not be appended to %s" % \
                                                                                 (fileToRecover,fileToAppend)))
                              finally:
                                 fpRecover.close()
                        fpAppend.close()


   def cleanupPegasusFiles(self):
      for ftype in '.dax','_sites.xml','_sites.yml','.pegasusrc','_tc.txt','_tc.yml':
         pegasusFile = os.path.join(self.instanceDirectory,"%s_%s%s" % (self.localJobId,self.instanceId,ftype))
         if os.path.isfile(pegasusFile):
            try:
               os.remove(pegasusFile)
            except:
               pass
         else:
            pegasusFile = os.path.join(self.instanceDirectory,"%s%s" % (self.localJobId,ftype))
            if os.path.isfile(pegasusFile):
               try:
                  os.remove(pegasusFile)
               except:
                  pass

      pegasusFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.localJobId,self.instanceId))
      if os.path.exists(pegasusFile):
         if(os.path.getsize(pegasusFile) == 0):
            try:
               os.remove(pegasusFile)
            except:
               pass


   def cleanupScriptTemplateLogFiles(self):
      for scriptTemplateLogFile in [self.batchScriptPath,self.appScriptPath,self.nodeFilePath,
                                    self.toolInputTemplateFilePath,self.toolOutputTemplateFilePath,
                                    self.batchLogPath]:
         if os.path.isfile(scriptTemplateLogFile):
            os.remove(scriptTemplateLogFile)


   def cleanupStageInTarFile(self):
      stageInTarFile = os.path.join(self.instanceDirectory,self.stageInTarFile)
      if os.path.isfile(stageInTarFile):
         try:
            os.remove(stageInTarFile)
         except:
            pass

      stageInTarFile += '.gz'
      if os.path.isfile(stageInTarFile):
         try:
            os.remove(stageInTarFile)
         except:
            pass


   def cleanupStageOutTarFile(self):
      stageOutTarFile = "%s_%s_output.tar" % (self.localJobId,self.instanceId)
      if os.path.isfile(stageOutTarFile):
         try:
            os.remove(stageOutTarFile)
         except:
            pass

      stageOutTarFile += '.gz'
      if os.path.isfile(stageOutTarFile):
         try:
            os.remove(stageOutTarFile)
         except:
            pass


   def setInstancesInitialJobStatus(self,
                                    instancesInitialJobStatus):
      self.instancesInitialJobStatus = instancesInitialJobStatus


   def writeHarvestInformationFile(self,
                                   runType=None,
                                   filesToRemove=None,
                                   emptyFilesToRemove=None):
      harvestInformationFile = "%s.%s_%s" % (HARVESTINFORMATION,self.localJobId,self.instanceId)
      harvestInformationPath = os.path.join(self.instanceDirectory,harvestInformationFile)
      try:
         fpHarvestInformationFile = open(harvestInformationPath,'w')
         try:
            if runType:
               fpHarvestInformationFile.write("runType = %s\n" % (runType))
            fpHarvestInformationFile.write("venueMechanism = %s\n" % (self.venueMechanism))
            fpHarvestInformationFile.write("localJobId = %s\n" % (self.localJobId))
            fpHarvestInformationFile.write("instanceId = %s\n" % (self.instanceId))
            fpHarvestInformationFile.write("nInstances = %d\n" % (self.nInstances))
            fpHarvestInformationFile.write("runName = %s\n" % (self.runName))
            fpHarvestInformationFile.write("isParametric = %s\n" % (self.isParametric))
            fpHarvestInformationFile.write("isMultiCoreRequest = %s\n" % (self.isMultiCoreRequest))
            fpHarvestInformationFile.write("nCpus = %d\n" % (self.nCpus))

            if self.jobStatistics[self.jobIndex]['event']:
               fpHarvestInformationFile.write("event = %s\n" % (self.jobStatistics[self.jobIndex]['event']))
            if self.destination:
               fpHarvestInformationFile.write("destination = %s\n" % (self.destination))
            if self.jobStatistics[self.jobIndex]['venue']:
               fpHarvestInformationFile.write("venue = %s\n" % (self.jobStatistics[self.jobIndex]['venue']))
            if self.jobStatistics[self.jobIndex]['remoteJobIdNumber']:
               fpHarvestInformationFile.write("remoteJobIdNumber = %s\n" % (self.jobStatistics[self.jobIndex]['remoteJobIdNumber']))
            if self.stageInTarFile:
               fpHarvestInformationFile.write("stageInTarFile = %s\n" % (self.stageInTarFile))
            if self.batchLogPath:
               fpHarvestInformationFile.write("batchLogPath = %s\n" % (self.batchLogPath))
            if self.appScriptPath:
               fpHarvestInformationFile.write("appScriptPath = %s\n" % (self.appScriptPath))
            if self.batchScriptPath:
               fpHarvestInformationFile.write("batchScriptPath = %s\n" % (self.batchScriptPath))
            if self.nodeFilePath:
               fpHarvestInformationFile.write("nodeFilePath = %s\n" % (self.nodeFilePath))
            if self.toolInputTemplateFilePath:
               fpHarvestInformationFile.write("toolInputTemplateFilePath = %s\n" % (self.toolInputTemplateFilePath))
            if self.toolOutputTemplateFilePath:
               fpHarvestInformationFile.write("toolOutputTemplateFilePath = %s\n" % (self.toolOutputTemplateFilePath))
            if filesToRemove:
               fpHarvestInformationFile.write("filesToRemove = %s\n" % (','.join(filesToRemove)))
            if emptyFilesToRemove:
               fpHarvestInformationFile.write("emptyFilesToRemove = %s\n" % (','.join(emptyFilesToRemove)))
            if self.x509SubmitProxy:
               fpHarvestInformationFile.write("x509SubmitProxy = %s\n" % (self.x509SubmitProxy))
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (harvestInformationFile)))
            self._writeToStderr("%s could not be written\n" % (harvestInformationFile))
         finally:
            fpHarvestInformationFile.close()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (harvestInformationFile)))
         self._writeToStderr("%s could not be opened\n" % (harvestInformationFile))


