# @package      hubzero-submit-distributor
# @file         GroupMembership.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
"""Functions for determining if user is a member of a particular group.
   """
import os
import pwd
import grp
import logging

from hubzero.submit.LogMessage import getLogJobIdMessage as getLogMessage

class GroupMembership:
   def __init__(self,
                restrictionUser=None):
      self.logger = logging.getLogger(__name__)

      if restrictionUser:
#
# For Python > 3.3
#        login,pw,uid,gid,name,homedir,shell = pwd.getpwnam(restrictionUser)
#        self.memberGroupIds = os.getgrouplist(restrictionUser,gid)
#
         self.memberGroupIds = [group.gr_gid for group in grp.getgrall() if restrictionUser in group.gr_mem]
         login,pw,uid,gid,name,homedir,shell = pwd.getpwnam(restrictionUser)
         if not gid in self.memberGroupIds:
            self.memberGroupIds.append(gid)
      else:
         self.memberGroupIds = os.getgroups()


   def isGroupMember(self,
                     groupName):
      """
      Is user a member of group groupName?
      """
      groupMember = False
      try:
         _,_,requestedGroupId,_ = grp.getgrnam(groupName)
      except:
         logger = logging.getLogger(__name__)
         logger.log(logging.ERROR,getLogMessage("Unable to get gid of '%s'" % (groupName)))
         return False
      else:
         groupMember = requestedGroupId in self.memberGroupIds

      return groupMember


