#!/usr/bin/env python
#
# @package      hubzero-submit-monitors
# @file         monitorJobR.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2004-2014 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2004-2014 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
# ----------------------------------------------------------------------
#  monitorJobR.py
#
#  script which queries monitorJob.py for active job(s) at site
#
import sys
import time
import re
import logging

from hubzero.submit.RemoteJobMonitor import RemoteJobMonitor

JOBMONITORHOST = ""
JOBMONITORPORT = 5727

APPLICATIONLOGGER = logging.getLogger('') 

def openLogger():
   class EmptyFilter(logging.Filter):
      """
      This is a filter which rejects empty messages

      """

      def filter(self,record):
         if record.getMessage() == "":
            emptyRecord = True
         else:
            emptyRecord = False

         return(not emptyRecord)

   APPLICATIONLOGGER.setLevel(logging.DEBUG)

   logHandler = logging.StreamHandler()

   emptyFilter = EmptyFilter()
   logHandler.addFilter(emptyFilter)

   logFormatter = logging.Formatter('%(asctime)s %(message)s','[%a %b %d %H:%M:%S %Y]')
   logHandler.setFormatter(logFormatter)
   APPLICATIONLOGGER.addHandler(logHandler)


def natsort(l):
   def tryfloat(s):
      try:
         return int(s)
      except:
         return s.lower()

   def alphanum_key(s):
      """ Turn a string into a list of string and number chunks.
          "z23a" -> ["z", 23, "a"]
      """
      return [ tryfloat(c) for c in re.split('([0-9.]+)', s) ]

   """ Sort the given list in the way that humans expect.
   """
   l.sort(key=alphanum_key)


def monitorJobR():
   """Report active jobs"""
   if   len(sys.argv) == 3:
      messageSite = sys.argv[1]
      localJobId  = sys.argv[2]
   elif len(sys.argv) == 2:
      messageSite = sys.argv[1]
      localJobId  = '*'
   else:
      messageSite = '*'
      localJobId  = '*'

   remoteJobMonitor = RemoteJobMonitor(JOBMONITORHOST,JOBMONITORPORT)

   lastReportTime,report = remoteJobMonitor.queryRemoteActiveJobStatus(messageSite,localJobId)
   if len(report) > 0:
      print "Current time              %s" % (time.ctime())
      print "Last time status reported %s" % (time.ctime(lastReportTime))

      sites = report.split('|')
      for site in sites:
         queues = site.split('@')
         messageSite,lastReportTime = queues[0].split()
         print "\nReport for monitor %s" % (messageSite)
         print "   Last time status reported %s" % (time.ctime(float(lastReportTime)))
         del queues[0]

         for queue in queues:
            jobs = queue.split(':')
            print "   Queue %s" % (jobs[0])
            del jobs[0]

            reportedJobs = {}
            for job in jobs:
               jobId,status,state = job.split()
               if not status in reportedJobs:
                  reportedJobs[status] = {}
               try:
                  reportedJobs[status][jobId] = state
               except:
                  print "job poorly formatted: %s" % (job)

            for status in reportedJobs.keys():
               print "      Jobs in status: %s" % (status)

               jobIds = reportedJobs[status].keys()
               natsort(jobIds)
               for jobId in jobIds:
                  print "         %s  %s" % (jobId,reportedJobs[status][jobId])

            del reportedJobs
   else:
      print "Current time %s" % (time.ctime())
      print "No jobs currently being processed"


if __name__ == '__main__':
   openLogger()
   monitorJobR()


