#!/usr/bin/env python3
#
# @package      hubzero-submit-server
# @file         submit-server.py
# @copyright    Copyright (c) 2006-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2006-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import os
import sys
import logging

from hubzero.submit.ApplicationUtilities import ApplicationUtilities as ApplicationUtilities
from hubzero.submit.SubmitServer         import SubmitServer
from hubzero.submit.LogMessage           import getLogPIDMessage as getLogMessage

SUBMITSERVERLOGLEVEL   = "INFO"
DISTRIBUTORLOGLEVEL    = "INFO"
DISTRIBUTORSYSLOGLOCAL = "LOCAL3"
JOBKILLLOGLEVEL        = "ERROR"
JOBSTATUSLOGLEVEL      = "ERROR"
VENUESTATUSLOGLEVEL    = "ERROR"

SUBMITSERVERLOGDIRECTORY    = os.path.join(os.sep,"var","log","submit")
SUBMITSERVERLOGFILE         = "submit-server.log"
SUBMITSERVERLOGCRITICALFILE = "submit-server_critical.log"

SUBMITSERVERPIDFILE = os.path.join(os.sep,"var","run","submit","submit-server.pid")

CONFIGURATIONDIRECTORY       = os.path.join(os.sep,"etc","submit")
SUBMITCONFIGURATIONFILE      = "submit-server.conf"
DISTRIBUTORCONFIGURATIONFILE = "distributor.conf"
INFOSDIRECTORY               = os.path.join(os.sep,"etc","submit")
INFOSCONFIGURATIONFILE       = "infos.conf"
DAEMONSDIRECTORY             = os.path.join(os.sep,"etc","submit")
DAEMONSCONFIGURATIONFILE     = "infos.conf"

DISTRIBUTORDIRECTORY = os.path.join(os.sep,"opt","submit")
DISTRIBUTORSCRIPT    = "distributor"
HARVESTERDIRECTORY   = os.path.join(os.sep,"opt","submit")
HARVESTERSCRIPT      = "harvester.py"
JOBSTATUSDIRECTORY   = os.path.join(os.sep,"opt","submit")
JOBSTATUSSCRIPT      = "jobStatus.py"
JOBKILLDIRECTORY     = os.path.join(os.sep,"opt","submit")
JOBKILLSCRIPT        = "jobKill.py"
VENUEPROBEDIRECTORY  = os.path.join(os.sep,"opt","submit")
VENUEPROBESCRIPT     = "venueProbeR.py"

if __name__ == '__main__':
   submitServerLogLevel         = os.getenv("SUBMITSERVERLOGLEVEL",SUBMITSERVERLOGLEVEL)
   distributorLogLevel          = os.getenv("DISTRIBUTORLOGLEVEL",DISTRIBUTORLOGLEVEL)
   distributorSysLogLocal       = os.getenv("DISTRIBUTORSYSLOGLOCAL",DISTRIBUTORSYSLOGLOCAL)
   jobKillLogLevel              = os.getenv("JOBKILLLOGLEVEL",JOBKILLLOGLEVEL)
   jobStatusLogLevel            = os.getenv("JOBSTATUSLOGLEVEL",JOBSTATUSLOGLEVEL)
   venueStatusLogLevel          = os.getenv("VENUESTATUSLOGLEVEL",VENUESTATUSLOGLEVEL)

   submitServerLogDirectory     = os.getenv("SUBMITSERVERLOGDIRECTORY",SUBMITSERVERLOGDIRECTORY)
   submitServerLogFile          = os.getenv("SUBMITSERVERLOGFILE",SUBMITSERVERLOGFILE)
   submitServerLogCriticalFile  = os.getenv("SUBMITSERVERLOGCRITICALFILE",SUBMITSERVERLOGCRITICALFILE)

   submitServerPidFile          = os.getenv("SUBMITSERVERPIDFILE",SUBMITSERVERPIDFILE)

   configurationDirectory       = os.getenv("CONFIGURATIONDIRECTORY",CONFIGURATIONDIRECTORY)
   submitConfigurationFile      = os.getenv("SUBMITCONFIGURATIONFILE",SUBMITCONFIGURATIONFILE)
   distributorConfigurationFile = os.getenv("DISTRIBUTORCONFIGURATIONFILE",DISTRIBUTORCONFIGURATIONFILE)
   infosDirectory               = os.getenv("INFOSDIRECTORY",INFOSDIRECTORY)
   infosConfigurationFile       = os.getenv("INFOSCONFIGURATIONFILE",INFOSCONFIGURATIONFILE)
   daemonsDirectory             = os.getenv("DAEMONSDIRECTORY",DAEMONSDIRECTORY)
   daemonsConfigurationFile     = os.getenv("DAEMONSCONFIGURATIONFILE",DAEMONSCONFIGURATIONFILE)
 
   distributorDirectory         = os.getenv("DISTRIBUTORDIRECTORY",DISTRIBUTORDIRECTORY)
   distributorScript            = os.getenv("DISTRIBUTORSCRIPT",DISTRIBUTORSCRIPT)
   harvesterDirectory           = os.getenv("HARVESTERDIRECTORY",HARVESTERDIRECTORY)
   harvesterScript              = os.getenv("HARVESTERSCRIPT",HARVESTERSCRIPT)
   jobStatusDirectory           = os.getenv("JOBSTATUSDIRECTORY",JOBSTATUSDIRECTORY)
   jobStatusScript              = os.getenv("JOBSTATUSSCRIPT",JOBSTATUSSCRIPT)
   jobKillDirectory             = os.getenv("JOBKILLDIRECTORY",JOBKILLDIRECTORY)
   jobKillScript                = os.getenv("JOBKILLSCRIPT",JOBKILLSCRIPT)
   venueProbeDirectory          = os.getenv("VENUEPROBEDIRECTORY",VENUEPROBEDIRECTORY)
   venueProbeScript             = os.getenv("VENUEPROBESCRIPT",VENUEPROBESCRIPT)
 
   applicationUtilities = ApplicationUtilities(SUBMITSERVERLOGLEVEL)

   applicationUtilities.openFileLogger(submitServerLogDirectory,submitServerLogFile,
                                       submitServerLogLevel,
                                       submitServerLogCriticalFile)
   logger = logging.getLogger(__name__)
   applicationUtilities.daemonize()

   applicationUtilities.writePidFile(submitServerPidFile)

   submitServer = SubmitServer(configurationDirectory,submitConfigurationFile,
                               infosDirectory,infosConfigurationFile,
                               distributorDirectory,distributorScript,
                               harvesterDirectory,harvesterScript,
                               jobStatusDirectory,jobStatusScript,
                               jobKillDirectory,jobKillScript,
                               venueProbeDirectory,venueProbeScript)

   exitCode = 0
   if submitServer.configure():
      if submitServer.setupMySQL():
         if submitServer.setupClientListeners():
            submitServer.run()
         else:
            logger.log(logging.ERROR,getLogMessage("Shutting down - no listeners configured."))
            exitCode = 1
      else:
         logger.log(logging.ERROR,getLogMessage("Shutting down - could not configure MySQL database."))
         exitCode = 1
   else:
      logger.log(logging.ERROR,getLogMessage("Shutting down - server configuration failed."))
      exitCode = 1

   sys.exit(exitCode)


