/**
 * @package     hubzero-vncproxy
 * @file		vncproxyd.c
 * @author      Nicholas J. Kisseberth <nkissebe@purdue.edu>
 * @copyright   Copyright (c) 2010-2012 HUBzero Foundation, LLC.
 * @license     http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
 *
 * Copyright (c) 2010-2012 HUBzero Foundation, LLC.
 *
 * This file is part of: The HUBzero(R) Platform for Scientific Collaboration
 *
 * The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
 * software: you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * HUBzero is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * HUBzero is a registered trademark of * All rights reserved.
 */

/**
 * @TODO:
 *
 * - test ssl input sink
 * - improve efficiency of input ssl sink
 * - add timeout to input ssl sink
 * - replace socat with own tunneling code
 * - use sigaction() instead of old signal() and handle signal based shutdowns cleanly
 * - use SIGCHLD signals to monitor child termination
 * - group some processes together, not everything should be disassociated
 * - process_connection should timeout after some period of time
 * - process_connection should quit after some number of headers
 * - test logmsg for buffer overflow conditions
 * - maybe close file descriptors before exec() of socat to close dbd fd
 * - test how well child processes terminate to prevent zombies
 * - check proper use of return -1, exit() or _exit() after various fork()s
 * - run with valgrind again
 * - if database parameters not set, read /etc/hubzero.conf and look
 *   in default document root for configuration.php and find parameters
 *   there
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <assert.h>
#include <time.h>
#include <netdb.h>
#include <apr_pools.h>
#include <apr_strings.h>
#include <apu.h>
#include <apr_dbd.h>
#include <getopt.h>
#include <syslog.h>
#include <libgen.h>
#include <signal.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/sysinfo.h>
#define __USE_GNU /* For O_NOFOLLOW */
#include <fcntl.h>
#undef __USE_GNU

#include "log.h"

volatile sig_atomic_t gTerm = 0;

static char progname[PATH_MAX] = {0};

int
process_connection(apr_pool_t *pool, int sink, char *remoteip, int portbase, const apr_dbd_driver_t *driver,
	char *dbd_params)
{
	char *result = NULL;
	char *connect_uri;
	char *token = NULL;
	char *c;
	int rv;
	int state = 0; // 0 = skipping blank lines, 1 = getting request, 2 = getting headers
	int portnum;
	char line[8191];
	char request[8191];

	request[0] = 0;

	while (1) {
		result = fgets(line, 8191, stdin);

		if ((result == NULL) && (feof(stdin))) {
			logmsg(MLOG_NOTICE, "EOF reached");
			break;
		}

		if (result == NULL) {
			logmsg(MLOG_NOTICE, "Error reading input.");
			break;
		}

		if (state == 0) // skipping blank lines
		{
			if ((line[0] == '\r' && line[1] == '\n') || (line[0] == '\n')) {
				logmsg(MLOG_NOTICE, "Request: []");
				continue;
			}

			state = 1;
			// fallthrough
		}

		if (state == 1) // getting request
		{
			int length = strlen(line);

			if (length > 1) {
				if (line[length - 2] == '\r' && line[length - 1] == '\n')
					line[length - 2] = 0;
			} else if (length > 0) {
				if (line[length - 1] == '\n')
					line[length - 1] = 0;
			}

			strncpy(request, line, 8191);
			logmsg(MLOG_NOTICE, "Request: [%s]", request);

			state = 2;
			continue;
		}

		if (state == 2) // getting headers
		{
			int length = strlen(line);

			if (length > 1) {
				if (line[length - 2] == '\r' && line[length - 1] == '\n')
					line[length - 2] = 0;
			} else if (length > 0) {
				if (line[length - 1] == '\n')
					line[length - 1] = 0;
			}

			logmsg(MLOG_INFO, "Header: [%s]", line);

			// if ((line[0] == '\r' && line[1] == '\n' ) || (line[0] == '\n'))
			if (line[0] == 0)
				break;
		}
	}

	if (strncmp(request, "GET / ", 6) == 0) {
		logmsg(MLOG_NOTICE, "Response: HTTP/1.1 200 OK");

		printf("HTTP/1.1 200 OK\n");
		printf("Content-Type: text/html\n\n");
		printf("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\" "
			"\"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\">\n"
			"<html xmlns=\"http://www.w3.org/1999/xhtml\" lang=\"en\" xml:lang=\"en\">\n"
			"\t<head>\n"
			"\t\t<title></title>\n"
			"\t</head>\n"
			"\t<body>\n"
			"\t\tConnection to proxy succeeded.\n"
			"\t</body>\n"
			"</html>\n");
		printf("\n\n");
	} else if (strncmp(request, "GET ", 4) == 0) {
		printf("HTTP/1.1 404 Not Found\n\n");
		logmsg(MLOG_NOTICE, "Response: HTTP/1.1 404 Not Found");
	} else if (strncmp(request, "CONNECT ", 8) != 0) {
		printf("HTTP/1.1 501 Not Implemented\n\n");
		logmsg(MLOG_NOTICE, "Response: HTTP/1.1 501 Not Implemented");
	} else if (strncmp(request, "CONNECT vncsession", 18) != 0) {
		printf("HTTP/1.1 404 Not Found\n\n");
		logmsg(MLOG_NOTICE, "Response: HTTP/1.1 404 Not Found");
	} else if (request[18] != '.' && request[18] != ':') {
		printf("HTTP/1.1 404 Not Found\n\n");
		logmsg(MLOG_NOTICE, "Response: HTTP/1.1 404 Not Found");
	} else {
		connect_uri = request + 19;
		c = connect_uri;

		while (*c) {
			if ((*c == ' ') || (*c == '\r') || (*c == '\n') || (*c == ':'))
				break;

			c++;
		}

		if (*c) {
			*c++ = 0;
			portnum = atoi(c);
		}

		token = connect_uri;
	}

	fflush(stdout);

	if (token != NULL) {
		return process_vnc_connect_method(pool, sink, remoteip, token, portbase, driver, dbd_params, 1);
	}

	return 0;
}

void
server_sighandler(int sig)
{
	size_t len = 0;
	char *msg, *c;

	if (sig == SIGHUP)
		msg = "Caught signal SIGHUP. Exiting.\n";
	else if (sig == SIGINT)
		msg = "Caught signal SIGINT. Exiting.\n";
	else if (sig == SIGQUIT)
		msg = "Caught signal SIGQUIT. Exiting.\n";
	else if (sig == SIGTERM)
		msg = "Caught signal SIGTERM. Exiting.\n";
	else if (sig == SIGSEGV)
		msg = "Caught signal SIGSEGV. Exiting.\n";
	else
		msg = "Caught unexpected signal. Exiting.\n";

	for (c = msg; *c; c++)
		len++;

	write(2, msg, len);
	_exit(1);
}

int
server(apr_pool_t *pool, int sink, int background, int disassociate, int listenp, int portbase,
	const apr_dbd_driver_t *driver, char *dbd_params)
{
	int l;
	int s;
	int on = 1;
	struct sockaddr_in saddr;
	struct sockaddr_in caddr;
	socklen_t caddr_len = sizeof (caddr);
	socklen_t saddr_len = sizeof (saddr);
	int status;
	char host[NI_MAXHOST];
	char service[NI_MAXSERV];
	pid_t pid;
	int rv;

	if (freopen("/dev/null", "r", stdin) == NULL) {
		logmsg(MLOG_ERR, "Failed to redirect /dev/null to stdin");
		return -1;
	}

	if (signal(SIGHUP, SIG_IGN) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGHUP");
		return -1;
	}

	if (signal(SIGINT, server_sighandler) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGINT");
		return -1;
	}

	if (signal(SIGTERM, server_sighandler) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGTERM");
		return -1;
	}

	if (signal(SIGSEGV, server_sighandler) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGSEGV");
		return -1;
	}

	l = socket(AF_INET, SOCK_STREAM, 0);

	if (l == -1) {
		logmsg(MLOG_ERR, "Failed to create server socket file descriptor");
		return -1;
	}

	rv = setsockopt(l, SOL_SOCKET, SO_REUSEADDR, &on, sizeof (on));

	if (rv != 0) {
		logmsg(MLOG_ERR, "Unable to set SO_REUSEADDR on server socket");
		return -1;
	}

	bzero(&saddr, sizeof (saddr));
	saddr.sin_family = AF_INET;
	saddr.sin_addr.s_addr = htonl(INADDR_ANY);
	saddr.sin_port = htons(listenp);

	if (bind(l, (struct sockaddr *) &saddr, saddr_len) < 0) {
		logmsg(MLOG_ERR, "Unable to start server, bind() failed");
		return -1;
	}

	if (listen(l, 10) < 0) {
		logmsg(MLOG_ERR, "Unable to start server, listen() failed");
		return -1;
	}

	logmsg(MLOG_NOTICE, "Server is ready");

	while (1) {

		logmsg(MLOG_DEBUG, "Server is waiting for a connection");

		s = accept(l, (struct sockaddr *) &caddr, &caddr_len);

		if (s < 0) {
			if (errno == EINTR) {
				logmsg(MLOG_DEBUG, "Accept loop interrupted");
				continue;
			}

			logmsg(MLOG_ERR, "Unable to continue server, accept() failed [%d]", errno);
			break;
		}

		logmsg(MLOG_DEBUG, "Got connection");

		if (background || disassociate) {
			pid = fork();

			if (pid == -1) {
				logmsg(MLOG_ERR, "Client handler backgrounding fork() failed");
				return -1;
			}

			if (pid) {
				rv = close(s);

				if (rv == -1) {
					logmsg(MLOG_ERR, "Failed to close serve after handoff");
					return -1;
				}

				if (disassociate) {
					logmsg(MLOG_DEBUG, "waiting for [%d] to exit", pid);
					waitpid(pid,&status,0); // wait for intermediate background process to exit if disassociating
				}

				continue;
			}

			logmsg(MLOG_DEBUG, "Client handler background process [%d] created", getpid());
		}

		if (disassociate && pid == 0) {

			rv = setsid();

			if (rv == -1) {
				logmsg(MLOG_ERR, "Can'tcreate new client handling process session");
				_exit(1);
			}

			pid = fork();

			if (pid == -1) {
				logmsg(MLOG_ERR, "Disassociating client handler fork() failed");
				return -1;
			}

			if (pid) {
				logmsg(MLOG_DEBUG, "Parent client handler process [%d] exiting [disassociating]", getpid());
				exit(0);
			}

			logmsg(MLOG_DEBUG, "Client handler disassociated process [%d] created", getpid());
		}

		if (background) {
			rv = close(l);

			if (rv == -1) {
				logmsg(MLOG_ERR, "Failed to close server socket in client handler");
				_exit(1);
			}
		}

		rv = getnameinfo((struct sockaddr *) &caddr, caddr_len, host, NI_MAXHOST, service, NI_MAXSERV, NI_NUMERICHOST);

		if (rv != 0) {
			logmsg(MLOG_ERR, "Unable to resolve client address info");
			_exit(1);
		}

		logmsg(MLOG_NOTICE, "Accepted connection from %s", host);

		rv = dup2(s, 0);

		if (rv == -1) {
			logmsg(MLOG_ERR, "Unable to dup() client socket to stdin");
			_exit(1);
		}

		rv = dup2(s, 1);

		if (rv == -1) {
			logmsg(MLOG_ERR, "Unable to dup() client socket to stdout");
			_exit(1);
		}

		rv = process_connection(pool, sink, host, portbase, driver, dbd_params);

		if (background) {
			return rv;
		}

		if (rv == 0) {
			logmsg(MLOG_DEBUG, "Successfully processed connection");
		} else {
			logmsg(MLOG_DEBUG, "Error while processing connection");
		}

		if (close(s) != 0) {
			logmsg(MLOG_ERR, "Failed close client socket [%d]", errno);
			break;
		}

		if (freopen("/dev/null", "r", stdin) == NULL) {
			logmsg(MLOG_ERR, "Failed to redirect /dev/null to stdin");
			return -1;
		}

		if (freopen("/dev/null", "w", stdout) == NULL) {
			logmsg(MLOG_ERR, "Failed to redirect stdout to /dev/null");
			return -1;
		}
	}

	rv = close(l);

	if (rv == -1) {
		logmsg(MLOG_ERR, "Failed to close listening socket");
	}

	return -1;
}

int
pidfile_write(const char *pidfile)
{
	int rv = 0;
	int fd = -1;
	int pid = 0;
    struct stat buf;
	char strbuf[80];
	int count = 0;

	rv = open(pidfile, O_NOFOLLOW | O_RDWR | O_CREAT | O_NONBLOCK,
		S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP);

	if (rv < 0) {
        logmsg(MLOG_ERR, "Unable to open or create pidfile [%s]", pidfile);
      	return -1;
    }

    fd = rv;

   	rv = fstat(fd, &buf);

    if (rv != 0) {
  		logmsg(MLOG_ERR, "Unable to read pidfile [%s] status", pidfile);
        return -1;
	}

    rv = S_ISREG(buf.st_mode);
		
    if ( ! rv ) {
    	logmsg(MLOG_ERR, "Pidfile [%s] is not a regular file", pidfile);
		return -1;
    }
		
	if (buf.st_size > 0) {
		struct sysinfo info;

		count = read(fd, strbuf, sizeof(strbuf) - 1);

		if (count < 0) {
			logmsg(MLOG_ERR, "Unable to read existing pidfile [%s]", pidfile);
			return -1;
		}

		rv = sscanf(strbuf, " %u ", &pid);

		if (rv != 1) {
			logmsg(MLOG_NOTICE, "Invalid pidfile [%s] contents", pidfile);
		}
		else {
			int pfd;

			rv = snprintf(strbuf,sizeof(strbuf),"/proc/%d",pid);

			if (rv <= 0) {
				logmsg(MLOG_NOTICE, "Unable to verify process id from pidfile [%s] contents [%d]", pidfile, pid);
				return -1;
			}

			pfd = open(strbuf,O_NOFOLLOW|O_RDONLY);

			if (pfd >= 0) {
				struct sysinfo info;

				rv = sysinfo(&info);

				if (rv != 0) {
					logmsg(MLOG_WARNING, "Unable to verify validity of existing pidfile [%s]", pidfile);
				}
				else {
					time_t uptime = 0;

					uptime = time(NULL) - info.uptime;

					if (buf.st_mtime < uptime) {
						logmsg(MLOG_WARNING,"Pidfile [%s] exists but is older than system uptime", pidfile, pid);
					}
					else {
						logmsg(MLOG_ERR, "Pidfile already exists for running process [%d]", pid);
						return -1;
					}
				}
			}
			else {
				logmsg(MLOG_NOTICE, "Ignoring pidfile for non-existent process [%d]", pid);
				ftruncate(fd,0);
			}
		}
	}

	// If we got this far pidfile belongs to us and should be cleaned up on error.

	rv = lseek(fd, 0, SEEK_SET);

    count = snprintf(strbuf, 80, "%d\n", getpid());

    rv = write(fd,strbuf,count);

    if (rv != count) {
    	logmsg(MLOG_ERR, "Unable to write pid [%d] to [%s]", getpid(), pidfile);
		unlink(pidfile);
		return -1;
	}

	rv = ftruncate(fd, count);

    if (rv != 0) {
    	logmsg(MLOG_ERR, "Unable to truncate pidfile [%s]", pidfile);
		unlink(pidfile);
		return -1;
	}

    rv = close(fd);

    if (rv == -1) {
    	logmsg(MLOG_ERR, "Unable to close pidfile [%s]", pidfile);
		unlink(pidfile);
    	return -1;
	}

    logmsg(MLOG_DEBUG, "Wrote pid [%d] to [%s]", getpid(), pidfile);

	return 0;
}

int
main(int argc, const char * const argv[])
{
	int listenp = -1; /* --port=#		*/
	char *logfile = NULL; /* --logfile=file		*/
	int portbase = -1; /* --portbase=#	*/
	char *pidfile = NULL; /* --pidfile=file		*/
	char *token = NULL; /* --token=string	*/
	char *remoteip = NULL; /* --remoteip=string		*/
	char *dbd_driver = NULL; /* DBD_DRIVER		*/
	char *dbd_params = NULL; /* DBD_PARAMS		*/
	int background = 0; /* -b                   */
	int disassociate = 0; /* -d                   */
	int sink = -1; /* -s                   */
    char *user = NULL; /* --user= */
	char *group = NULL; /* --group= */
	apr_status_t rv = 0;
	const apr_dbd_driver_t *driver = NULL;
	apr_pool_t *pool = NULL;
	long maxfd = 0;
	pid_t pid = 0;
	uid_t uid = 0;
	gid_t gid = 0;
	uid_t new_uid = 0;
	gid_t new_gid = 0;
	int status = 0;
	int fd = 0;
	int c = 0;

	strncpy(progname, basename((char *) argv[0]), sizeof (progname));

	opterr = 0;

	while (1) {
		static struct option long_options[] = {
			{ "portbase", required_argument, 0, 'P'},
			{ "port", required_argument, 0, 'p'},
			{ "logfile", required_argument, 0, 'f'},
			{ "pidfile", required_argument, 0, 'i'},
			{ "token", required_argument, 0, 't'},
			{ "remoteip", required_argument, 0, 'r'},
			{ "loglevel", required_argument, 0, 'l'},
			{ "sysloglevel", required_argument, 0, 's' },
			{ "user", required_argument, 0, 'u' },
			{ "group", required_argument, 0, 'g' },
			{ 0, 0, 0, 0}
		};

		int option_index = 0;

		c = getopt_long(argc, (char **) argv, "wbzdt:f:", long_options, &option_index);

		if (c == -1)
			break;

		switch (c) {

			case 0:
				if (long_options[option_index].flag != 0)
					break;
				printf("option %s", long_options[option_index].name);
				if (optarg)
					printf(" with arg %s", optarg);
				printf("\n");
				break;

			case 'b':
				background = 1;
				break;

			case 'd':
				disassociate = 1;
				break;

			case 'g':
				group = optarg;
				break;

			case 'p':
				listenp = atoi(optarg);
				break;

			case 'P':
				portbase = atoi(optarg);
				break;

			case 'f':
				logfile = optarg;
				break;

			case 'i':
				pidfile = optarg;
				break;

			case 't':
				token = optarg;
				break;

			case 'u':
				user = optarg;
				break;

			case 'r':
				remoteip = optarg;
				break;

			case 'l':
				if (strcasecmp(optarg,"none") == 0)
					log_priority = -1;
				else {
					log_priority = atoi(optarg);
					if ((log_priority < 0) || (log_priority > LOG_DEBUG))
						log_priority = LOG_NOTICE;
				}
				break;

			case 's':
				if (strcasecmp(optarg,"none") == 0)
					syslog_priority = -1;
				else {
					syslog_priority = atoi(optarg);
					if ((syslog_priority < 0) || (syslog_priority > LOG_DEBUG))
						syslog_priority = LOG_WARNING;
				}
				break;

			case 'z':
				sink = 1;
				break;

			case '?':
				break;

			default:
				break;
		
		}
	}

	/* Print any remaining command line arguments (not options). */
	
	if (optind < argc) {
		printf("non-option ARGV-elements: ");

		while (optind < argc)
			printf("%s ", argv[optind++]);

		putchar('\n');
	}

	if ( (user != NULL) || (group != NULL)) {
		uid = getuid();
		gid = getgid();

		if (uid != 0) {
			logmsg(MLOG_ERR, "Not running as root, unable to change user/group credentials");
			exit(1);
		}

		if (user != NULL) {
			struct passwd pwbuf, *pwbufp;
			long buflen = sysconf(_SC_GETPW_R_SIZE_MAX);
			char *buf;
			int ret;

			buf = malloc(buflen);

			if (buf == NULL) {
				logmsg(MLOG_ERR, "Ran out of memory, unable to change user/group credentials");
				exit(1);
			}

			ret = getpwnam_r(user, &pwbuf, buf, buflen, &pwbufp);

			if (pwbufp == NULL) {
				logmsg(MLOG_ERR, "Failed to find user information, unable to change user/group credentials");
				exit(1);
			}

			new_uid = pwbuf.pw_uid;
			new_gid = pwbuf.pw_gid;

			free(buf); 
		} else {
			new_uid = uid;
			new_gid = gid;
		}

		if (group != NULL) {
			struct group gbuf, *gbufp;
			long buflen = sysconf(_SC_GETGR_R_SIZE_MAX);
			char *buf;
			int ret;

			buf = malloc(buflen);

			if (buf == NULL) {
				logmsg(MLOG_ERR, "Ran out of memory, unable to change user/group credentials");
				exit(1);
			}

			ret = getgrnam_r(group, &gbuf, buf, buflen, &gbufp);

			if (gbufp == NULL) {
				logmsg(MLOG_ERR, "Failed to find group information, unable to change user/group credentials");
				exit(1);
			}

			new_gid = gbuf.gr_gid;

			free(buf); 
		} else {
			new_gid = gid;
		}
		
		if (setgroups(1, &new_gid) != 0) {
			logmsg(MLOG_ERR, "Failed to set supplementary group IDs, unable to change user/group credentials");
			exit(1);
		}

		if (setgid(new_gid) != 0) {
			logmsg(MLOG_ERR, "Failed to set group identity, unable to change user/group credentials");
			exit(1);
		}

		if (setuid(new_uid) != 0) {
			logmsg(MLOG_ERR, "Failed to set user identity, unable to change user/group credentials");
			exit(1);
		}
	}

	if (log_priority >= 0) {
		char *source = "";
		mode_t mode = 0;
		FILE *fp = NULL;

		if (logfile == NULL) {
			if ((background == 1) || (disassociate == 1)) {
				logfile = "/var/log/vncproxyd/vncproxyd.log";
				source = " (default)";
			}
		}

		if (logfile != NULL) { 
			mode = umask(0137);
			fp = freopen(logfile, "a", stderr);
			umask(mode);

			if (fp == NULL) {
				logmsg(MLOG_ERR, "Unable to open logfile [%s]", logfile);
				logmsg(MLOG_DEBUG, "Logging to [stderr] (failover)");
			}
		}

		if (logfile == NULL) {
			logmsg(MLOG_DEBUG, "Logging to [stderr] (default)");
		} 
		else {
			logmsg(MLOG_DEBUG, "Logging to [%s]%s", logfile, source);
		}
	}
	else {
		logmsg(MLOG_DEBUG, "Logging to [/dev/null] (implied)");
		freopen("/dev/null","a",stderr);
	}

	if (syslog_priority >= 0) {
		openlog(progname, LOG_PID, (token == NULL) ? LOG_DAEMON : LOG_USER);
	}

	logmsg(MLOG_DEBUG, "Started vncproxyd process [%d]", getpid());

	rv = apr_app_initialize(&argc, &argv, NULL);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Unable to initialize application");
		exit(1);
	}

	rv = atexit(apr_terminate);

	if (rv != 0) {
		logmsg(MLOG_ERR, "Failed to register apr_terminate() with atexit()");
		apr_terminate();
		exit(2);
	}

	rv = apr_pool_create(&pool, NULL);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Unable to create memory pool");
		exit(2);
	}

	rv = apr_dbd_init(pool);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Unable to initialize APR DBD environment");
		exit(3);
	}

	maxfd = sysconf(_SC_OPEN_MAX);

	closelog();

	for (fd = 3; fd < maxfd; fd++) {
		rv = close(fd);

		if (rv == -1) {
			if (errno == EBADF)
				continue;

			logmsg(MLOG_ERR, "Failed to close file descriptor [%d]", fd);
			exit(8);
		} else if (rv == 0) {
			logmsg(MLOG_DEBUG, "Closed file descriptor [%d]", fd);
		}
	}

	if (logfile == NULL) {
		logmsg(MLOG_DEBUG, "Will log to stderr (default)");
	} else {
		logmsg(MLOG_DEBUG, "Will log to [%s]", logfile);
	}

	if (disassociate == 0) {
		logmsg(MLOG_DEBUG, "Will not disassociate from parent process (default)");
	} else {
		logmsg(MLOG_DEBUG, "Will disassociate from parent process");
	}

	if (background == 0) {
		background = disassociate;
		if (disassociate)
			logmsg(MLOG_DEBUG, "Will run in background (implied)");
		else
			logmsg(MLOG_DEBUG, "Will run in foreground (default)");
	} else {
		logmsg(MLOG_DEBUG, "Will run in %s", (!background) ? "foreground" : "background");
	}

	if (sink == -1) {
		sink = 0;
		logmsg(MLOG_DEBUG, "Will not sink input waiting for ssl handshake (default)");
	} else {
		logmsg(MLOG_DEBUG, "Will sink input waitiing for ssl handshake");
	}

	if (log_priority >=0) {
		logmsg(MLOG_DEBUG, "Log level is [%d]", log_priority);
	} else {
		logmsg(MLOG_DEBUG, "Log level is [None]");
	}

	if (syslog_priority >=0) {
		logmsg(MLOG_DEBUG, "Syslog level is [%d]", syslog_priority);
	} else {
		logmsg(MLOG_DEBUG, "Syslog level is [None]");
	}

	if (portbase == -1) {
		portbase = 4000;
		logmsg(MLOG_DEBUG, "Will use portbase of [4000] (default)");
	} else {
		logmsg(MLOG_DEBUG, "Will use portbase of [%d]", portbase);
	}

	if (token == NULL) {
		logmsg(MLOG_DEBUG, "Will run as server (default)");

		if (listenp == -1) {
			listenp = 8080;
			logmsg(MLOG_DEBUG, "Will listen on port [8080] (default)");
		} else {
			logmsg(MLOG_DEBUG, "Will listen on port [%d]", listenp);
		}

		if (pidfile == NULL) {
			pidfile = "/var/run/vncproxyd/vncproxyd.pid";
			logmsg(MLOG_DEBUG, "Will record PID in [%s] (default)", pidfile);
		} else {
			logmsg(MLOG_DEBUG, "Will record PID in [%s]", pidfile);
		}
	} else { // Running as helper uses remoteip, token, portbase, dbd_driver, dbd_params

		if (remoteip == NULL) {
			remoteip = "0.0.0.0";
			logmsg(MLOG_DEBUG, "Will record remote ip as [%s] (default)", remoteip);
		} else {
			logmsg(MLOG_DEBUG, "Will record remote ip as [%s]", remoteip);
		}

		logmsg(MLOG_DEBUG, "Will lookup token [%s]", token);
	}

	dbd_driver = getenv("DBD_DRIVER");

	if ((dbd_driver == NULL) || (*dbd_driver == 0)) {
		logmsg(MLOG_DEBUG, "Will use database driver [mysql] (default)");
		dbd_driver = "mysql";
	} else {
		logmsg(MLOG_DEBUG, "Will use database driver [%s]", dbd_driver);
	}

	dbd_driver = apr_pstrdup(pool, dbd_driver);

	if (dbd_driver == NULL) {
		logmsg(MLOG_ERR, "Unable to allocate memory for DBD_DRIVER variable");
		exit(4);
	}

	dbd_params = getenv("DBD_PARAMS");

	if ((dbd_params == NULL) || (*dbd_params == 0)) {
		logmsg(MLOG_DEBUG, "Will use default database parameters (default)");
		dbd_params = "";
	} else {
		logmsg(MLOG_DEBUG, "Will use database parameters [*redacted*]");
	}

	dbd_params = apr_pstrdup(pool, dbd_params);

	if (dbd_params == NULL) {
		logmsg(MLOG_ERR, "Unable to allocate memory for DBD_PARAMS variable");
		exit(5);
	}

	rv = apr_dbd_get_driver(pool, dbd_driver, &driver);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Unable to load APR_DBD driver [%s]", dbd_driver);
		exit(6);
	}

	if (chdir("/") != 0) {
		logmsg(MLOG_ERR, "Unable to change directory to [/]");
		exit(7);
	}

	umask(0);

	if (background || disassociate) {
		logmsg(MLOG_NOTICE, "Backgrounding process.");

		pid = fork();

		if (pid == -1) {
			logmsg(MLOG_ERR, "Backgrounding fork() failed");
			exit(8);
		}

		if (pid) {
			logmsg(MLOG_DEBUG, "Parent process [%d] exiting [backgrounding]", getpid());
			_exit(0);
		}

		logmsg(MLOG_DEBUG, "Background process [%d] created", getpid());
	}

	if (disassociate) {
		rv = setsid();

		if (rv == -1) {
			logmsg(MLOG_ERR, "Can't create new process session");
			exit(8);
		}

		pid = fork();

		if (pid == -1) {
			logmsg(MLOG_ERR, "Disassociating fork() failed");
			exit(8);
		}

		if (pid) {
			logmsg(MLOG_DEBUG, "Parent process [%d] exiting [disassociating]", getpid());
			_exit(0);
		}

		logmsg(MLOG_DEBUG, "Disassociated process [%d] created", getpid());
	}

	int error = 0;

	logmsg(MLOG_INFO, "Starting vncproxyd");

	rv = pidfile_write(pidfile);

	if (rv == 0) {

		error = server(pool, sink, background, disassociate, listenp, portbase, driver, dbd_params);

		if (pidfile != NULL) {
			rv = unlink(pidfile);

			if ((rv == -1) && (errno != ENOENT)) {
				logmsg(MLOG_ERR, "Unable to delete pidfile [%s]", pidfile);
				error = 1;
			}
		}
		
		if (!error) {
			logmsg(MLOG_DEBUG, "The vncproxyd quit normally");
			exit(0);
		}
	}

	logmsg(MLOG_ERR, "The vncproxyd quit due to an error");
	exit(9);
}
