#!/usr/bin/env python

'''
A WebSocket to TCP socket proxy with support for "wss://" encryption.
Copyright 2011 Joel Martin
Licensed under LGPL version 3 (see docs/LICENSE.LGPL-3)

You can make a cert/key with openssl using:
openssl req -new -x509 -days 365 -nodes -out self.pem -keyout self.pem
as taken from http://docs.python.org/dev/library/ssl.html#certificates

'''

import signal, socket, optparse, time, os, sys, subprocess, logging, re, MySQLdb, grp, pwd, datetime, dateutil.tz
from select import select
import websocket
try:
    from urllib.parse import parse_qs, urlparse
except:
    from cgi import parse_qs
    from urlparse import urlparse

class ProxyRequestHandler(websocket.WebSocketRequestHandler):

    traffic_legend = """
Traffic Legend:
    }  - Client receive
    }. - Client receive partial
    {  - Target receive

    >  - Target send
    >. - Target send partial
    <  - Client send
    <. - Client send partial
"""
    #
    # WebSocketRequestHandler logging/output functions
    #

    def print_traffic(self, token="."):
        """ Show traffic flow mode. """
        if self.traffic and not self.daemon:
            sys.stdout.write(token)
            sys.stdout.flush()

    def msg(self, msg, *args, **kwargs):
        """ Output message with handler_id prefix. """
        prefix = "%s %s websockify[%d] [info] [%03d] " % (
            datetime.datetime.now(dateutil.tz.tzlocal()).isoformat("T"),
            self.address_string(),
            os.getpid(),
            self.handler_id)

        self.logger.log(logging.INFO, "%s%s" % (prefix, msg), *args, **kwargs)

    def vmsg(self, msg, *args, **kwargs):
        """ Same as msg() but as debug. """
        prefix = "%s %s websockify[%d] [debug] [%03d] " % (
            datetime.datetime.now(dateutil.tz.tzlocal()).isoformat("T"),
            self.address_string(),
            os.getpid(),
            self.handler_id)

        self.logger.log(logging.DEBUG, "%s%s" % (prefix, msg), *args, **kwargs)

    def warn(self, msg, *args, **kwargs):
        """ Same as msg() but as warning. """
        prefix = "%s %s websockify[%d] [warn] [%03d] " % (
            datetime.datetime.now(dateutil.tz.tzlocal()).isoformat("T"),
            self.address_string(),
            os.getpid(),
            self.handler_id)

        self.logger.log(logging.WARN, "%s%s" % (prefix, msg), *args, **kwargs)

    def log_message(self, format, *args):
        self.logger.log(logging.INFO,"%s %s websockify[%d] [info] [%03d] %s" %
                         (datetime.datetime.now(dateutil.tz.tzlocal()).isoformat("T"),
                          self.address_string(),
                          os.getpid(),
                          self.handler_id,
                          format%args))

    def new_websocket_client(self):
        """
        Called after a new WebSocket connection has been established.
        """
        # Checks if we receive a token, and look
        # for a valid target for it then
        if self.server.target_cfg:
            target = self.get_target(self.server.target_cfg, self.path)
            self.server.target_host = target['host']
            self.server.target_port = target['port']
        else:
            target = {}

        # Connect to the target
        if self.server.wrap_cmd:
            msg = "connecting to command: '%s' (port %s)" % (" ".join(self.server.wrap_cmd), self.server.target_port)
        elif self.server.unix_target:
            msg = "connecting to unix socket: %s" % self.server.unix_target
        else:
            msg = "connecting to: %s:%s" % (
                                    self.server.target_host, self.server.target_port)

        if self.server.ssl_target:
            msg += " (using SSL)"
        self.log_message(msg)

        tsock = websocket.WebSocketServer.socket(self.server.target_host,
                                                 self.server.target_port,
                connect=True, use_ssl=self.server.ssl_target, unix_socket=self.server.unix_target)

        if 'dsn' in target:
            (viewid, starttime) = self.start_view(target['dsn'], target['sessnum'], target['username'])

        self.print_traffic(self.traffic_legend)

        # Start proxying
        try:
            if 'dsn' in target:
                self.do_proxy(tsock,target['dsn'], viewid, target['timeout']*0.4)
            else:
                self.do_proxy(tsock)
        except:
            if tsock:
                tsock.shutdown(socket.SHUT_RDWR)
                tsock.close()
                if self.verbose: 
                    self.log_message("%s:%s: Closed target",
                            self.server.target_host, self.server.target_port)
                if 'dsn' in target:
                    self.end_view(target['dsn'], viewid, starttime, target['sessnum'], target['username'])

            raise

    def db_connect(self, dsn):

        driver_re = "^\s*(.*?)\s*:\s*(.*)$"

        m = re.search(driver_re, dsn)

        if m == None:
            raise ValueError

        driver = m.group(1)
        dsn = m.group(2)

        params = {}
        params['host'] = 'localhost'
        params['user'] = ''
        params['password'] = ''
        params['port'] = 3306
        params['db'] = ''

        while (dsn != ""):
            #              key         value                 remainder
            kv_re = "\s*([^;]+?)\s*=\s*(.+?)(?:(?<!\\\\);|$)\s*(.*)$"

            m = re.search(kv_re, dsn)

            if m == None:
                raise ValueError

            params[m.group(1)] = m.group(2).replace('\;',';')
            dsn = m.group(3)

        delay = 1

        db = None

        while delay < 60:
            try:
                db = MySQLdb.connect(host=params['host'], user=params['user'], passwd=params['password'], db=params['db'], port=params['port'])
                return db

            except Exception as e:
                self.log_message(e.args[1])

            time.sleep(delay)

            delay = delay * 2

        return db

    def mysql(self, c, cmd):

        try:
            count = c.execute(cmd)
            return c.fetchall()

        except MySQLdb.MySQLError, (num, expl):
            self.log_message("%s.  SQL was: %s" % (expl,cmd))
            return ()

        except:
            self.log_message("Some other MySQL exception.")
            return ()

    def parse_line(self, line = ""):

        result = {}

        # ignore blank lines or commented lines

        m = re.search(r'^\s*($|#)', line)

        if m != None:
            return None

        # remove token from line

        m = re.search(r'^(.*?)(?:(?<!\\):|$)(.*)', line)

        if m == None:
            raise ValueError

        result['token'] = m.group(1).replace('\:',':')

        line = m.group(2)

        # remove dsn from line

        m = re.search(r'(?:(^)|^(.*?)(?<!\\):)\s*(mysql\s*:.+=.+)$',line)

        if m != None:
            line = m.group(1) or m.group(2) or ""
            result['dsn'] = m.group(3)

        spl = line.split(':')

        if (spl.__len__() == 1):
            if 'dsn' in result:
                result['scope'] = spl[0]
            elif 'dsn' not in result:
                result['host'] = spl[0]
        elif (spl.__len__() == 2):
            if 'dsn' in result:
                raise ValueError

            result['host'] = spl[0]

            if spl[1] != '':
                result['port'] = int(spl[1])
        else:
            raise ValueError

        if ('dsn' not in result):
            if 'host' not in result or result['host'] == "":
                result['host'] = 'localhost'
            if 'port' not in result or result['port'] == "":
                result['port'] = 5901
        else:
            if 'scope' not in result:
                result['scope'] = ''

        return result

    def translate(self,ip,dsn,token):

        result = {}

        db = self.db_connect(dsn)

        if (db == None):
            return result

        c = db.cursor()

        if (c == None):
            db.close()
            return result

        arr = self.mysql(c,"""
            SELECT viewperm.sessnum,viewuser,display.hostname,session.dispnum,
                session.timeout,portbase
                FROM viewperm
                JOIN display ON viewperm.sessnum = display.sessnum
                JOIN session ON session.sessnum = display.sessnum
                JOIN host ON display.hostname = host.hostname
                WHERE viewperm.viewtoken='%s'""" % token)

        db.close()

        if len(arr) != 1:
            self.log_message("No database entry found for token %s." % token)
            return result

        row = arr[0]

        result['dsn'] = dsn
        result['sessnum'] = row[0] # sessnum
        result['username'] = row[1] # username
        result['host'] = row[2] # host
        result['dispnum'] = int(row[3]) # dispnum
        result['timeout'] = int(row[4]) # timeout
        result['portbase'] = int(row[5]) # portbase
        result['portbase'] = 4000 # portbase
        result['port'] = result['dispnum'] + result['portbase'] # port

        if not row[4] or int(row[4]) < 1:
            result['timeout'] = 0
        elif int(row[4]) < 300:
            result['timeout'] = 300
        else:
            result['timeout'] = int(row[4])

        return result

    def mysql_act(self, c, cmd):
        try:
            count = c.execute(cmd)
            return ""
        except MySQLdb.MySQLError, (num, expl):
            return expl

    def refresh_view(self, dsn, viewid):

        self.log_message("Refreshing heartbeat for view %d" % viewid)
        
        db = self.db_connect(dsn)

        c = db.cursor()

        sql = """UPDATE view SET heartbeat=now() WHERE viewid='%d'""" % viewid

        if self.verbose:
            self.log_message(sql)

        arr = self.mysql(c, sql)

        db.close()

    def end_view(self, dsn, viewid, starttime, sessnum, username):

        endtime = time.time()

        db = self.db_connect(dsn)

        remoteip = self.client_address[0]

        c = db.cursor()
        
        sql = """INSERT INTO viewlog(sessnum,username,remoteip,time,duration) SELECT sessnum, username, remoteip, start, %f FROM view WHERE viewid='%d'""" % \
            ((endtime-starttime), viewid)

        err = self.mysql_act(c,sql)

        if self.verbose:
            self.log_message(sql)

        if err != "":
            self.log_message("ERROR: %s" % err)

        sql = """UPDATE session JOIN view ON session.sessnum=view.sessnum SET session.accesstime=now() WHERE viewid='%d'""" % viewid

        err = self.mysql_act(c,sql)

        if self.verbose:
            self.log_message(sql)

        if err != "":
            self.log_message("ERROR: %s" % err)

        sql = """DELETE FROM view WHERE viewid='%d'""" % viewid

        err = self.mysql_act(c,sql)

        if self.verbose:
            self.log_message(sql)

        if err != "":
            self.log_message("ERROR: %s" % err)

        db.close()

        self.log_message("View %d (%s@%s) ended after %f seconds." % (viewid,username,remoteip,endtime-starttime))


    def start_view(self, dsn, sessnum, username):

        db = self.db_connect(dsn)

        remoteip = self.client_address[0]

        starttime = time.time()

        c = db.cursor()

        sql = """UPDATE session SET accesstime=now() WHERE sessnum = %d""" % sessnum
        
        err = self.mysql_act(c,sql)

        if self.verbose:
            self.log_message(sql)

        if err != "":
            self.log_message("ERROR: %s" % err)

        sql ="""INSERT INTO view(sessnum,username,remoteip,start,heartbeat) VALUES(%d,'%s','%s',now(),now())""" %\
            (sessnum,username,remoteip)
  
        err = self.mysql_act(c,sql)

        if self.verbose:
            self.log_message(sql)

        if err != "":
            self.log_message("ERROR: %s" % err)
            return None
  
        sql = """SELECT LAST_INSERT_ID()"""

        arr = self.mysql(c,sql)

        if self.verbose:
            self.log_message(sql)

        db.close()

        viewid = arr[0][0]

        self.log_message("View %d (%s@%s) started at %f seconds." % (viewid,username,remoteip,starttime))

        return (viewid, starttime)

    def get_target(self, target_cfg, path):
        """
        Parses the path, extracts a token, and looks for a valid
        target for that token in the configuration file(s). Sets
        target_host and target_port if successful
        """
        # The files in targets contain the lines
        # in the form of token: host:port

        # Extract the token parameter from url
        args = parse_qs(urlparse(path)[4]) # 4 is the query from url

        if not args.has_key('token') or not len(args['token']):
            raise self.CClose("Token not present")

        token = args['token'][0].rstrip('\n')

        # target_cfg can be a single config file or directory of
        # config files
        if os.path.isdir(target_cfg):
            cfg_files = [os.path.join(target_cfg, f)
                         for f in os.listdir(target_cfg)]
        else:
            cfg_files = [target_cfg]

        targets = {}
        for f in cfg_files:
            for line in [l.strip() for l in file(f).readlines()]:
                result = self.parse_line(line)

                if result == None:
                    continue

                if (result['token'] == token):
                    regex = re.escape(token)
                elif (result['token'] == '*'):
                    regex = ".*"
                else:
                    regex = result['token']

                if not re.search(regex,token):
                    continue

                if 'dsn' not in result:
                    targets[token] = result
                else:
                    result = self.translate("",result['dsn'],token)

                    if result != None:
                        targets[token] = result

                continue

        self.vmsg("Target config: %s" % repr(targets))

        if targets.has_key(token):
            result = targets[token]
            ip = self.client_address[0]
            if 'username' in result:
                self.log_message("Map %s@%s for %s to %s:%d" % (result['username'],ip,token,result['host'],result['port']))
            else:
                self.log_message("Map unknown@%s for %s to %s:%d" % (ip,token,result['host'],result['port']))

            return result
        else:
            raise self.CClose("Token '%s' not found" % token)

    def do_proxy(self, target, dsn = None, viewid = None, interval = 0):
        """
        Proxy client WebSocket to normal target socket.
        """
        cqueue = []
        c_pend = 0
        tqueue = []
        rlist = [self.request, target]

        if interval:
            heartbeat = time.time() + interval

        while True:
            wlist = []

            if tqueue: wlist.append(target)
            if cqueue or c_pend: wlist.append(self.request)
            ins, outs, excepts = select(rlist, wlist, [], 1)
            if excepts: raise Exception("Socket exception")

            if self.request in outs:
                # Send queued target data to the client
                c_pend = self.send_frames(cqueue)

                cqueue = []

            if self.request in ins:
                # Receive client data, decode it, and queue for target
                bufs, closed = self.recv_frames()
                tqueue.extend(bufs)

                if closed:
                    # TODO: What about blocking on client socket?
                    if self.verbose: 
                        self.log_message("%s:%s: Client closed connection",
                                self.server.target_host, self.server.target_port)
                    raise self.CClose(closed['code'], closed['reason'])


            if target in outs:
                # Send queued client data to the target
                dat = tqueue.pop(0)
                sent = target.send(dat)
                if sent == len(dat):
                    self.print_traffic(">")
                else:
                    # requeue the remaining data
                    tqueue.insert(0, dat[sent:])
                    self.print_traffic(".>")


            if target in ins:
                # Receive target data, encode it and queue for client
                buf = target.recv(self.buffer_size)
                if len(buf) == 0:
                    if self.verbose:
                        self.log_message("%s:%s: Target closed connection",
                                self.server.target_host, self.server.target_port)
                    raise self.CClose(1000, "Target closed")

                cqueue.append(buf)
                self.print_traffic("{")

            if interval and time.time() > heartbeat: # timeout_condition:
                self.refresh_view(dsn, viewid)
                heartbeat = time.time() + interval


class WebSocketProxy(websocket.WebSocketServer):
    """
    Proxy traffic to and from a WebSockets client to a normal TCP
    socket server target. All traffic to/from the client is base64
    encoded/decoded to allow binary data to be sent/received to/from
    the target.
    """

    buffer_size = 65536

    def __init__(self, RequestHandlerClass=ProxyRequestHandler, *args, **kwargs):
        # Save off proxy specific options
        self.target_host    = kwargs.pop('target_host', None)
        self.target_port    = kwargs.pop('target_port', None)
        self.wrap_cmd       = kwargs.pop('wrap_cmd', None)
        self.wrap_mode      = kwargs.pop('wrap_mode', None)
        self.unix_target    = kwargs.pop('unix_target', None)
        self.ssl_target     = kwargs.pop('ssl_target', None)
        self.target_cfg     = kwargs.pop('target_cfg', None)
        self.pidfile        = kwargs.pop('pidfile', None)
        self.logfile        = kwargs.pop('logfile', None)
        self.user           = kwargs.pop('user', None)
        self.group          = kwargs.pop('group', None)
        # Last 3 timestamps command was run
        self.wrap_times    = [0, 0, 0]

        if self.pidfile:
            self.pidfile = os.path.abspath(self.pidfile)
        if self.logfile:
            self.logfile = os.path.abspath(self.logfile)

        if self.wrap_cmd:
            wsdir = os.path.dirname(sys.argv[0])
            rebinder_path = [os.path.join(wsdir, "..", "lib"),
                             os.path.join(wsdir, "..", "lib", "websockify"),
                             wsdir]
            self.rebinder = None

            for rdir in rebinder_path:
                rpath = os.path.join(rdir, "rebind.so")
                if os.path.exists(rpath):
                    self.rebinder = rpath
                    break

            if not self.rebinder:
                raise Exception("rebind.so not found, perhaps you need to run make")
            self.rebinder = os.path.abspath(self.rebinder)

            self.target_host = "127.0.0.1"  # Loopback
            # Find a free high port
            sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            sock.bind(('', 0))
            self.target_port = sock.getsockname()[1]
            sock.close()

            os.environ.update({
                "LD_PRELOAD": self.rebinder,
                "REBIND_OLD_PORT": str(kwargs['listen_port']),
                "REBIND_NEW_PORT": str(self.target_port)})

        websocket.WebSocketServer.__init__(self, RequestHandlerClass, *args, **kwargs)

    def run_wrap_cmd(self):
        self.msg("Starting '%s'", " ".join(self.wrap_cmd))
        self.wrap_times.append(time.time())
        self.wrap_times.pop(0)
        self.cmd = subprocess.Popen(
                self.wrap_cmd, env=os.environ, preexec_fn=_subprocess_setup)
        self.spawn_message = True

    def drop_privileges(self):

        if self.user == None and self.group == None:
            return

        uid = os.getuid()
        gid = os.getgid()

        if uid != 0:
            self.log_message("Not running as root, unable to change user/group credentials")
            sys.exit(1)

        if self.user != None:
            try:
                pwbuf = pwd.getpwnam(self.user)
            except:
                self.log_message("Failed to find user information, unable to change user/group credentials")
                sys.exit(1)

            new_uid = pwbuf[2]
            new_gid = pwbuf[3]
        else:
            new_uid = uid
            new_gid = gid

        if self.group != None:
            try:
                grbuf = grp.getgrnam(self.group)
            except:
                self.log_message("Failed to find group information, unable to change user/group credentials")
                sys.exit(1)

            new_gid = grbuf[2]
        else:
            new_gid = gid

        try:
            if new_gid != gid:
                os.setgroups([new_gid])
        except:
            self.log_message("Failed to set supplementary group IDs, unable to change user/group credentials")
            sys.exit(1)

        try:
            if new_gid != gid:
                os.setgid(new_gid)
        except:
            self.log_message("Failed to set group identity, unable to change user/group credentials")
            sys.exit(1)

        try:
            if new_uid != uid:
                os.setuid(new_uid)
        except:
            self.log_message("Failed to set user identity, unable to change user/group credentials")
            sys.exit(1)


    @staticmethod
    def daemonize(keepfd=None, chdir='/'):
        logger = websocket.WebSocketServer.get_logger()

        while logger:
            for handler in logger.handlers:
                logger.removeHandler(handler)

            if logger.parent:
                logger = logger.parent
            else:
                logger = None

        websocket.WebSocketServer.daemonize(keepfd,chdir)

    def print_traffic(self, token="."):
        """ Show traffic flow mode. """
        if self.traffic and not self.daemon:
            sys.stdout.write(token)
            sys.stdout.flush()

    #
    # WebSocketServer logging/output functions
    #

    def msg(self, msg, *args, **kwargs):
        """ Output message as info """

        prefix = "%s %s websockify[%d] [info] [%03d] " % (
            datetime.datetime.now(dateutil.tz.tzlocal()).isoformat("T"),
            socket.getfqdn(),
            os.getpid(),
            self.handler_id)

        self.logger.log(logging.INFO, "%s%s" % (prefix, msg), *args, **kwargs)

    def vmsg(self, msg, *args, **kwargs):
        """ Same as msg() but as debug. """
        prefix = "%s %s websockify[%d] [debug] [%03d] " % (
            datetime.datetime.now(dateutil.tz.tzlocal()).isoformat("T"),
            socket.getfqdn(),
            os.getpid(),
            self.handler_id)

        self.logger.log(logging.DEBUG, "%s%s" % (prefix, msg), *args, **kwargs)

    def warn(self, msg, *args, **kwargs):
        """ Same as msg() but as warning. """
        prefix = "%s %s websockify[%d] [warn] [%03d] " % (
            datetime.datetime.now(dateutil.tz.tzlocal()).isoformat("T"),
            socket.getfqdn(),
            os.getpid(),
            self.handler_id)

        self.logger.log(logging.WARN, "%s%s" % (prefix, msg), *args, **kwargs)

    def log_message(self, format, *args):
        self.logger.log(logging.INFO,"%s %s websockify[%d] [info] [%03d] %s" %
                         (datetime.datetime.now(dateutil.tz.tzlocal()).isoformat("T"),
                          socket.getfqdn(),
                          os.getpid(),
                          self.handler_id,
                          format%args))

    def started(self):
        """
        Called after Websockets server startup (i.e. after daemonize)
        """
   
        if self.pidfile:
            try:
                u = os.umask(077)
                f = open(self.pidfile,"w")
                f.write("%d\n" % os.getpid())
                f.close()
                os.umask(u)
                pidfile_success = True
            except:
                pidfile_success = False

        self.drop_privileges()

        if self.logfile == None:
            self.logger.addHandler(logging.StreamHandler())
        else:
            self.logger.addHandler(logging.FileHandler(self.logfile))

        if not pidfile_success:
            self.log_message("Unable to write pid (%d) to %s" % (os.getpid(),self.pidfile))

        # Need to call wrapped command after daemonization so we can
        # know when the wrapped command exits
        if self.wrap_cmd:
            dst_string = "'%s' (port %s)" % (" ".join(self.wrap_cmd), self.target_port)
        elif self.unix_target:
            dst_string = self.unix_target
        else:
            dst_string = "%s:%s" % (self.target_host, self.target_port)

        if self.target_cfg:
            msg = "  - proxying from %s:%s to targets in %s" % (
                self.listen_host, self.listen_port, self.target_cfg)
        else:
            msg = "  - proxying from %s:%s to %s" % (
                self.listen_host, self.listen_port, dst_string)

        if self.ssl_target:
            msg += " (using SSL)"

        self.msg("%s", msg)

        if self.wrap_cmd:
            self.run_wrap_cmd()

    def poll(self):
        # If we are wrapping a command, check it's status

        if self.wrap_cmd and self.cmd:
            ret = self.cmd.poll()
            if ret != None:
                self.vmsg("Wrapped command exited (or daemon). Returned %s" % ret)
                self.cmd = None

        if self.wrap_cmd and self.cmd == None:
            # Response to wrapped command being gone
            if self.wrap_mode == "ignore":
                pass
            elif self.wrap_mode == "exit":
                sys.exit(ret)
            elif self.wrap_mode == "respawn":
                now = time.time()
                avg = sum(self.wrap_times)/len(self.wrap_times)
                if (now - avg) < 10:
                    # 3 times in the last 10 seconds
                    if self.spawn_message:
                        self.warn("Command respawning too fast")
                        self.spawn_message = False
                else:
                    self.run_wrap_cmd()


def _subprocess_setup():
    # Python installs a SIGPIPE handler by default. This is usually not what
    # non-Python successfulbprocesses expect.
    signal.signal(signal.SIGPIPE, signal.SIG_DFL)

def websockify_init():
    usage = "\n    %prog [options]"
    usage += " [source_addr:]source_port [target_addr:target_port]"
    usage += "\n    %prog [options]"
    usage += " [source_addr:]source_port -- WRAP_COMMAND_LINE"
    parser = optparse.OptionParser(usage=usage)
    parser.add_option("--verbose", "-v", action="store_true",
            help="verbose messages")
    parser.add_option("--traffic", action="store_true",
            help="per frame traffic")
    parser.add_option("--record",
            help="record sessions to FILE.[session_number]", metavar="FILE")
    parser.add_option("--daemon", "-D",
            dest="daemon", action="store_true",
            help="become a daemon (background process)")
    parser.add_option("--run-once", action="store_true",
            help="handle a single WebSocket connection and exit")
    parser.add_option("--timeout", type=int, default=0,
            help="after TIMEOUT seconds exit when not connected")
    parser.add_option("--idle-timeout", type=int, default=0,
            help="server exits after TIMEOUT seconds if there are no "
                 "active connections")
    parser.add_option("--user", default=None,
            help="File to log activity/messages to", metavar="FILE")
    parser.add_option("--group", default=None,
            help="File to log activity/messages to", metavar="FILE")
    parser.add_option("--logfile", default=None,
            help="File to log activity/messages to", metavar="FILE")
    parser.add_option("--pidfile", default="/var/run/websockify.pid",
            help="File to save process id to aid process management", metavar="FILE")
    parser.add_option("--cert", default="self.pem",
            help="SSL certificate file")
    parser.add_option("--key", default=None,
            help="SSL key file (if separate from cert)")
    parser.add_option("--ssl-only", action="store_true",
            help="disallow non-encrypted client connections")
    parser.add_option("--ssl-target", action="store_true",
            help="connect to SSL target as SSL client")
    parser.add_option("--unix-target",
            help="connect to unix socket target", metavar="FILE")
    parser.add_option("--web", default=None, metavar="DIR",
            help="run webserver on same port. Serve files from DIR.")
    parser.add_option("--wrap-mode", default="exit", metavar="MODE",
            choices=["exit", "ignore", "respawn"],
            help="action to take when the wrapped program exits "
            "or daemonizes: exit (default), ignore, respawn")
    parser.add_option("--prefer-ipv6", "-6",
            action="store_true", dest="source_is_ipv6",
            help="prefer IPv6 when resolving source_addr")
    parser.add_option("--target-config", metavar="FILE",
            dest="target_cfg",
            help="Configuration file containing valid targets "
            "in the form 'token: host:port' or, alternatively, a "
            "directory containing configuration files of this form")
    (opts, args) = parser.parse_args()

    # Sanity checks
    if len(args) < 2 and not (opts.target_cfg or opts.unix_target):
        parser.error("Too few arguments")
    if sys.argv.count('--'):
        opts.wrap_cmd = args[1:]
    else:
        opts.wrap_cmd = None
        if len(args) > 2:
            parser.error("Too many arguments")

    if not websocket.ssl and opts.ssl_target:
        parser.error("SSL target requested and Python SSL module not loaded.");

    if opts.ssl_only and not os.path.exists(opts.cert):
        parser.error("SSL only and %s not found" % opts.cert)

    # Parse host:port and convert ports to numbers
    if args[0].count(':') > 0:
        opts.listen_host, opts.listen_port = args[0].rsplit(':', 1)
        opts.listen_host = opts.listen_host.strip('[]')
    else:
        opts.listen_host, opts.listen_port = '', args[0]

    try:    opts.listen_port = int(opts.listen_port)
    except: parser.error("Error parsing listen port")

    if opts.wrap_cmd or opts.unix_target or opts.target_cfg:
        opts.target_host = None
        opts.target_port = None
    else:
        if args[1].count(':') > 0:
            opts.target_host, opts.target_port = args[1].rsplit(':', 1)
            opts.target_host = opts.target_host.strip('[]')
        else:
            parser.error("Error parsing target")
        try:    opts.target_port = int(opts.target_port)
        except: parser.error("Error parsing target port")

    # Transform to absolute path as daemon may chdir
    if opts.target_cfg:
        opts.target_cfg = os.path.abspath(opts.target_cfg)

    opts.logger = logging.getLogger('websocket')

    opts.logger.propagate = False

    if opts.verbose:
        opts.logger.setLevel(logging.DEBUG)
    else:
        opts.logger.setLevel(logging.INFO)

    if opts.logfile == None:
        h = logging.StreamHandler()
    else:
        h = logging.FileHandler(opts.logfile)

    h.setFormatter(logging.Formatter("%(message)s"))

    opts.logger.addHandler(h)

    # Create and start the WebSockets proxy
    server = WebSocketProxy(**opts.__dict__)
    
    server.start_server()


if __name__ == '__main__':
    websockify_init()
