/**
 * @package     libapache2-vncproxy
 * @file        mod_vncproxy.c
 * @author      Nicholas J. Kisseberth <nkissebe@purdue.edu>
 * @copyright   Copyright (c) 2010-2011 Purdue University. All rights reserved.
 * @license     http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
 *
 * Copyright (c) 2010-2011 Purdue University
 * All rights reserved.
 *
 * This file is part of: The HUBzero(R) Platform for Scientific Collaboration
 *
 * The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
 * software: you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * HUBzero is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * HUBzero is a registered trademark of Purdue University.
 */

/**
 * This product is derived, in part, from software developed at
 * The Apache Software Foundation (http://www.apache.org/):
 *
 * Apache HTTP Server
 * Copyright 2008 The Apache Software Foundation
 * http://httpd.apache.org/
 *
 * mod_example.c
 * mod_proxy_connect.c
 *
 * Which are used under the Apache License, Version 2.0,
 * http://www.apache.org/licenses/
 */

/**
 * @TODO:
 *
 *	- filter shouldn't be restricted to getline mode
 *	- add mode to operate independently of mod_proxy
 *    (nearly done now, disallow vncProxyUseModProxy On if no compile-time support)
 *	- close dbd connection if one was open before exec()
 *  - possibly close all open file descriptors
 *	  before exec() so we don't have that as
 *    a requirement for the helpers
 *  - remove material from mod_example and mod_proxy_client we don't need
 *  - add connection query mode w/ mod_dbd support
 *  - add connection query mode w/ apr dbd support
 *  - add encoded hostname mode support (vncsession.host.domain:port)
 *  - add host and port query results to availble helper variables
 *  - connection query + socat helper should be a usable configuration
 *  - triple check apache process lifecycle works right when do the clean
 *    shutdown rather than abrupt exit().
 *  - DBD_PARAMS and DBD_DRIVER should not be required when using helper
 *  - Use /usr/bin/vncproxy helper by default
 *  - Only required configuration (usable) should be vncProxy On
 */

#define USE_MODPROXY /* enable mod_proxy integration, may have fewer module conflict issues */

#define CORE_PRIVATE /* access to core_module */

#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_log.h"
#include "http_main.h"
#include "http_protocol.h"
#include "http_request.h"
#include "util_script.h"
#include "http_connection.h"

#include "apr_strings.h"
#include "apr_dbd.h"
#include "mod_dbd.h"
#include "apr_buckets.h"
#include "apr_general.h"
#include "apr_lib.h"
#include "util_filter.h"
#include "ap_listen.h"
#include "apr_pools.h"
#include "apr_poll.h"
#include "apr_file_io.h"

#include <assert.h>

#include <stdio.h>
#include <unistd.h> /* _exit(), execl() */

#include "mod_proxy.h"
#include "mod_ssl.h"

#define CONN_BLKSZ AP_IOBUFSIZE

/* We have to peek into the internal apr_socket_t structure
 * to pull out the raw file descriptor so we can pass it
 * to our helper program. I could find no legitimate way
 * to obtain this, there is more to the structure but I only
 * need socketdes.
 */

struct apr_socket_t
{
	apr_pool_t *pool;
	int socketdes;
};

#define EXAMPLE_LOG_EACH 0

/*
 * Declare ourselves so the configuration routines can find and know us.
 * We'll fill it in at the end of the module.
 */
module AP_MODULE_DECLARE_DATA vncproxy_module;

/*
 * vncProxy configuration record.  Used for both per-directory and per-server
 * configuration data.
 *
 * It's perfectly reasonable to have two different structures for the two
 * different environments.  The same command handlers will be called for
 * both, though, so the handlers need to be able to tell them apart.  One
 * possibility is for both structures to start with an int which is 0 for
 * one and 1 for the other.
 *
 * Note that while the per-directory and per-server configuration records are
 * available to most of the module handlers, they should be treated as
 * READ-ONLY by all except the command and merge handlers.  Sometimes handlers
 * are handed a record that applies to the current location by implication or
 * inheritance, and modifying it will change the rules for other locations.
 */

#define CONFIG_MODE_SERVER 1
#define CONFIG_MODE_DIRECTORY 2
#define CONFIG_MODE_COMBO 3     /* Shouldn't ever happen. */

typedef struct vncproxy_cfg
{
	int cmode; /* Environment to which record applies
	        * (directory, server, or combination).
	        */
	int local; /* Boolean: "Example" directive declared
	        * here?
	        */
	int congenital; /* Boolean: did we inherit an "Example"? */
	char *trace; /* Pointer to trace string. */
	char *loc; /* Location to which this record applies. */

	const char *dir;
	const char *vncProxyConnectionQuery;
	const char *_vncProxyConnectionQueryLabel;
	const char *vncProxyHelperStderr;
	const char *vncProxyHelper;
	const char *vncProxyDBDriver;
	const char *vncProxyDBDParams;
	apr_array_header_t *vncProxyAllowedSSLPorts;
	apr_array_header_t *vncProxyAllowedPorts;
	int vncProxy;
	int vncProxyUseModProxy;
	int vncProxyShutdownSSLOnExec;
} vncproxy_cfg;

static const char *vncproxy_banner() {
	return "mod_vncproxy/1.0.2";
	return ap_get_server_version();
}

/*
 * Locate our directory configuration record for the current request.
 */
static vncproxy_cfg *our_dconfig(const request_rec *r)
{
	vncproxy_cfg *cfg = NULL;
	
	if (r != NULL) 
		cfg = ap_get_module_config(r->per_dir_config, &vncproxy_module);

	return cfg;
}

/*
 * Locate our server configuration record for the specified server.
 */
static vncproxy_cfg *our_sconfig(const server_rec *s)
{
	vncproxy_cfg *cfg = NULL;
	
	if (s != NULL)
		cfg = ap_get_module_config(s->module_config, &vncproxy_module);

	return cfg;
}

/*
 * Likewise for our configuration record for the specified request.
 */
static vncproxy_cfg *our_rconfig(const request_rec *r)
{
	vncproxy_cfg *cfg = NULL;
	
	if (r != NULL)
		cfg = ap_get_module_config(r->request_config, &vncproxy_module);

	return cfg;
}

/*
 * Likewise for our configuration record for a connection.
 */
static vncproxy_cfg *our_cconfig(const conn_rec *c)
{
	vncproxy_cfg *cfg = NULL;
	
	if (c != NULL)
		cfg = ap_get_module_config(c->conn_config, &vncproxy_module);

	return cfg;
}

/*
 * Let's set up a module-local static cell to point to the accreting callback
 * trace.  As each API callback is made to us, we'll tack on the particulars
 * to whatever we've already recorded.  To avoid massive memory bloat as
 * directories are walked again and again, we record the routine/environment
 * the first time (non-request context only), and ignore subsequent calls for
 * the same routine/environment.
 */
static const char *trace = NULL;
static apr_table_t *static_calls_made = NULL;

/*
 * To avoid leaking memory from pools other than the per-request one, we
 * allocate a module-private pool, and then use a sub-pool of that which gets
 * freed each time we modify the trace.  That way previous layers of trace
 * data don't get lost.
 */
static apr_pool_t *vncproxy_pool = NULL;
static apr_pool_t *vncproxy_subpool = NULL;

/*
 * This routine sets up some module-wide cells if they haven't been already.
 */
static void setup_module_cells(void)
{
	/*
	 * If we haven't already allocated our module-private pool, do so now.
	 */
	if (vncproxy_pool == NULL) {
		apr_pool_create(&vncproxy_pool, NULL);
	};
	/*
	 * Likewise for the table of routine/environment pairs we visit outside of
	 * request context.
	 */
	if (static_calls_made == NULL) {
		static_calls_made = apr_table_make(vncproxy_pool, 16);
	};
}

/*
 * This routine is used to add a trace of a callback to the list.  We're
 * passed the server record (if available), the request record (if available),
 * a pointer to our private configuration record (if available) for the
 * environment to which the callback is supposed to apply, and some text.  We
 * turn this into a textual representation and add it to the tail of the list.
 * The list can be displayed by the vncproxy_handler() routine.
 *
 * If the call occurs within a request context (i.e., we're passed a request
 * record), we put the trace into the request apr_pool_t and attach it to the
 * request via the notes mechanism.  Otherwise, the trace gets added
 * to the static (non-request-specific) list.
 *
 * Note that the r->notes table is only for storing strings; if you need to
 * maintain per-request data of any other type, you need to use another
 * mechanism.
 */

#define TRACE_NOTE "vncproxy-trace"

static void trace_add(server_rec *s, request_rec *r, vncproxy_cfg *cfg, const char *note)
{
	const char *sofar;
	char *addon;
	char *where;
	apr_pool_t *p;
	const char *trace_copy;

	/*
	 * Make sure our pools and tables are set up - we need 'em.
	 */
	setup_module_cells();
	/*
	 * Now, if we're in request-context, we use the request pool.
	 */
	if (r != NULL) {
		p = r->pool;
		if ((trace_copy = apr_table_get(r->notes, TRACE_NOTE)) == NULL) {
			trace_copy = "";
		}
	}
	else {
		/*
		 * We're not in request context, so the trace gets attached to our
		 * module-wide pool.  We do the create/destroy every time we're called
		 * in non-request context; this avoids leaking memory in some of
		 * the subsequent calls that allocate memory only once (such as the
		 * key formation below).
		 *
		 * Make a new sub-pool and copy any existing trace to it.  Point the
		 * trace cell at the copied value.
		 */
		apr_pool_create(&p, vncproxy_pool);
		if (trace != NULL) {
			trace = apr_pstrdup(p, trace);
		}
		/*
		 * Now, if we have a sub-pool from before, nuke it and replace with
		 * the one we just allocated.
		 */
		if (vncproxy_subpool != NULL) {
			apr_pool_destroy(vncproxy_subpool);
		}
		vncproxy_subpool = p;
		trace_copy = trace;
	}
	/*
	 * If we weren't passed a configuration record, we can't figure out to
	 * what location this call applies.  This only happens for co-routines
	 * that don't operate in a particular directory or server context.  If we
	 * got a valid record, extract the location (directory or server) to which
	 * it applies.
	 */
	where = (cfg != NULL) ? cfg->loc : "nowhere";
	where = (where != NULL) ? where : "";
	/*
	 * Now, if we're not in request context, see if we've been called with
	 * this particular combination before.  The apr_table_t is allocated in the
	 * module's private pool, which doesn't get destroyed.
	 */
	if (r == NULL) {
		char *key;

		key = apr_pstrcat(p, note, ":", where, NULL);
		if (apr_table_get(static_calls_made, key) != NULL) {
			/*
			 * Been here, done this.
			 */
			return;
		}
		else {
			/*
			 * First time for this combination of routine and environment -
			 * log it so we don't do it again.
			 */
			apr_table_set(static_calls_made, key, "been here");
		}
	}
	addon = apr_pstrcat(p,
						"   <li>\n"
						"    <dl>\n"
						"     <dt><samp>", note, "</samp></dt>\n"
						"     <dd><samp>[", where, "]</samp></dd>\n"
						"    </dl>\n"
						"   </li>\n",
						NULL);
	sofar = (trace_copy == NULL) ? "" : trace_copy;
	trace_copy = apr_pstrcat(p, sofar, addon, NULL);
	if (r != NULL) {
		apr_table_set(r->notes, TRACE_NOTE, trace_copy);
	}
	else {
		trace = trace_copy;
	}
	/*
	 * You *could* change the following if you wanted to see the calling
	 * sequence reported in the server's error_log, but beware - almost all of
	 * these co-routines are called for every single request, and the impact
	 * on the size (and readability) of the error_log is considerable.
	 */

	if (EXAMPLE_LOG_EACH && (s != NULL)) {
		ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, s, "TRACE: %s", note);
	}
}

static const char *cmd_vncProxy(cmd_parms *cmd, void *mconfig, int flag)
{
	vncproxy_cfg *cfg = (vncproxy_cfg *) mconfig;

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, cmd->server, "cmd_vncProxy(%d)", flag);

	cfg->vncProxy = flag;
	return NULL;	
}

static const char *cmd_vncProxyUseModProxy(cmd_parms *cmd, void *mconfig, int flag)
{
	vncproxy_cfg *cfg = (vncproxy_cfg *) mconfig;

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, cmd->server, "cmd_vncProxyUseModProxy(%d)", flag);

	cfg->vncProxyUseModProxy = flag;
	return NULL;
}

static const char *cmd_vncProxyShutdownSSLOnExec(cmd_parms *cmd, void *mconfig, int flag)
{
	vncproxy_cfg *cfg = (vncproxy_cfg *) mconfig;

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, cmd->server, "cmd_vncProxyShutdownSSLOnExec(%d)", flag);

	cfg->vncProxyShutdownSSLOnExec = flag;
	return NULL;
}

static const char *cmd_vncProxyConnectionQuery(cmd_parms *cmd, void *mconfig, const char *param)
{
	vncproxy_cfg *cfg = (vncproxy_cfg *) mconfig;

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, cmd->server, "cmd_vncProxyConnectionQuery(\"%s\")", param);

	cfg->vncProxyConnectionQuery = param;

	return NULL;
}

static const char *cmd_vncProxyDBDriver(cmd_parms *cmd, void *mconfig, const char *param)
{
	vncproxy_cfg *cfg = (vncproxy_cfg *) mconfig;

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, cmd->server, "cmd_vncProxyDBDriver(\"%s\")", param);

	if (strlen(param) > 0)
		cfg->vncProxyDBDriver = param;

	return NULL;
}

static const char *cmd_vncProxyDBDParams(cmd_parms *cmd, void *mconfig, const char *param)
{
	vncproxy_cfg *cfg = (vncproxy_cfg *) mconfig;

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, cmd->server, "cmd_vncProxyDBDParams(*redacted* length=%d)", strlen(param));

	if (strlen(param) > 0)
		cfg->vncProxyDBDParams = param;

	return NULL;
}

/*
 * vncProxyHelper <program to run>
 *
 * Replace %t with session token
 * Replace %p with port
 * Replace %u with uri
 * Replace %r with remote ip address
 *
 * Example:
 *
 *      vncProxyHelper /usr/bin/vncproxy %t 
 *
 */

static const char *cmd_vncProxyHelperStderr(cmd_parms *cmd, void *mconfig, const char *param)
{
	vncproxy_cfg *cfg = (vncproxy_cfg *) mconfig;
	apr_status_t rv;

	if (param && *param) {
		cfg->vncProxyHelperStderr = param;
	}

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, cmd->server, "cmd_vncProxyHelperStderr(\"%s\")", param);

	return NULL;
}

static const char *cmd_vncProxyHelper(cmd_parms *cmd, void *mconfig, const char *param)
{
	vncproxy_cfg *cfg = (vncproxy_cfg *) mconfig;

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, cmd->server, "cmd_vncProxyHelper(\"%s\")", param);

	if (strlen(param) > 0) {
		cfg->vncProxyHelper = param;
	}

	return NULL;
}

static const char *cmd_vncProxyAllowedSSLPorts(cmd_parms *cmd, void *mconfig, const char *param)
{
	vncproxy_cfg *cfg = (vncproxy_cfg *) mconfig;
    int *list;

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, cmd->server, "cmd_vncProxyAllowedSSLPorts(\"%s\")", param);

    if (!apr_isdigit(param[0]))
        return "vncProxyAllowedSSLPorts: port number must be numeric";

    list = apr_array_push(cfg->vncProxyAllowedSSLPorts);
    *list = atoi(param);
    return NULL;
}

static const char *cmd_vncProxyAllowedPorts(cmd_parms *cmd, void *mconfig, const char *param)
{
	vncproxy_cfg *cfg = (vncproxy_cfg *) mconfig;
    int *New;

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, cmd->server, "cmd_vncProxyAllowedPorts(\"%s\")", param);

    if (!apr_isdigit(param[0]))
	    return "vncProxyAllowedSSLPorts: port number must be numeric";

    New = apr_array_push(cfg->vncProxyAllowedPorts);
    *New = atoi(param);
    return NULL;
}

/*
	Build ProxyHelperString with substitutions made. Its a wee bit
	inefficient as written.
 */

static const char *buildProxyHelperString(apr_pool_t *pool, const char *param, const char *token, int port,
										  const char *uri, const char *remoteip)
{
	vncproxy_cfg *cfg = NULL;
	char *p, *s, *r;

	ap_log_perror(APLOG_MARK, APLOG_DEBUG, 0, pool, "buildProxyHelperString(%s): start", param);

	if (strlen(param) <= 0) {
		ap_log_perror(APLOG_MARK, APLOG_DEBUG, 0, pool, "buildProxyHelperString(): end [no helper defined]");
		return NULL;
	}

	p = strchr(param, '%');

	if (p == NULL) {
		ap_log_perror(APLOG_MARK, APLOG_DEBUG, 0, pool, "buildProxyHelperString(): end [no substitutions needed]");
		return param;
	}

	r = apr_pstrndup(pool, param, p - param);

	s = p;

	while (*s) {
		if (s[1] == 't') {
			r = apr_pstrcat(pool, r, token, NULL);
			s += 2;
		}
		else if (s[1] == 'p') {
			if (port == 0) {
				port = 4900;
			}
			r = apr_pstrcat(pool, r, apr_itoa(pool, port), NULL);
			s += 2;
		}
		else if (s[1] == 'r') {
			if (remoteip == NULL) {
				remoteip = "unknown";
			}
			r = apr_pstrcat(pool, r, remoteip, NULL);
			s += 2;
		}
		else if (s[1] == 'u') {
			r = apr_pstrcat(pool, r, uri, NULL);
			s += 2;
		}
		else {
			r = apr_pstrcat(pool, r, "%", NULL);
			s += 1;
		}

		p = strchr(s, '%');

		if (p == NULL) {
			r = apr_pstrcat(pool, r, s, NULL);
			ap_log_perror(APLOG_MARK, APLOG_DEBUG, 0, pool, "buildProxyHelperString(): end [%s]", r);
			return (r);
		}

		r = apr_pstrcat(pool, r, apr_pstrndup(pool, s, p - s), NULL);

		s = p;
	}

	ap_log_perror(APLOG_MARK, APLOG_DEBUG, 0, pool, "buildProxyHelperString(): end [%s]", r);
	return r;
}

static const command_rec vncproxy_cmds[] = {
	AP_INIT_FLAG("vncProxy", cmd_vncProxy, NULL, RSRC_CONF, "on if VNC proxy requests should be accepted"),
	AP_INIT_FLAG("vncProxyUseModProxy", cmd_vncProxyUseModProxy, NULL, RSRC_CONF, "on if VNC proxy requests should "
		"be handled with a mod_proxy handler"),
	AP_INIT_FLAG("vncProxyShutdownSSLOnExec", cmd_vncProxyShutdownSSLOnExec, NULL, RSRC_CONF, "on if an existing"
		"SSL session should be shutdown before executing vncProxyHelper"),
	AP_INIT_ITERATE("vncProxyAllowedSSLPorts", cmd_vncProxyAllowedSSLPorts, NULL, RSRC_CONF, "A list of ports which "
		"VNC proxy CONNECT requests may connect to during an SSL session"),
	AP_INIT_ITERATE("vncProxyAllowedPorts", cmd_vncProxyAllowedPorts, NULL, RSRC_CONF, "A list of ports which "
		"VNC proxy CONNECT requests may connect to during an non-SSL session"),
	AP_INIT_TAKE1(
				"vncProxyConnectionQuery", /* directive name */
				cmd_vncProxyConnectionQuery, /* config action routine */
				NULL, /* argument to include in call */
				OR_OPTIONS, /* where available */
				"Query used to fetch hostname and port from a given token" /* directive description */
				),
	AP_INIT_TAKE1(
				"vncProxyHelperStderr", /* directive name */
				cmd_vncProxyHelperStderr, /* config action routine */
				NULL, /* argument to include in call */
				OR_OPTIONS, /* where available */
				"the filename of the vncproxy log" /* directive description */
				),
	AP_INIT_TAKE1(
				"vncProxyDBDriver", /* directive name */
				cmd_vncProxyDBDriver, /* config action routine */
				NULL, /* argument to include in call */
				OR_OPTIONS, /* where available */
				"DBD Driver Name" /* directive description */
				),
	AP_INIT_TAKE1(
				"vncProxyDBDParams", /* directive name */
				cmd_vncProxyDBDParams, /* config action routine */
				NULL, /* argument to include in call */
				OR_OPTIONS, /* where available */
				"DBD Parameters" /* directive description */
				),
	AP_INIT_TAKE1(
				"vncProxyHelper", /* directive name */
				cmd_vncProxyHelper, /* config action routine */
				NULL, /* argument to include in call */
				OR_OPTIONS, /* where available */
				"vncProxy Helper Executable" /* directive description */
				),
	{
		NULL
	}
};

/*
 * This function gets called to create a per-directory configuration
 * record.  This will be called for the "default" server environment, and for
 * each directory for which the parser finds any of our directives applicable.
 * If a directory doesn't have any of our directives involved (i.e., they
 * aren't in the .htaccess file, or a <Location>, <Directory>, or related
 * block), this routine will *not* be called - the configuration for the
 * closest ancestor is used.
 *
 * The return value is a pointer to the created module-specific
 * structure.
 */
static void *vncproxy_create_dir_config(apr_pool_t *p, char *dirspec)
{
	vncproxy_cfg *cfg;
	char *dname = dirspec;

	cfg = (vncproxy_cfg *) apr_pcalloc(p, sizeof (vncproxy_cfg));

	ap_log_perror(APLOG_MARK, APLOG_DEBUG, 0, p, "vncproxy_create_dir_config(p,\"%s\")", dirspec);

	/*
	 * Now fill in the defaults.  If there are any `parent' configuration
	 * records, they'll get merged as part of a separate callback.
	 */
	cfg->local = 0;
	cfg->congenital = 0;
	cfg->cmode = CONFIG_MODE_DIRECTORY;
	cfg->vncProxyConnectionQuery = NULL;
	cfg->vncProxyHelperStderr = "/var/log/vncproxy/vncproxy.log";
	cfg->vncProxyDBDriver = "mysql";
	cfg->vncProxyDBDParams = "";
	cfg->vncProxyHelper = NULL;
	cfg->vncProxy = 0;
	cfg->vncProxyUseModProxy = 0;
	cfg->vncProxyShutdownSSLOnExec = 1;
	cfg->vncProxyAllowedSSLPorts = apr_array_make(p, 10, sizeof(int));
	cfg->vncProxyAllowedPorts = apr_array_make(p, 10, sizeof(int));

	/*
	 * Finally, add our trace to the callback list.
	 */
	dname = (dname != NULL) ? dname : "";
	cfg->loc = apr_pstrcat(p, "DIR(", dname, ")", NULL);
	cfg->dir = apr_pstrcat(p, dname, NULL);
	trace_add(NULL, NULL, cfg, "vncproxy_create_dir_config()");

	return (void *) cfg;
}

/*
 * This function gets called to merge two per-directory configuration
 * records.  This is typically done to cope with things like .htaccess files
 * or <Location> directives for directories that are beneath one for which a
 * configuration record was already created.  The routine has the
 * responsibility of creating a new record and merging the contents of the
 * other two into it appropriately.  If the module doesn't declare a merge
 * routine, the record for the closest ancestor location (that has one) is
 * used exclusively.
 *
 * The routine MUST NOT modify any of its arguments!
 *
 * The return value is a pointer to the created module-specific structure
 * containing the merged values.
 */
static void *vncproxy_merge_dir_config(apr_pool_t *p, void *parent_conf, void *newloc_conf)
{
	vncproxy_cfg *cfg = (vncproxy_cfg *) apr_pcalloc(p, sizeof (vncproxy_cfg));
	vncproxy_cfg *pcfg = (vncproxy_cfg *) parent_conf;
	vncproxy_cfg *ncfg = (vncproxy_cfg *) newloc_conf;
	char *note;

	ap_log_perror(APLOG_MARK, APLOG_DEBUG, 0, p, "vncproxy_merge_dir_config()");

	/*
	 * Some things get copied directly from the more-specific record, rather
	 * than getting merged.
	 */
	cfg->local = ncfg->local;
	cfg->loc = apr_pstrdup(p, ncfg->loc);
	cfg->dir = apr_pstrdup(p, ncfg->dir);
	cfg->vncProxyConnectionQuery = ncfg->vncProxyConnectionQuery;
	cfg->vncProxyDBDriver = ncfg->vncProxyDBDriver;
	cfg->vncProxyDBDParams = ncfg->vncProxyDBDParams;
	cfg->vncProxyHelperStderr = ncfg->vncProxyHelperStderr;
	cfg->vncProxyHelper = ncfg->vncProxyHelper;
	cfg->vncProxy = ncfg->vncProxy;
	cfg->vncProxyUseModProxy = ncfg->vncProxyUseModProxy;
	cfg->vncProxyShutdownSSLOnExec = ncfg->vncProxyShutdownSSLOnExec;
	cfg->vncProxyAllowedSSLPorts = apr_array_append(p, pcfg->vncProxyAllowedSSLPorts, ncfg->vncProxyAllowedSSLPorts);
	cfg->vncProxyAllowedPorts = apr_array_append(p, pcfg->vncProxyAllowedPorts, ncfg->vncProxyAllowedPorts);
	/*
	 * Others, like the setting of the `congenital' flag, get ORed in.  The
	 * setting of that particular flag, for instance, is TRUE if it was ever
	 * true anywhere in the upstream configuration.
	 */
	cfg->congenital = (pcfg->congenital | pcfg->local);

	/*
	 * If we're merging records for two different types of environment (server
	 * and directory), mark the new record appropriately.  Otherwise, inherit
	 * the current value.
	 */
	cfg->cmode = (pcfg->cmode == ncfg->cmode) ? pcfg->cmode : CONFIG_MODE_COMBO;

	/*
	 * Now just record our being called in the trace list.  Include the
	 * locations we were asked to merge.
	 */
	note = apr_pstrcat(p, "vncproxy_merge_dir_config(\"", pcfg->loc, "\",\"", ncfg->loc, "\")", NULL);
	trace_add(NULL, NULL, cfg, note);

	return (void *) cfg;
}

/*
 * This function gets called to create a per-server configuration
 * record.  It will always be called for the "default" server.
 *
 * The return value is a pointer to the created module-specific
 * structure.
 */
static void *vncproxy_create_server_config(apr_pool_t *p, server_rec *s)
{
	vncproxy_cfg *cfg;
	cfg = (vncproxy_cfg *) apr_pcalloc(p, sizeof (vncproxy_cfg));
	char *sname = s->server_hostname;

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, s, "vncproxy_create_server_config()");

	/*
	 * As with the vncproxy_create_dir_config() routine, we allocate and fill
	 * in an empty record.
	 */
	cfg->local = 0;
	cfg->congenital = 0;
	cfg->cmode = CONFIG_MODE_SERVER;
	cfg->vncProxyConnectionQuery = NULL;
	cfg->vncProxyDBDriver = "mysql";
	cfg->vncProxyDBDParams = "";
	cfg->vncProxyHelperStderr = "/var/log/vncproxy/vncproxy.log";
	cfg->vncProxyHelper = NULL;
	cfg->vncProxy = 0;
	cfg->vncProxyUseModProxy = 0;
	cfg->vncProxyShutdownSSLOnExec = 1;
	cfg->vncProxyAllowedSSLPorts = apr_array_make(p, 10, sizeof(int));
	cfg->vncProxyAllowedPorts = apr_array_make(p, 10, sizeof(int));

	/*
	 * Note that we were called in the trace list.
	 */
	sname = (sname != NULL) ? sname : "";
	cfg->loc = apr_pstrcat(p, "SVR(", sname, ")", NULL);
	trace_add(s, NULL, cfg, "vncproxy_create_server_config()");

	return (void *) cfg;
}

/*
 * This function gets called to merge two per-server configuration
 * records.  This is typically done to cope with things like virtual hosts and
 * the default server configuration  The routine has the responsibility of
 * creating a new record and merging the contents of the other two into it
 * appropriately.  If the module doesn't declare a merge routine, the more
 * specific existing record is used exclusively.
 *
 * The routine MUST NOT modify any of its arguments!
 *
 * The return value is a pointer to the created module-specific structure
 * containing the merged values.
 */
static void *vncproxy_merge_server_config(apr_pool_t *p, void *base_conf, void *virt_conf)
{
	vncproxy_cfg *cfg = (vncproxy_cfg *) apr_pcalloc(p, sizeof (vncproxy_cfg));
	vncproxy_cfg *base = (vncproxy_cfg *) base_conf;
	vncproxy_cfg *virt = (vncproxy_cfg *) virt_conf;
	char *note;

	ap_log_perror(APLOG_MARK, APLOG_DEBUG, 0, p, "vncproxy_merge_server_config()");
	/*
	 * Our inheritance rules are our own, and part of our module's semantics.
	 * Basically, just note whence we came.
	 */
	cfg->cmode = (base->cmode == virt->cmode) ? base->cmode : CONFIG_MODE_COMBO;
	cfg->local = virt->local;
	cfg->congenital = (base->congenital | virt->local);
	cfg->loc = apr_pstrdup(p, virt->loc);
	ap_log_perror(APLOG_MARK, APLOG_DEBUG, 0, p, "base->vncProxyHelper = %s", base->vncProxyHelper);
	ap_log_perror(APLOG_MARK, APLOG_DEBUG, 0, p, "virt->vncProxyHelper = %s", virt->vncProxyHelper);

	cfg->vncProxyConnectionQuery =
		(virt->vncProxyConnectionQuery == NULL) ? base->vncProxyConnectionQuery : virt->vncProxyConnectionQuery;
	cfg->vncProxyDBDParams =
		(virt->vncProxyDBDParams == NULL) ? base->vncProxyDBDParams : virt->vncProxyDBDParams;
	cfg->vncProxyDBDriver =
		(virt->vncProxyDBDriver == NULL) ? base->vncProxyDBDriver : virt->vncProxyDBDriver;
	cfg->vncProxyHelperStderr =
		(virt->vncProxyHelperStderr == NULL) ? base->vncProxyHelperStderr : virt->vncProxyHelperStderr;
	cfg->vncProxyHelper =
		(virt->vncProxyHelper == NULL) ? base->vncProxyHelper : virt->vncProxyHelper;
	cfg->vncProxy =
		(virt->vncProxy == 0) ? base->vncProxy : virt->vncProxy;
	cfg->vncProxyUseModProxy =
		(virt->vncProxyUseModProxy == 0) ? base->vncProxyUseModProxy : virt->vncProxyUseModProxy;
	cfg->vncProxyShutdownSSLOnExec =
		(virt->vncProxyShutdownSSLOnExec == 0) ? base->vncProxyShutdownSSLOnExec : virt->vncProxyShutdownSSLOnExec;
	cfg->vncProxyAllowedSSLPorts = apr_array_append(p, base->vncProxyAllowedSSLPorts, virt->vncProxyAllowedSSLPorts);
	cfg->vncProxyAllowedPorts = apr_array_append(p, base->vncProxyAllowedPorts, virt->vncProxyAllowedPorts);

	/*
	 * Trace our call, including what we were asked to merge.
	 */
	note = apr_pstrcat(p, "vncproxy_merge_server_config(\"", base->loc, "\",\"", virt->loc, "\")", NULL);
	trace_add(NULL, NULL, cfg, note);

	return (void *) cfg;
}

static int
open_database(vncproxy_cfg *cfg, const apr_dbd_driver_t *driver, request_rec *r, const char *dbd_params,
			  apr_dbd_t **handle)
{

	int delay = 1;
	int tries = 1;

	while (apr_dbd_open(driver, r->pool, dbd_params, handle) != APR_SUCCESS) {
		if (tries > 6) {
			ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "Unable to open database connection. Tried %d times. Aborting", tries);
			return -1;
		}

		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "Unable to open database connection. Retrying in %ds", delay);

		sleep(delay);

		delay *= 2;

		if (delay > 8) {
			delay = 8;
		}

		tries++;
	}

	return 0;
}

static int translate_aprdbd(request_rec *r, const char *token, const char **host, int *port)
{
	vncproxy_cfg *cfg;
	apr_status_t rv;
	apr_dbd_prepared_t *statement;
	apr_dbd_results_t *res = NULL;
	apr_dbd_row_t *row = NULL;
	const char *dbd_hostname = "";
	const char *dbd_port = "";
	ap_dbd_t *dbd = NULL;
	conn_rec *c = r->connection;
	const apr_dbd_driver_t *driver;
	char query[1024];
	const char *etoken;
	apr_dbd_t *handle;
	const char *value;
	char hostname[41];
	int portnum;

	cfg = our_dconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "translate_aprdbd(\"%s\"", token);

	if (strcmp(token, "") == 0) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "No session token available to translate");
		return -1;
	}

	if (cfg->vncProxyDBDriver == "") {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "No database driver specified");
		return -1;
	}

	if (cfg->vncProxyDBDParams == NULL) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "No database driver parameters specified");
		return -1;
	}

	rv = apr_dbd_get_driver(r->pool, cfg->vncProxyDBDriver, &driver);

	if (rv != APR_SUCCESS) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "Unable to load APR_DBD driver [%s]", cfg->vncProxyDBDriver);
		return -1;
	}

	if (cfg->vncProxyConnectionQuery == NULL) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "No vncProxyConnectionQuery specified");
		return -1;
	}

	if (open_database(cfg, driver, r, cfg->vncProxyDBDParams, &handle) != 0) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "Unable to open database to lookup token");
		return -1;
	}

	etoken = apr_dbd_escape(driver, r->pool, token, handle);

	if (etoken == NULL) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "Unable to build query safe token string");
		return -1;
	}

	rv = snprintf(query, sizeof (query), cfg->vncProxyConnectionQuery, etoken);

	if (rv < 0 || rv >= sizeof (query)) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "Failed to build query string");
		return -1;
	}

	rv = apr_dbd_select(driver, r->pool, handle, &res, query, 0);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, query);

	row = NULL;

	if ((rv != APR_SUCCESS) || (apr_dbd_get_row(driver, r->pool, res, &row, -1) != 0)) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "No database entry found for token [%s]", token);

		rv = apr_dbd_close(driver, handle);

		if (rv != APR_SUCCESS) {
			ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "Unable to close database connection");
			return -1;
		}

		return 0;
	}

	value = apr_dbd_get_entry(driver, row, 0);

	if (value == NULL) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "Unable to read hostname column from result row");
		return -1;
	}

	strncpy(hostname, value, sizeof (hostname));

	value = apr_dbd_get_entry(driver, row, 1);

	if (value == NULL) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "Unable to read host column from result row");
		return -1;
	}

	portnum = atoi(value);

	if (portnum <= 0) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "Invalid port number read from result row");
		return -1;
	}

	rv = apr_dbd_get_row(driver, r->pool, res, &row, -1); // clears connection

	if (rv != -1) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "Database transaction not finished when expected.");
		return -1;
	}

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "Query execution complete %s %d", hostname, portnum);
	*host = apr_pstrdup(r->pool, hostname);
	*port = portnum;

	return 0;
}

static int translate_moddbd(request_rec *r, const char *token, const char **hostname, int *port)
{
	vncproxy_cfg *cfg;
	apr_status_t rv;
	apr_dbd_prepared_t *statement;
	apr_dbd_results_t *res = NULL;
	apr_dbd_row_t *row = NULL;
	const char *dbd_hostname = NULL;
	const char *dbd_port = NULL;
	ap_dbd_t *dbd = NULL;
	conn_rec *c = r->connection;
	static ap_dbd_t * (*vncproxy_dbd_acquire_fn)(request_rec*) = NULL;
	static void (*vncproxy_dbd_prepare_fn)(server_rec*, const char*, const char*) = NULL;
	static unsigned int label_num = 0;
	char *label;
	const char *value;

	cfg = our_dconfig(r);

	if (vncproxy_dbd_prepare_fn == NULL) {
		vncproxy_dbd_prepare_fn = APR_RETRIEVE_OPTIONAL_FN(ap_dbd_prepare);
		if (vncproxy_dbd_prepare_fn == NULL) {
			ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "You must load mod_dbd to enable vncProxy DBD functions");
			return -1;
		}
		vncproxy_dbd_acquire_fn = APR_RETRIEVE_OPTIONAL_FN(ap_dbd_acquire);
	}

	if (cfg->_vncProxyConnectionQueryLabel == NULL) {
		cfg->_vncProxyConnectionQueryLabel = apr_psprintf(r->pool, "vncproxy_dbd_%d", label_num++);

		vncproxy_dbd_prepare_fn(r->server, cfg->vncProxyConnectionQuery, cfg->_vncProxyConnectionQueryLabel);
	}

	trace_add(r->server, NULL, NULL, cfg->vncProxyConnectionQuery);
	trace_add(r->server, NULL, NULL, cfg->_vncProxyConnectionQueryLabel);

	if (strcmp(token, "") == 0) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "No session token available to translate");
		return -1;
	}

	if (vncproxy_dbd_acquire_fn != NULL) {
		dbd = vncproxy_dbd_acquire_fn(r);
	}

	if (dbd == NULL) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "Failed to acquire database connection to look up " "token '%s'", token);
		return -1;
	}

	if (cfg->vncProxyConnectionQuery == NULL) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "Unable to find vncProxyConnectionQuery info");
		return -1;
	}

	statement = apr_hash_get(dbd->prepared, cfg->vncProxyConnectionQuery, APR_HASH_KEY_STRING);

	if (statement == NULL) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "A prepared statement could not be found for "
				"vncProxyConnectionQuery with the key '%s'", cfg->vncProxyConnectionQuery);
		return -1;
	}

	res = NULL;

	if (apr_dbd_pvselect(dbd->driver, r->pool, dbd->handle, &res, statement, 0, token, NULL) != 0) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "Query execution error looking up '%s' " "in database", token);
		return -1;
	}

	row = NULL;
	rv = apr_dbd_get_row(dbd->driver, r->pool, res, &row, -1);

	while (rv != -1) {
		if (rv != 0) {
			ap_log_rerror(APLOG_MARK, APLOG_ERR, rv, r, "Error retrieving results while looking up '%s' " "in database", token);
			return -1;
		}

		if (dbd_hostname == NULL) {
			value = apr_dbd_get_entry(dbd->driver, row, 0);

			if (value == NULL) {
				ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "Unable to retrieve hostname from record");
				return -1;
			}

			dbd_hostname = value;

			value = apr_dbd_get_entry(dbd->driver, row, 1);

			if (value == NULL) {
				ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "Unable to retrieve port number from record");
				return -1;
			}

			dbd_port = value;
		}
		/* we can't break out here or row won't get cleaned up */
		rv = apr_dbd_get_row(dbd->driver, r->pool, res, &row, -1);
	}


	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "Query execution complete %s %s", dbd_hostname, dbd_port);
	*hostname = apr_pstrdup(r->pool, dbd_hostname);
	*port = atoi(dbd_port);

	return 0;
}

static APR_OPTIONAL_FN_TYPE(ssl_is_https) *proxy_is_https = NULL;

static int allowed_port(request_rec *r, proxy_server_conf *conf, int port)
{
	int i = 0;
	int *list = NULL;
	apr_array_header_t *ar = NULL;
	vncproxy_cfg *cfg = NULL;
	int default_sslports[1] = { 4900 };
    int default_ports[2] = { 4900, 5900 };
	int n;

	if (proxy_is_https == NULL) {
		proxy_is_https = APR_RETRIEVE_OPTIONAL_FN(ssl_is_https);
	}

	cfg = our_dconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "allowed_port(%d): start", port);

	if (cfg->vncProxyUseModProxy) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "allowed_port(%d): mod_proxy AllowedCONNECT check", port);
		if (conf == NULL || conf->allowed_connect_ports->nelts == 0) {
			/* Default setting if not overridden by AllowCONNECT */
			if ( port == APR_URI_HTTPS_DEFAULT_PORT || port == APR_URI_SNEWS_DEFAULT_PORT ) {
				ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "allowed_port(%d): end [allowed, default rules]", port);
				return 1;
			}
		}

		ar = (conf) ? conf->allowed_connect_ports : NULL;
	}
	else if (proxy_is_https && proxy_is_https(r->connection)) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "allowed_port(%d): vncProxyAllowedSSLPorts check", port);

		list =  default_sslports;
		n = sizeof(default_sslports);
	
		if (cfg && cfg->vncProxyAllowedSSLPorts && cfg->vncProxyAllowedSSLPorts->nelts > 0) {
			list = (int *) cfg->vncProxyAllowedSSLPorts->elts;
			n = cfg->vncProxyAllowedSSLPorts->nelts;
		}
	} else {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "allowed_port(%d): vncProxyAllowedPorts check", port);

		list =  default_ports;
		n = sizeof(default_ports);
	
		if (cfg && cfg->vncProxyAllowedPorts && cfg->vncProxyAllowedPorts->nelts > 0) {
			list = (int *) cfg->vncProxyAllowedPorts->elts;
			n = cfg->vncProxyAllowedPorts->nelts;
		}
	}

	for (i = 0; i < n; i++) {
		if ((port == list[i]) && (port != 0)) {
			ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "allowed_port(%d): end [allowed]", port);
			return 1;
		}
	}

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "allowed_port(%d): end [denied]", port);
	return 0;
}

static int vncproxy_connect(request_rec *r)
{
	apr_socket_t *sock;
	apr_bucket_brigade *bb = apr_brigade_create(r->pool, r->connection->bucket_alloc);
	apr_status_t rv;
	apr_size_t nbytes;
	char buffer[HUGE_STRING_LEN];
	apr_socket_t *client_socket = ap_get_module_config(r->connection->conn_config, &core_module);
	apr_sockaddr_t *uri_addr;
	apr_uri_t uri;
	char *token;
	vncproxy_cfg *cfg = NULL;
	int status = 0;
	pid_t pid;
	const char *vncProxyHelper;
	char **argv_out;
	apr_bucket *eos, *e;
	int fd = -1;
	int maxfd = 0;

	proxy_is_https = APR_RETRIEVE_OPTIONAL_FN(ssl_is_https);

	cfg = our_dconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_connect(): start");

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_connect(): URL [%s]", r->unparsed_uri);

	if (APR_SUCCESS != apr_uri_parse_hostinfo(r->pool, r->unparsed_uri, &uri)) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_connect(): end [URI cannot be parsed %s]", r->unparsed_uri);
		return HTTP_BAD_REQUEST;
	}

	if (strncmp(uri.hostname, "vncsession.", 11) != 0) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_connect(): end [declined]");
		return DECLINED;
	}

	token = uri.hostname + 11;

	ap_log_rerror(APLOG_MARK, APLOG_INFO, 0, r, "vncproxy_connect(): connecting to %s:%d", uri.hostname, uri.port);

	/* do a DNS lookup for the destination host */

	rv = apr_sockaddr_info_get(&uri_addr, NULL /* uri.hostname */, APR_UNSPEC, uri.port, 0, r->pool);

	if (APR_SUCCESS != rv) {
		ap_log_rerror(APLOG_MARK, APLOG_WARNING, 0, r, "vncproxy_connect(): end [DNS lookup failure for: %s]", uri.hostname);
		return HTTP_BAD_GATEWAY;
	}

	uri_addr->hostname = uri.hostname;

	/* Check if it is an allowed port */
	if (!allowed_port(r, NULL, uri.port)) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_connect(): end [connect to remote machine blocked]");
		return HTTP_FORBIDDEN;
	}

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_connect(): handle connection");

	nbytes = apr_snprintf(buffer, sizeof (buffer), "HTTP/1.0 200" CRLF);
	ap_xlate_proto_to_ascii(buffer, nbytes);
	ap_fwrite(r->connection->output_filters, bb, buffer, nbytes);
	ap_log_rerror(APLOG_MARK, APLOG_INFO, 0, r, "vncproxy_connect(): HTTP/1.0 200 Connection Established");

	nbytes = apr_snprintf(buffer, sizeof (buffer), "Proxy-agent: %s" CRLF CRLF, vncproxy_banner());
	ap_xlate_proto_to_ascii(buffer, nbytes);
	ap_fwrite(r->connection->output_filters, bb, buffer, nbytes);

	ap_fflush(r->connection->output_filters, bb);

	if (cfg && cfg->vncProxyShutdownSSLOnExec) {
		/* Send end of stream to trigger SSL module to send close-notify alert */
		e = ap_bucket_eoc_create(r->connection->bucket_alloc);
		eos = APR_BRIGADE_LAST(bb);
		while ((APR_BRIGADE_SENTINEL(bb) != eos)
			&& !APR_BUCKET_IS_EOS(eos)) {
			eos = APR_BUCKET_PREV(eos);
		}
		if (eos == APR_BRIGADE_SENTINEL(bb)) {
			APR_BRIGADE_INSERT_TAIL(bb, e);
		}
		else {
			APR_BUCKET_INSERT_BEFORE(eos, e);
		}
		ap_pass_brigade(r->connection->output_filters, bb);
	}
	
	vncProxyHelper = buildProxyHelperString(r->pool, cfg->vncProxyHelper, token, 0, "", r->connection->remote_ip);
	
	apr_tokenize_to_argv(vncProxyHelper, &argv_out, r->pool);

	if (chdir("/") != 0) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "vncproxy_connect(): end [unable to change directory to /]");
		return HTTP_INTERNAL_SERVER_ERROR;
	}

	umask(0);

	pid = fork();

	if (pid == -1) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "vncproxy_connect(): end [backgrounding fork() failed]");
		return HTTP_INTERNAL_SERVER_ERROR;
	}

	if (pid == 0) {

		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_connect(): Background process [%d] created", getpid());

		rv = setsid();

		if (rv == -1) {
			ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "vncproxy_connect(): Can't create new process session");
			_exit(1);
		}

		pid = fork();

		if (pid == -1) {
			ap_log_rerror(APLOG_MARK, APLOG_WARNING, 0, r, "vncproxy_connect(): Failed to fork() prior to execution of vncProxyHelper [%s](%d)",
					vncProxyHelper, errno);
			_exit(2);
		}

		if (pid == 0) {
			ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_connect(): Disassociated process [%d] created.", getpid());

			if (cfg && cfg->vncProxyDBDriver)
				setenv("DBD_DRIVER", cfg->vncProxyDBDriver, 1);
			else
				setenv("DBD_DRIVER", "mysql", 1);

			if (cfg && cfg->vncProxyDBDParams)
				setenv("DBD_PARAMS", cfg->vncProxyDBDParams, 1);
			else
				setenv("DBD_PARAMS", "", 1);

			dup2(client_socket->socketdes, 0);
			dup2(client_socket->socketdes, 1);

			if (cfg->vncProxyHelperStderr) {

				fd = open(cfg->vncProxyHelperStderr, O_NOFOLLOW | O_APPEND | O_WRONLY | O_CREAT | O_NONBLOCK, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP);

				if (fd < 0) 
				{
					ap_log_rerror(APLOG_MARK, APLOG_WARNING, 0, r, "Unable to open logfile [%s]", cfg->vncProxyHelperStderr);
				}
				else {
					dup2(fd, 2);
					close(fd);
				}
			}

			ap_log_rerror(APLOG_MARK, APLOG_INFO, 0, r, "Executing [%s]", vncProxyHelper);

			maxfd = sysconf(_SC_OPEN_MAX);

			for (fd = 3; fd < maxfd; fd++) {
				rv = close(fd);

				if (rv == -1) {
					if (errno == EBADF)
						continue;

					ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "vncproxy_connect(): Failed to close file descriptor [%d]", fd);
				} else if (rv == 0) {
					ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_connect(): Closed file descriptor [%d]", fd);
				}
			}

			execvp(argv_out[0], argv_out);

			ap_log_rerror(APLOG_MARK, APLOG_WARNING, 0, r, "vncproxy_connect(): Failed to execute vncProxyHelper [%s](%d)",
					vncProxyHelper, errno);

			_exit(3);
		}

		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_connect(): Exiting background process [%d]", getpid());
		_exit(0); // let session leader parent die
	}

	waitpid(pid, &status, 0); // wait for intermediate background process to exit

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_connect(): Apache parent worker process [%d] continuing", getpid());

	/* abort connection */
	r->input_filters = NULL; // Parent will shutdown SSL session if we don't clear filters and signal abort
	r->output_filters = NULL;
	r->connection->output_filters = NULL;
	r->proto_output_filters = NULL;
	r->connection->input_filters = NULL;
	r->input_filters = NULL;
	r->connection->aborted = 1;
	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_connect(): end [success]");
	return OK;
}

static int vncproxy_handler(request_rec *r)
{
	vncproxy_cfg *cfg;
	apr_status_t rv;

	cfg = our_dconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_handler(): start");

	if (!cfg || !cfg->vncProxy) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_handler(): end [declined, vncProxy Off]");
		return DECLINED;
	}

	if (cfg->vncProxyUseModProxy) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_handler(): end [declined, vncProxyUseModProxy On]");
		return DECLINED;
	}

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_handler(): [%s]", r->the_request);

	if (strcmp(r->handler, "vncproxy-handler") == 0) {

		if (r->proxyreq && cfg->vncProxyUseModProxy) {
			ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_handler(): end [declined, mod_proxy in use]");
			return DECLINED;
		}

		rv = vncproxy_connect(r);

		if (rv == OK) {
			ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_handler(): end [connect, ok]");
		} else {
			ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_handler(): end [connect, failed (%d)]", rv);
		}

		return rv;
	}

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_handler(): end [declined, not marked for vncproxy-handler]");
	return DECLINED;
}

/*--------------------------------------------------------------------------*/
/*                                                                          */
/* Now let's declare routines for each of the callback phase in order.      */
/* (That's the order in which they're listed in the callback list, *not     */
/* the order in which the server calls them!  See the command_rec           */
/* declaration near the bottom of this file.)  Note that these may be       */
/* called for situations that don't relate primarily to our function - in   */
/* other words, the fixup handler shouldn't assume that the request has     */
/* to do with "example" stuff.                                              */
/*                                                                          */
/* With the exception of the content handler, all of our routines will be   */
/* called for each request, unless an earlier handler from another module   */
/* aborted the sequence.                                                    */
/*                                                                          */
/* Handlers that are declared as "int" can return the following:            */
/*                                                                          */
/*  OK          Handler accepted the request and did its thing with it.     */
/*  DECLINED    Handler took no action.                                     */
/*  HTTP_mumble Handler looked at request and found it wanting.             */
/*                                                                          */
/* What the server does after calling a module handler depends upon the     */
/* handler's return value.  In all cases, if the handler returns            */
/* DECLINED, the server will continue to the next module with an handler    */
/* for the current phase.  However, if the handler return a non-OK,         */
/* non-DECLINED status, the server aborts the request right there.  If      */
/* the handler returns OK, the server's next action is phase-specific;      */
/* see the individual handler comments below for details.                   */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#ifdef ALL_HOOKS

/*
 * This routine is called before the server processes the configuration
 * files.  There is no return value.
 */
static int vncproxy_pre_config(apr_pool_t *pconf, apr_pool_t *plog, apr_pool_t *ptemp)
{
	vncproxy_cfg *cfg = NULL;

	ap_log_perror(APLOG_MARK, APLOG_DEBUG, 0, p, "vncproxy_pre_config()");

	/* Log the call and exit. */
	trace_add(NULL, NULL, NULL, "vncproxy_pre_config()");

	return OK;
}
#endif

#ifdef ALL_HOOKS

/*
 * This routine is called to perform any module-specific fixing of header
 * fields, et cetera.  It is invoked just before any content-handler.
 *
 * The return value is OK, DECLINED, or HTTP_mumble.  If we return OK, the
 * server will still call any remaining modules with an handler for this
 * phase.
 */
static int vncproxy_post_config(apr_pool_t *pconf, apr_pool_t *plog, apr_pool_t *ptemp, server_rec *s)
{
	vncproxy_cfg *cfg = NULL;

	cfg = our_sconfig(s);

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, s, "vncproxy_post_config()");

	/*
	 * Log the call and exit.
	 */
	trace_add(NULL, NULL, NULL, "vncproxy_post_config()");
	return OK;
}
#endif

#ifdef ALL_HOOKS

/*
 * This routine is called to perform any module-specific log file
 * openings. It is invoked just before the post_config phase
 *
 * The return value is OK, DECLINED, or HTTP_mumble.  If we return OK, the
 * server will still call any remaining modules with an handler for this
 * phase.
 */
static int vncproxy_open_logs(apr_pool_t *pconf, apr_pool_t *plog, apr_pool_t *ptemp, server_rec *s)
{
	vncproxy_cfg *cfg = NULL;

	cfg = our_sconfig(s);

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, s, "vncproxy_open_logs()");

	/*
	 * Log the call and exit.
	 */
	trace_add(s, NULL, NULL, "vncproxy_open_logs()");

	for (; s; s = s->next) {
		if (!open_log(s, plog)) {
			return HTTP_INTERNAL_SERVER_ERROR;
		}
	}

	return OK;
}
#endif

#ifdef ALL_HOOKS

/*
 * All our process-death routine does is add its trace to the log.
 */
static apr_status_t vncproxy_child_exit(void *data)
{
	char *note;
	server_rec *s = data;
	char *sname = s->server_hostname;

	vncproxy_cfg *cfg = NULL;

	cfg = our_sconfig(s);

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, s, "vncproxy_child_exit()");

	/*
	 * The arbitrary text we add to our trace entry indicates for which server
	 * we're being called.
	 */
	sname = (sname != NULL) ? sname : "";
	note = apr_pstrcat(s->process->pool, "vncproxy_child_exit(", sname, ")", NULL);
	trace_add(s, NULL, NULL, note);
	return APR_SUCCESS;
}
#endif

#ifdef ALL_HOOKS

/*
 * All our process initialiser does is add its trace to the log.
 */
static void vncproxy_child_init(apr_pool_t *p, server_rec *s)
{
	char *note;
	char *sname = s->server_hostname;

	vncproxy_cfg *cfg = NULL;

	cfg = our_sconfig(s);

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, s, "vncproxy_child_init()");

	/*
	 * Set up any module cells that ought to be initialised.
	 */
	setup_module_cells();
	/*
	 * The arbitrary text we add to our trace entry indicates for which server
	 * we're being called.
	 */
	sname = (sname != NULL) ? sname : "";
	note = apr_pstrcat(p, "vncproxy_child_init(", sname, ")", NULL);
	trace_add(s, NULL, NULL, note);

	apr_pool_cleanup_register(p, s, vncproxy_child_exit, vncproxy_child_exit);
}
#endif

/*
 * This method is registered by ap_hook_create_request()
 * This hook allows modules to modify a request while it is being
 * created.  This hook is called for all request_rec's, main request,
 * sub request, and internal redirect.  When this hook is called, the
 * the r->main, r->prev, r->next pointers have been set, so modules
 * can determine what kind of request this is.
 *
 * This can be used for adding filters that should ONLY be run in a
 * subrequest, or ONLY run in a main request.
 *
 * You should return OK in this function on success.
 *
 * XXX - nobody checks the return value of this function so who knows
 *       what the correct behavior is
 */
static int vncproxy_create_request(request_rec *r)
{
	vncproxy_cfg *cfg = NULL;

	if (!r->main && !r->prev) {

		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_create_request(pri): start");
		ap_add_input_filter("VNCPROXY_INPUT_FILTER", NULL, r, r->connection);
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_create_request(pri): ADD vncproxy_input_filter()");
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_create_request(pri): end [ok]");
	}
	else {
		ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, 0, "vncproxy_create_request(sub): start");
		ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, 0, "vncproxy_create_request(sub): end [ok]");
	}

	return OK;
}

#ifdef ALL_HOOKS

/*
 * XXX: This routine is called XXX
 *
 * The return value is OK, DECLINED, or HTTP_mumble.  If we return OK, the
 * server will still call any remaining modules with an handler for this
 * phase.
 */
static const char *vncproxy_http_scheme(const request_rec *r)
{
	vncproxy_cfg *cfg;

	cfg = our_dconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_http_scheme()");

	/*
	 * Log the call and exit.
	 */
	trace_add(r->server, NULL, cfg, "vncproxy_http_scheme()");
	return "example";
}
#endif

#ifdef ALL_HOOKS

/*
 * XXX: This routine is called XXX
 *
 * The return value is OK, DECLINED, or HTTP_mumble.  If we return OK, the
 * server will still call any remaining modules with an handler for this
 * phase.
 */
static apr_port_t vncproxy_default_port(const request_rec *r)
{
	vncproxy_cfg *cfg;

	cfg = our_dconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_http_scheme()");

	/*
	 * Log the call and exit.
	 */
	trace_add(r->server, NULL, cfg, "vncproxy_default_port()");
	return 80;
}
#endif

/*
	For each request, checks if the first non empty line is a legacy vnc connect
	method. If it is we fix it to be HTTP/apache compliant. Otherwise we remove
	the filter from the filter list as soon as possible.

	This is currently implemented as a line filter and assume the data we require
	is all in a single brigade, if it gets called in a non AP_MODE_GETLINE mode it
	removes itself from the filter list for the remainder of the request.

	This could be rewritten to be mode independent using a small state
	variable. I'll do this later, but it should work well as written.
 */

static apr_status_t vncproxy_input_filter(ap_filter_t *f, apr_bucket_brigade *bb,
										 ap_input_mode_t mode, apr_read_type_e block,
										 apr_off_t readbytes)
{
	apr_size_t read_count = 0;
	char *data = NULL;
	apr_bucket *e;
	apr_status_t rv;
	apr_size_t len = 0;
	request_rec *r = f->r;
	conn_rec *c = (f != NULL) ? f->c : NULL;
	server_rec *srv = (c != NULL) ? c->base_server : NULL;
	int LimitRequestLine = (srv != NULL) ? srv->limit_req_line : 8190;
	apr_bucket *patmatch_bucket, *nexte;
	apr_size_t bytes_handled = 0, current_alloc = 0, patmatch_offset = 0;
	char *pos;
	vncproxy_cfg *cfg = NULL;

	const char *pattern = "CONNECT vncsession:";
	apr_size_t patlen = 19;
	apr_size_t n;
	apr_bucket *sep_bucket = NULL;
	apr_off_t *sep_bucket_offset = 0;

	apr_bucket *port_bucket = NULL;
	apr_off_t *endofhost_bucket_offset = 0;
	const char *str = NULL;
	char mydata[10000];

	int failed = 0;

	cfg = our_dconfig(r);

	ap_log_cerror(APLOG_MARK, APLOG_DEBUG, 0, c, "vncproxy_input_filter(): start");

	rv = ap_get_brigade(f->next, bb, AP_MODE_GETLINE, APR_BLOCK_READ, 0);

	if (rv != APR_SUCCESS) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [eof]");
		return rv;
	}

	if (!cfg || !cfg->vncProxy) {
		ap_remove_input_filter(f);

		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [APR_SUCCESS, vncProxy Off, REMOVE filter]");
		return APR_SUCCESS;
	}

	if (readbytes == 0) {
		readbytes = LimitRequestLine + 2;
	}

	readbytes -= 5; // save room for :4900

	if (mode != AP_MODE_GETLINE) {
		ap_remove_input_filter(f);

		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [not in getline mode, REMOVE filter]");
		return APR_SUCCESS;
	}

	if (r == NULL) {
		ap_remove_input_filter(f);

		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [not in a request, REMOVE filter]");
		return APR_SUCCESS;
	}

	if (block != APR_BLOCK_READ) {
		ap_remove_input_filter(f);

		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [not in blocking mode, REMOVE filter]");
		return APR_SUCCESS;
	}

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): reading line of input");

	/* insure ap_rgetline allocates memory each time thru the loop
	 * if there are empty lines
	 */

	if (APR_BRIGADE_EMPTY(bb)) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [no input]");
		return APR_EGENERAL;
	}

	for (e = APR_BRIGADE_FIRST(bb); e != APR_BRIGADE_SENTINEL(bb); e = APR_BUCKET_NEXT(e)) {
		str = NULL;
		len = 0;
		n = 0;

		if (APR_BUCKET_IS_EOS(e)) {
			ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [EOS]");
			return APR_EGENERAL;
		}

		rv = apr_bucket_read(e, &str, &len, APR_BLOCK_READ);

		read_count += len;

		if (rv != APR_SUCCESS) {
			ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [read error]");
			return rv;
		}

		while (*pattern && len) {

			if (*pattern != *str) {
				if (read_count > 2) {
					ap_remove_input_filter(f);
					ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [no match, REMOVE filter]");
					return APR_SUCCESS;
				}

				ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [no match]");
				return APR_SUCCESS;
			}
			pattern++;
			str++;
			len--;
			n++;

			if ((n == 18) && (cfg && cfg->vncProxyShutdownSSLOnExec) ) {
					apr_table_setn(r->subprocess_env, "ssl-accurate-shutdown", "1");
					ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, 
						"vncproxy_input_filter(): is vncsession, enable ssl-accurate-shutdown");
			}
		}

		if (*pattern) /* ran out of bucket data, get another bucket */
			continue;

		break;
	}

	if (e == APR_BRIGADE_SENTINEL(bb)) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [unexpected end of input]");
		return APR_EGENERAL;
	}

	patmatch_bucket = e;
	patmatch_offset = n - 1;

	apr_bucket_split(patmatch_bucket, patmatch_offset);
	sep_bucket = apr_bucket_immortal_create(".", 1, bb->bucket_alloc);
	APR_BUCKET_INSERT_AFTER(e, sep_bucket);
	e = APR_BUCKET_NEXT(sep_bucket);
	apr_bucket_split(e, 1);
	nexte = APR_BUCKET_NEXT(e);
	APR_BUCKET_REMOVE(e);
	apr_bucket_destroy(e);
	e = sep_bucket;

	for (e = sep_bucket; e != APR_BRIGADE_SENTINEL(bb); e = APR_BUCKET_NEXT(e)) {
		str = NULL;
		len = 0;
		n = 0;

		if (APR_BUCKET_IS_EOS(e)) {
			ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [EOS while fixing method]");
			return APR_EGENERAL;
		}

		rv = apr_bucket_read(e, &str, &len, APR_BLOCK_READ);

		if (rv != APR_SUCCESS) {
			ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [bucket read error while fixing method]");
			return rv;
		}

		while (len && *str != ' ') {
			len--;
			str++;
			n++;
		}

		if (*str == ' ')
			break;
	}

	if (e == APR_BRIGADE_SENTINEL(bb)) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [unexpected end of data while fixing method]");
		return APR_EGENERAL;
	}

	apr_bucket_split(e, n);
	port_bucket = apr_bucket_immortal_create(":4900", 5, bb->bucket_alloc);
	APR_BUCKET_INSERT_AFTER(e, port_bucket);

	ap_remove_input_filter(f);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [fixed method, REMOVE filter]");
	return APR_SUCCESS;
}

#ifdef ALL_HOOKS

/*
 * XXX: This routine is called XXX
 *
 * The return value is OK, DECLINED, or HTTP_mumble.  If we return OK, the
 * server will still call any remaining modules with an handler for this
 * phase.
 */
static void vncproxy_insert_filter(request_rec *r)
{
	vncproxy_cfg *cfg;

	cfg = our_dconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_insert_filter()")

	/*
	 * Log the call and exit.
	 */
	trace_add(r->server, NULL, cfg, "vncproxy_insert_filter()");
}
#endif

#ifdef ALL_HOOKS

/*
 * XXX: This routine is called XXX
 *
 * The return value is OK, DECLINED, or HTTP_mumble.  If we return OK, the
 * server will still call any remaining modules with an handler for this
 * phase.
 */
static int vncproxy_quick_handler(request_rec *r, int lookup_uri)
{
	vncproxy_cfg *cfg;

	cfg = our_dconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_quick_handler()")

	/*
	 * Log the call and exit.
	 */
	trace_add(r->server, NULL, cfg, "vncproxy_quick_handler()");
	return DECLINED;
}
#endif

#ifdef ALL_HOOKS

static conn_rec *vncproxy_core_create_conn(apr_pool_t *ptrans, server_rec *server,
										   apr_socket_t *csd, long id, void *sbh,
										   apr_bucket_alloc_t *alloc)
{
	vncproxy_cfg *cfg;

	cfg = our_sconfig(server);

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, server, "vncproxy_core_create_conn()");

	return DECLINED;
}
#endif

#ifdef ALL_HOOKS

/*
 * This routine is called just after the server accepts the connection,
 * but before it is handed off to a protocol module to be served.  The point
 * of this hook is to allow modules an opportunity to modify the connection
 * as soon as possible. The core server uses this phase to setup the
 * connection record based on the type of connection that is being used.
 *
 * The return value is OK, DECLINED, or HTTP_mumble.  If we return OK, the
 * server will still call any remaining modules with an handler for this
 * phase.
 */
static int vncproxy_pre_connection(conn_rec *c, void *csd)
{
	vncproxy_cfg *cfg;

	cfg = our_cconfig(c);

	ap_log_cerror(APLOG_MARK, APLOG_DEBUG, 0, c, "vncproxy_pre_connection()");

	/*
	 * Log the call and exit.
	 */
	trace_add(NULL, NULL, cfg, "vncproxy_pre_connection()");

	return OK;
}
#endif

#ifdef ALL_HOOKS

/* This routine is used to actually process the connection that was received.
 * Only protocol modules should implement this hook, as it gives them an
 * opportunity to replace the standard HTTP processing with processing for
 * some other protocol.  Both echo and POP3 modules are available as
 * examples.
 *
 * The return VALUE is OK, DECLINED, or HTTP_mumble.  If we return OK, no
 * further modules are called for this phase.
 */
static int vncproxy_process_connection(conn_rec *c)
{
	vncproxy_cfg *cfg;

	cfg = our_cconfig(c);

	ap_log_cerror(APLOG_MARK, APLOG_DEBUG, 0, c, "vncproxy_process_connection()");

	return DECLINED;
}
#endif

/*
 * This routine is called after the request has been read but before any other
 * phases have been processed.  This allows us to make decisions based upon
 * the input header fields.
 *
 * The return value is OK, DECLINED, or HTTP_mumble.  If we return OK, no
 * further modules are called for this phase.
 */
static int vncproxy_post_read_request(request_rec *r)
{
	vncproxy_cfg *cfg;

	cfg = our_dconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_post_read_request(): start");

	if ((!cfg) || (!cfg->vncProxy)) {
	        ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_post_read_request(): end [declined, vncProxy Off]");
	        return DECLINED;
	}

	if (cfg->vncProxyUseModProxy) {
	        ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_post_read_request(): end [declined, vncProxyUseModProxy On]");
	        return DECLINED;
	}

	if (r->method_number != M_CONNECT) {
	        ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_post_read_request(): end [declined, not CONNECT method]");
	        return DECLINED;
	}

	if (strncmp(r->parsed_uri.hostname, "vncsession", 10) != 0) {
	        ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_post_read_request(): end [declined, not vncsession]");
	        return DECLINED;
	}

	r->proxyreq = PROXYREQ_PROXY;
	r->uri = r->unparsed_uri;
	r->handler = "vncproxy-handler";
	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_post_read_request(): end [declined, but marked for vncproxy-handler]");

	return DECLINED;
}

/*
 * This routine gives our module an opportunity to translate the URI into an
 * actual filename.  If we don't do anything special, the server's default
 * rules (Alias directives and the like) will continue to be followed.
 *
 * The return value is OK, DECLINED, or HTTP_mumble.  If we return OK, no
 * further modules are called for this phase.
 */
static int vncproxy_translate_name(request_rec *r)
{
	size_t dirlen;

	vncproxy_cfg *cfg;

	cfg = our_dconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_translate_name(): start");

	if ((!cfg) || (!cfg->vncProxy)) {
	        ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_translate_name(): end [declined, vncProxy Off]");
	        return DECLINED;
	}

	if (cfg->vncProxyUseModProxy) {
	        ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_translate_name(): end [declined, vncProxyUseModProxy On]");
	        return DECLINED;
	}

	if (r->proxyreq) {
        	/* someone has already set up the proxy, it was possibly ourselves
         	* in vncproxy_post_read_request()
         	*/
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_translate_name(): end [ok, vncproxy request detected]");
        	return OK;
    	}

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_translate_name(): end [declined, vncproxy request not detected]");

	return DECLINED;


	trace_add(r->server, r, cfg, "vncproxy_translate_name()");
	trace_add(r->server, r, cfg, r->uri);

	dirlen = strlen(cfg->dir);

	/* No longer include option to serve applets, but keeping code to remind me how to do it in case we come back to this idea */
	/*
	if (strncmp(r->uri, cfg->dir, dirlen) == 0) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "URI matches directory config");
		if (strcmp(r->uri + dirlen, "/SignedVncViewer.jar") == 0) {
			ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "Matched Signed Applet Name");
			r->filename = strdup(cfg->vncProxySignedApplet);
			trace_add(r->server, r, cfg, r->filename);
			return OK;
		}

		if (strcmp(r->uri + dirlen, "/VncViewer.jar") == 0) {
			ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "Matched Unsigned Applet Name");
			r->filename = strdup(cfg->vncProxyUnsignedApplet);
			trace_add(r->server, r, cfg, r->filename);
			return OK;
		}
	}
	*/

	return DECLINED;
}

#ifdef ALL_HOOKS
/*
 * This routine maps r->filename to a physical file on disk.  Useful for
 * overriding default core behavior, including skipping mapping for
 * requests that are not file based.
 *
 * The return value is OK, DECLINED, or HTTP_mumble.  If we return OK, no
 * further modules are called for this phase.
 */
static int vncproxy_map_to_storage_handler(request_rec *r)
{
	vncproxy_cfg *cfg;

	cfg = our_dconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_map_to_storage_handler(): start");

        if ((!cfg) || (!cfg->vncProxy) || (cfg->vncProxyUseModProxy)) {
                ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_map_to_storage_handler(): end [declined]");
                return DECLINED;
        }

	if (!r->proxyreq != 0) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_map_to_storage_handler(): end [declined, not vncproxy request]");
		return DECLINED;
	}

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_map_to_storage_handler(): end [ok]");
	return OK;
}
#endif

#ifdef ALL_HOOKS

/*
 * this routine gives our module another chance to examine the request
 * headers and to take special action. This is the first phase whose
 * hooks' configuration directives can appear inside the <Directory>
 * and similar sections, because at this stage the URI has been mapped
 * to the filename. For example this phase can be used to block evil
 * clients, while little resources were wasted on these.
 *
 * The return value is OK, DECLINED, or HTTP_mumble.  If we return OK,
 * the server will still call any remaining modules with an handler
 * for this phase.
 */
static int vncproxy_header_parser(request_rec *r)
{

	vncproxy_cfg *cfg;

	cfg = our_dconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_header_parser()");

	/*
	 * We don't actually *do* anything here, except note the fact that we were
	 * called.
	 */
	trace_add(r->server, r, cfg, "vncproxy_header_parser_handler()");
	return DECLINED;
}
#endif

#ifdef ALL_HOOKS

/*
 * This routine is called to check the authentication information sent with
 * the request (such as looking up the user in a database and verifying that
 * the [encrypted] password sent matches the one in the database).
 *
 * The return value is OK, DECLINED, or some HTTP_mumble error (typically
 * HTTP_UNAUTHORIZED).  If we return OK, no other modules are given a chance
 * at the request during this phase.
 */
static int vncproxy_check_user_id(request_rec *r)
{

	vncproxy_cfg *cfg;

	cfg = our_dconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_check_user_id()");

	/*
	 * Don't do anything except log the call.
	 */
	trace_add(r->server, r, cfg, "vncproxy_check_user_id()");
	return DECLINED;
}
#endif

#ifdef ALL_HOOKS

/*
 * This routine is called to check to see if the resource being requested
 * requires authorisation.
 *
 * The return value is OK, DECLINED, or HTTP_mumble.  If we return OK, no
 * other modules are called during this phase.
 *
 * If *all* modules return DECLINED, the request is aborted with a server
 * error.
 */
static int vncproxy_auth_checker(request_rec *r)
{
	vncproxy_cfg *cfg;

	cfg = our_dconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_auth_checker()");

	/*
	 * Log the call and return OK, or access will be denied (even though we
	 * didn't actually do anything).
	 */
	trace_add(r->server, r, cfg, "vncproxy_auth_checker()");
	return DECLINED;
}
#endif

#ifdef ALL_HOOKS

/*
 * This routine is called to check for any module-specific restrictions placed
 * upon the requested resource.  (See the mod_access module for an example.)
 *
 * The return value is OK, DECLINED, or HTTP_mumble.  All modules with an
 * handler for this phase are called regardless of whether their predecessors
 * return OK or DECLINED.  The first one to return any other status, however,
 * will abort the sequence (and the request) as usual.
 */
static int vncproxy_access_checker(request_rec *r)
{
	vncproxy_cfg *cfg;

	cfg = our_dconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_access_checker()");

	trace_add(r->server, r, cfg, "vncproxy_access_checker()");
	return DECLINED;
}
#endif

#ifdef ALL_HOOKS

/*
 * This routine is called to determine and/or set the various document type
 * information bits, like Content-type (via r->content_type), language, et
 * cetera.
 *
 * The return value is OK, DECLINED, or HTTP_mumble.  If we return OK, no
 * further modules are given a chance at the request for this phase.
 */
static int vncproxy_type_checker(request_rec *r)
{

	vncproxy_cfg *cfg;

	cfg = our_dconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_type_checker()");

	/*
	 * Log the call, but don't do anything else - and report truthfully that
	 * we didn't do anything.
	 */
	trace_add(r->server, r, cfg, "vncproxy_type_checker()");
	return DECLINED;
}
#endif

#ifdef ALL_HOOKS
/*
 * This routine is called to perform any module-specific fixing of header
 * fields, et cetera.  It is invoked just before any content-handler.
 *
 * The return value is OK, DECLINED, or HTTP_mumble.  If we return OK, the
 * server will still call any remaining modules with an handler for this
 * phase.
 */
static int vncproxy_fixups(request_rec *r)
{
	char *url, *p;
	vncproxy_cfg *cfg;

	cfg = our_dconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_fixups(): start");

        if ((!cfg) || (!cfg->vncProxy) || (cfg->vncProxyUseModProxy)) {
                ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_fixups(): end [declined]");
                return DECLINED;
        }

	if (!r->proxyreq) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_fixups(): end [declined, not a vncproxy request]");
        	return DECLINED;
	}

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_fixups(): end [ok]");

	return OK;
}
#endif

#ifdef ALL_HOOKS

/*
 * This routine is called to perform any module-specific logging activities
 * over and above the normal server things.
 *
 * The return value is OK, DECLINED, or HTTP_mumble.  If we return OK, any
 * remaining modules with an handler for this phase will still be called.
 */
static int vncproxy_log_transaction(request_rec *r)
{
	vncproxy_cfg *cfg;

	cfg = our_dconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_log_transaction()");

	trace_add(r->server, r, cfg, "vncproxy_log_transaction()");
	return DECLINED;
}
#endif


#ifdef USE_MODPROXY

/*
 * This handles Netscape CONNECT method secure proxy requests.
 * A connection is opened to the specified host and data is
 * passed through between the WWW site and the browser.
 *
 * This code is based on the INTERNET-DRAFT document
 * "Tunneling SSL Through a WWW Proxy" currently at
 * http://www.mcom.com/newsref/std/tunneling_ssl.html.
 *
 * If proxyhost and proxyport are set, we send a CONNECT to
 * the specified proxy..
 *
 * FIXME: this doesn't log the number of bytes sent, but
 *        that may be okay, since the data is supposed to
 *        be transparent. In fact, this doesn't log at all
 *        yet. 8^)
 * FIXME: doesn't check any headers initally sent from the
 *        client.
 * FIXME: should allow authentication, but hopefully the
 *        generic proxy authentication is good enough.
 * FIXME: no check for r->assbackwards, whatever that is.
 */



/* canonicalise CONNECT URLs. */
static int vncproxy_canon_handler(request_rec *r, char *url)
{
	vncproxy_cfg *cfg = NULL;

	cfg = our_dconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_canon_handler(): start");

	if (!cfg || !cfg->vncProxy) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_canon_handler(): end [vncProxy Off]");
		return DECLINED;
	}

	if (!cfg->vncProxyUseModProxy) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_canon_handler(): end [vncProxyUseModProxy Off]");
		return DECLINED;
	}

	if (r->method_number != M_CONNECT) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_canon_handler(): end [declined, not CONNECT method]");
		return DECLINED;
	}

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_canon_handler(): canonicalising URL %s", url);
	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_canon_handler(): end [success]");

	return OK;
}

/* read available data (in blocks of CONN_BLKSZ) from c_i and copy to c_o */
static int vncproxy_transfer(request_rec *r, conn_rec *c_i, conn_rec *c_o,
							 apr_bucket_brigade *bb, char *name)
{
	int rv;
	apr_off_t len;
	vncproxy_cfg *cfg = NULL;

	cfg = our_rconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_transfer(): start");

	do {
		apr_brigade_cleanup(bb);
		rv = ap_get_brigade(c_i->input_filters, bb, AP_MODE_READBYTES,
							APR_NONBLOCK_READ, CONN_BLKSZ);
		if (rv == APR_SUCCESS) {
			if (APR_BRIGADE_EMPTY(bb))
				break;

			len = -1;
			apr_brigade_length(bb, 0, &len);
			ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "CONNECT: read %" APR_OFF_T_FMT " bytes from %s", len, name);

			rv = ap_pass_brigade(c_o->output_filters, bb);
			if (rv == APR_SUCCESS) {
				ap_fflush(c_o->output_filters, bb);
			}
			else {
				ap_log_rerror(APLOG_MARK, APLOG_ERR, rv, r, "CONNECT: error on %s - ap_pass_brigade", name);
			}
		}
		else if (!APR_STATUS_IS_EAGAIN(rv)) {
			ap_log_rerror(APLOG_MARK, APLOG_DEBUG, rv, r, "CONNECT: error on %s - ap_get_brigade", name);
		}
	}
	while (rv == APR_SUCCESS);

	if (APR_STATUS_IS_EAGAIN(rv)) {
		rv = APR_SUCCESS;
	}
	return rv;
}

/* CONNECT handler */
static int vncproxy_scheme_handler(request_rec *r, proxy_worker *worker,
								   proxy_server_conf *conf,
								   char *url, const char *proxyname,
								   apr_port_t proxyport)
{
	apr_pool_t *p = r->pool;
	apr_socket_t *sock;
	conn_rec *c = r->connection;
	conn_rec *backconn;

	apr_bucket_brigade *bb = apr_brigade_create(p, c->bucket_alloc);
	apr_status_t err, rv;
	apr_size_t i, o, nbytes;
	char buffer[HUGE_STRING_LEN];
	apr_socket_t *client_socket = ap_get_module_config(c->conn_config, &core_module);
	int failed, rc;
	apr_pollset_t *pollset;
	apr_pollfd_t pollfd;
	const apr_pollfd_t *signalled;
	apr_int32_t pollcnt, pi;
	apr_int16_t pollevent;
	apr_sockaddr_t *uri_addr, *connect_addr;

	apr_uri_t uri;
	const char *connectname;
	int connectport = 0;
	char *token;
	vncproxy_cfg *cfg = NULL;
	int status = 0;

	cfg = our_dconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_scheme_handler(): start");

	if (!cfg || !cfg->vncProxy) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_scheme_handler(): end [vncProxyProxy Off]");
		return DECLINED;
	}

	if (!cfg->vncProxyUseModProxy) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_scheme_handler(): end [vncProxyUseModProxy Off]");
		return DECLINED;
	}

	/* is this for us? */
	if (r->method_number != M_CONNECT) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_scheme_handler(): end [declining URL %s]", url);
		return DECLINED;
	}

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_scheme_handler(): URL [%s]", url);

	/*
	 * Step One: Determine Who To Connect To
	 *
	 * Break up the URL to determine the host to connect to
	 */

	/* we break the URL into host, port, uri */
	if (APR_SUCCESS != apr_uri_parse_hostinfo(p, url, &uri)) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_scheme_handler(): end [URI cannot be parsed %s]", url);
		return ap_proxyerror(r, HTTP_BAD_REQUEST,
							apr_pstrcat(p, "URI cannot be parsed: ", url,
										NULL));
	}

	/* BEGIN_VNCPROXY */
	/* is this a vncsession connect request? */
	if (strncmp(uri.hostname, "vncsession.", 11) != 0) {
		ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, r->server, "vncproxy_scheme_handler(): end [declined, not vncsession]");
		return DECLINED;
	}

	token = uri.hostname + 11;
	/* END_VNCPROXY */

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, r->server, "vncproxy_scheme_handler(): connecting to %s:%d", uri.hostname, uri.port);

	/* do a DNS lookup for the destination host */
	/* BEGIN_VNCPROXY */
	err = apr_sockaddr_info_get(&uri_addr, NULL /* uri.hostname */, APR_UNSPEC, uri.port, 0, p);
	/* END_VNCPROXY */
	if (APR_SUCCESS != err) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_scheme_handler(): end [DNS lookup failure for: %s]", uri.hostname);
		return ap_proxyerror(r, HTTP_BAD_GATEWAY,
							apr_pstrcat(p, "DNS lookup failure for: ", uri.hostname, NULL));
	}
	/* BEGIN_VNCPROXY */
	uri_addr->hostname = uri.hostname;
	/* END_VNCPROXY */

	/* are we connecting directly, or via a proxy? */
	if (proxyname) {
		connectname = proxyname;
		connectport = proxyport;
		err = apr_sockaddr_info_get(&connect_addr, proxyname, APR_UNSPEC, proxyport, 0, p);
	
		ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, r->server, "vncproxy_scheme_handler(): connecting to remote proxy %s on port %d",
			connectname, connectport);

	}
	else {
		connectname = uri.hostname;
		connectport = uri.port;
		connect_addr = uri_addr;
	}
	
	/* check if ProxyBlock directive on this host */
	if (OK != ap_proxy_checkproxyblock(r, conf, uri_addr)) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_scheme_handler(): end [connect to remote machine blocked]");
		return ap_proxyerror(r, HTTP_FORBIDDEN, "Connect to remote machine blocked");
	}

	/* Check if it is an allowed port */
	if (!allowed_port(r, conf, uri.port)) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_scheme_handler(): end [connect to remote machine blocked]");
		return ap_proxyerror(r, HTTP_FORBIDDEN, "Connect to remote machine blocked");
	}

	/*
	 * Step Two: Make the Connection
	 *
	 * We have determined who to connect to. Now make the connection.
	 */

	/* get all the possible IP addresses for the destname and loop through them
	 * until we get a successful connection
	 */
	if (APR_SUCCESS != err) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_scheme_handler(): end [DNS lookup failure for : %s]", connectname);
		return ap_proxyerror(r, HTTP_BAD_GATEWAY,
							apr_pstrcat(p, "DNS lookup failure for : ",
										connectname, NULL));
	}

	/* START_VNCPROXY */
	if (!proxyport) {
		pid_t pid;
		const char *vncProxyHelper;
		char **argv_out;
		apr_bucket *eos, *e;

		nbytes = apr_snprintf(buffer, sizeof (buffer), "HTTP/1.0 200" CRLF);
		ap_xlate_proto_to_ascii(buffer, nbytes);
		ap_fwrite(c->output_filters, bb, buffer, nbytes);
		ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, r->server, "vncproxy_scheme_handler(): HTTP/1.0 200 Connection Established");

		nbytes = apr_snprintf(buffer, sizeof (buffer), "Proxy-agent: %s" CRLF CRLF, vncproxy_banner());
		ap_xlate_proto_to_ascii(buffer, nbytes);
		ap_fwrite(c->output_filters, bb, buffer, nbytes);

		ap_fflush(c->output_filters, bb);

		if (cfg && cfg->vncProxyShutdownSSLOnExec) {
			/* Send end of stream to trigger SSL module to send close-notify alert */
			e = ap_bucket_eoc_create(c->bucket_alloc);
			eos = APR_BRIGADE_LAST(bb);
			while ((APR_BRIGADE_SENTINEL(bb) != eos)
				&& !APR_BUCKET_IS_EOS(eos)) {
				eos = APR_BUCKET_PREV(eos);
			}
			if (eos == APR_BRIGADE_SENTINEL(bb)) {
				APR_BRIGADE_INSERT_TAIL(bb, e);
			}
			else {
				APR_BUCKET_INSERT_BEFORE(eos, e);
			}
			ap_pass_brigade(c->output_filters, bb);
		}

		vncProxyHelper = buildProxyHelperString(r->pool, cfg->vncProxyHelper, token, 0, "", c->remote_ip);
		apr_tokenize_to_argv(vncProxyHelper, &argv_out, r->pool);

		if (chdir("/") != 0) {
			ap_log_error(APLOG_MARK, APLOG_ERR, 0, r->server, "vncproxy_scheme_handler(): end [unable to change directory to /]");
			return HTTP_INTERNAL_SERVER_ERROR;
		}

		umask(0);

		pid = fork();

		if (pid == -1) {
			ap_log_error(APLOG_MARK, APLOG_ERR, 0, r->server, "vncproxy_scheme_handler(): end [backgrounding fork() failed]");
			return HTTP_INTERNAL_SERVER_ERROR;
		}

		if (pid == 0) {

			ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, r->server, "vncproxy_scheme_handler(): Background process [%d] created", getpid());

			rv = setsid();

			if (rv == -1) {
				ap_log_error(APLOG_MARK, APLOG_ERR, 0, r->server, "vncproxy_scheme_handler(): Can't create new process session");
				_exit(1);
			}

			pid = fork();

			if (pid == -1) {
				ap_log_error(APLOG_MARK, APLOG_WARNING, 0, r->server, "vncproxy_scheme_handler(): Failed to fork() prior to execution of vncProxyHelper [%s](%d)",
						vncProxyHelper, errno);
				_exit(2);
			}

			if (pid == 0) {
				ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, r->server, "vncproxy_scheme_handler(): Disassociated process [%d] created.", getpid());

				if (cfg && cfg->vncProxyDBDriver)
					setenv("DBD_DRIVER", cfg->vncProxyDBDriver, 1);
				else
					setenv("DBD_DRIVER", "mysql", 1);

				if (cfg && cfg->vncProxyDBDParams)
					setenv("DBD_PARAMS", cfg->vncProxyDBDParams, 1);
				else
					setenv("DBD_PARAMS", "", 1);

				dup2(client_socket->socketdes, 0);
				dup2(client_socket->socketdes, 1);

				ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, r->server, "vncproxy_scheme_handler(): Executing [%s]", vncProxyHelper);

				execvp(argv_out[0], argv_out);

				ap_log_error(APLOG_MARK, APLOG_WARNING, 0, r->server, "vncproxy_scheme_handler(): Failed to execute vncProxyHelper [%s](%d)",
						vncProxyHelper, errno);
				_exit(3);
			}

			ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, r->server, "vncproxy_scheme_handler(): Exiting background process [%d]", getpid());
			_exit(0); // let session leader parent die
		}

		waitpid(pid, &status, 0); // wait for intermediate background process to exit

		ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, r->server, "vncproxy_scheme_handler(): Apache parent worker process [%d] continuing", getpid());

		/* abort connection */
		r->input_filters = NULL; // Parent will shutdown SSL session if we don't clear filters and signal abort
		r->output_filters = NULL;
		r->connection->output_filters = NULL;
		r->proto_output_filters = NULL;
		r->connection->input_filters = NULL;
		r->input_filters = NULL;
		r->connection->aborted = 1;
		ap_log_error(APLOG_MARK, APLOG_ERR, 0, r->server, "vncproxy_scheme_handler(): end [success]");
		return OK;
	}

	/* At the moment no other operational modes work */
	ap_log_error(APLOG_MARK, APLOG_ERR, 0, r->server, "vncproxy_scheme_handler(): end [mode failure]");
	return HTTP_INTERNAL_SERVER_ERROR;


	/* END_VNCPROXY */

	/*
	 * At this point we have a list of one or more IP addresses of
	 * the machine to connect to. If configured, reorder this
	 * list so that the "best candidate" is first try. "best
	 * candidate" could mean the least loaded server, the fastest
	 * responding server, whatever.
	 *
	 * For now we do nothing, ie we get DNS round robin.
	 * XXX FIXME
	 */
	failed = ap_proxy_connect_to_backend(&sock, "CONNECT", connect_addr,
										connectname, conf, r->server,
										r->pool);

	/* handle a permanent error from the above loop */
	if (failed) {
		if (proxyname) {
			return DECLINED;
		}
		else {
			return HTTP_SERVICE_UNAVAILABLE;
		}
	}

	/* setup polling for connection */
	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "CONNECT: setting up poll()");

	if ((rv = apr_pollset_create(&pollset, 2, r->pool, 0)) != APR_SUCCESS) {
		apr_socket_close(sock);
		ap_log_rerror(APLOG_MARK, APLOG_ERR, rv, r, "CONNECT: error apr_pollset_create()");
		return HTTP_INTERNAL_SERVER_ERROR;
	}

	/* Add client side to the poll */
	pollfd.p = r->pool;
	pollfd.desc_type = APR_POLL_SOCKET;
	pollfd.reqevents = APR_POLLIN;
	pollfd.desc.s = client_socket;
	pollfd.client_data = NULL;
	apr_pollset_add(pollset, &pollfd);

	/* Add the server side to the poll */
	pollfd.desc.s = sock;
	apr_pollset_add(pollset, &pollfd);

	/*
	 * Step Three: Send the Request
	 *
	 * Send the HTTP/1.1 CONNECT request to the remote server
	 */

	backconn = ap_run_create_connection(c->pool, r->server, sock, c->id, c->sbh, c->bucket_alloc);

	if (!backconn) {
		/* peer reset */
		ap_log_rerror(APLOG_MARK, APLOG_WARNING, 0, r, "an error occurred creating a new connection to %pI (%s)", connect_addr, connectname);
		apr_socket_close(sock);
		return HTTP_INTERNAL_SERVER_ERROR;
	}
	ap_proxy_ssl_disable(backconn);
	rc = ap_run_pre_connection(backconn, sock);
	if (rc != OK && rc != DONE) {
		backconn->aborted = 1;
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "CONNECT: pre_connection setup failed (%d)", rc);
		return HTTP_INTERNAL_SERVER_ERROR;
	}

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "CONNECT: connection complete to %pI (%s)", connect_addr, connectname);

	/* If we are connecting through a remote proxy, we need to pass
	 * the CONNECT request on to it.
	 */
	if (proxyport) {
		/* FIXME: Error checking ignored.
		 */
		ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, r->server, "CONNECT: sending the CONNECT request to the remote proxy");

		// @MOD: VNCPROXY remangle "vncsession." uri for legacy vncproxy.py script
		r->uri[10] = ':';
		r->uri[43] = 0;

		ap_fprintf(backconn->output_filters, bb,
				"CONNECT %s HTTP/1.0" CRLF, r->uri);
		ap_fprintf(backconn->output_filters, bb,
				"Proxy-agent: %s" CRLF CRLF, ap_get_server_version());
		ap_fflush(backconn->output_filters, bb);
	}
	else {
		ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, r->server, "CONNECT: Returning 200 OK Status");
		nbytes = apr_snprintf(buffer, sizeof (buffer), "HTTP/1.0 200 Connection Established" CRLF);
		ap_xlate_proto_to_ascii(buffer, nbytes);
		ap_fwrite(c->output_filters, bb, buffer, nbytes);
		nbytes = apr_snprintf(buffer, sizeof (buffer), "Proxy-agent: %s" CRLF CRLF, vncproxy_banner());
		ap_xlate_proto_to_ascii(buffer, nbytes);
		ap_fwrite(c->output_filters, bb, buffer, nbytes);
		ap_fflush(c->output_filters, bb);

		/* This is safer code, but it doesn't work yet.  I'm leaving it
		 * here so that I can fix it later.
		 */
		/*
		r->status = HTTP_OK;
		r->header_only = 1;
		apr_table_set(r->headers_out, "Proxy-agent: %s", vncproxy_banner());
		ap_rflush(r);
		 */
	}

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, r->server, "CONNECT: setting up poll()");

	/*
	 * Step Four: Handle Data Transfer
	 *
	 * Handle two way transfer of data over the socket (this is a tunnel).
	 */
	/* we are now acting as a tunnel - the input/output filter stacks should
	 * not contain any non-connection filters.
	 */
	r->output_filters = c->output_filters;
	r->proto_output_filters = c->output_filters;
	r->input_filters = c->input_filters;
	r->proto_input_filters = c->input_filters;

	/*    r->sent_bodyct = 1;*/

	while (1) { /* Infinite loop until error (one side closes the connection) */
		if ((rv = apr_pollset_poll(pollset, -1, &pollcnt, &signalled)) != APR_SUCCESS) {
			apr_socket_close(sock);
			ap_log_rerror(APLOG_MARK, APLOG_ERR, rv, r, "CONNECT: error apr_poll()");
			return HTTP_INTERNAL_SERVER_ERROR;
		}

		ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, r->server, "CONNECT: woke from poll(), i=%d", pollcnt);

		for (pi = 0; pi < pollcnt; pi++) {
			const apr_pollfd_t *cur = &signalled[pi];

			if (cur->desc.s == sock) {
				pollevent = cur->rtnevents;
				if (pollevent & APR_POLLIN) {
					ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "CONNECT: sock was readable");
					rv = vncproxy_transfer(r, backconn, c, bb, "sock");
				}
				else if ((pollevent & APR_POLLERR) || (pollevent & APR_POLLHUP)) {
					rv = APR_EPIPE;
					ap_log_rerror(APLOG_MARK, APLOG_WARNING, 0, r, "CONNECT: err/hup on backconn");
				}
			}
			else if (cur->desc.s == client_socket) {
				pollevent = cur->rtnevents;
				if (pollevent & APR_POLLIN) {
					ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "CONNECT: client was readable");
					rv = vncproxy_transfer(r, c, backconn, bb, "client");
				}
			}
			else {
				rv = APR_EBADF;
				ap_log_rerror(APLOG_MARK, APLOG_WARNING, 0, r, "CONNECT: unknown socket in pollset");
			}

		}
		if (rv != APR_SUCCESS) {
			break;
		}
	}

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, r->server, "CONNECT: finished with poll() - cleaning up");

	/*
	 * Step Five: Clean Up
	 *
	 * Close the socket and clean up
	 */

	ap_lingering_close(backconn);

	c->aborted = 1;

	return OK;
}

#endif /* USE_MODPROXY */

/*--------------------------------------------------------------------------*/
/*                                                                          */
/* Which functions are responsible for which hooks in the server.           */
/*                                                                          */
/*--------------------------------------------------------------------------*/

/*
 * Each function our module provides to handle a particular hook is
 * specified here.  The functions are registered using
 * ap_hook_foo(name, predecessors, successors, position)
 * where foo is the name of the hook.
 *
 * The args are as follows:
 * name         -> the name of the function to call.
 * predecessors -> a list of modules whose calls to this hook must be
 *                 invoked before this module.
 * successors   -> a list of modules whose calls to this hook must be
 *                 invoked after this module.
 * position     -> The relative position of this module.  One of
 *                 APR_HOOK_FIRST, APR_HOOK_MIDDLE, or APR_HOOK_LAST.
 *                 Most modules will use APR_HOOK_MIDDLE.  If multiple
 *                 modules use the same relative position, Apache will
 *                 determine which to call first.
 *                 If your module relies on another module to run first,
 *                 or another module running after yours, use the
 *                 predecessors and/or successors.
 *
 * The number in brackets indicates the order in which the routine is called
 * during request processing.  Note that not all routines are necessarily
 * called (such as if a resource doesn't have access restrictions).
 * The actual delivery of content to the browser [9] is not handled by
 * a hook; see the handler declarations below.
 */
static void vncproxy_register_hooks(apr_pool_t *p)
{
	static const char * const aszSucc[] = {"mod_proxy.c", "mod_proxy_connect.c", "mod_rewrite.c", NULL};
	static const char * const proxyHandlerSucc[] = {"mod_proxy_connect.c", NULL};
	static const char * const aszPred[] = {"mod_rewrite.c", NULL };

	static const char * const modproxy[] = { "mod_proxy.c" , NULL };
	static const char * const modrewrite[] = { "mod_rewrite.c" , NULL };
	static const char * const modproxyrewrite[] = { "mod_proxy.c", "mod_rewrite.c" , NULL };
	vncproxy_cfg *cfg = NULL;

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, 0, "vncproxy_register_hooks()");

	/* http_request.h declared hooks */
	ap_hook_create_request(vncproxy_create_request, NULL, aszSucc, APR_HOOK_FIRST);
	ap_hook_translate_name(vncproxy_translate_name, NULL, modproxyrewrite, APR_HOOK_FIRST);
#ifdef ALL_HOOKS
	ap_hook_check_user_id(vncproxy_check_user_id, NULL, NULL, APR_HOOK_MIDDLE);
	ap_hook_type_checker(vncproxy_type_checker, NULL, NULL, APR_HOOK_MIDDLE);
	ap_hook_fixups(vncproxy_fixups, NULL, aszSucc, APR_HOOK_FIRST);
	ap_hook_access_checker(vncproxy_access_checker, NULL, NULL, APR_HOOK_MIDDLE);
	ap_hook_auth_checker(vncproxy_auth_checker, NULL, NULL, APR_HOOK_MIDDLE);
	ap_hook_insert_filter(vncproxy_insert_filter, NULL, NULL, APR_HOOK_MIDDLE);
	ap_hook_map_to_storage(vncproxy_map_to_storage_handler, NULL, aszSucc, APR_HOOK_FIRST);
#endif

	/* http_connection.h declared hooks */
#ifdef ALL_HOOKS
	ap_hook_create_connection(vncproxy_create_connection, NULL, NULL, APR_HOOK_MIDDLE);
	ap_hook_pre_connection(vncproxy_pre_connection, NULL, NULL, APR_HOOK_MIDDLE);
	ap_hook_process_connection(vncproxy_process_connection, NULL, NULL, APR_HOOK_MIDDLE);
#endif

	/* http_protocol.h declared hooks */
	ap_hook_post_read_request(vncproxy_post_read_request, modproxy, NULL, APR_HOOK_FIRST);
#ifdef ALL_HOOKS
	*ap_hook_insert_error_filter(vncproxy_insert_error_filter, NULL, NULL, APR_HOOK_MIDDLE);
	ap_hook_log_transaction(vncproxy_log_transaction, NULL, NULL, APR_HOOK_MIDDLE);
	ap_hook_http_scheme(vncproxy_http_scheme, NULL, NULL, APR_HOOK_MIDDLE);
	ap_hook_default_port(vncproxy_default_port, NULL, NULL, APR_HOOK_MIDDLE);
#endif

	/* http_config.h declared hooks */
#ifdef ALL_HOOKS
	ap_hook_header_parser(vncproxy_header_parser, NULL, NULL, APR_HOOK_MIDDLE);
	ap_hook_pre_config(vncproxy_pre_config, NULL, NULL, APR_HOOK_MIDDLE);
	*ap_hook_test_config(vncproxy_test_config, NULL, NULL, APR_HOOK_MIDDLE);
	ap_hook_post_config(vncproxy_post_config, NULL, NULL, APR_HOOK_MIDDLE);
	ap_hook_open_logs(vncproxy_open_logs, NULL, NULL, APR_HOOK_MIDDLE);
	ap_hook_child_init(vncproxy_child_init, NULL, NULL, APR_HOOK_MIDDLE);
#endif
	ap_hook_handler(vncproxy_handler, NULL, aszSucc, APR_HOOK_FIRST);
#ifdef ALL_HOOKS
	ap_hook_quick_handler(vncproxy_quick_handler, NULL, NULL, APR_HOOK_MIDDLE);
	*ap_hook_optional_fn_retrieve(vncproxy_optional_fn_retrieve, NULL, NULL, APR_HOOK_MIDDLE);
#endif

	/* ap_mpm.h declared hooks */
#ifdef ALL_HOOKS
	*ap_hook_fatal_exception(vncproxy_fatal_exception, NULL, NULL, APR_HOOK_MIDDLE);
#endif

	/* http_core.h declared hooks */
#ifdef ALL_HOOKS
	*ap_hook_get_mgmt_items(vncproxy_get_mgmt_items, NULL, NULL, APR_HOOK_MIDDLE);
#endif

	/* scoreboard.h declared hooks */
#ifdef ALL_HOOKS
	*ap_hook_pre_mpm(vncproxy_pre_mpm, NULL, NULL, APR_HOOK_MIDDLE);
#endif

	/* mpm_common.h declared hooks */
#ifdef ALL_HOOKS
	*ap_hook_monitor(vncproxy_monitor, NULL, NULL, APR_HOOK_MIDDLE);
#endif

	/* unixd.h declared hooks */
#ifdef ALL_HOOKS
	*ap_hook_get_suexec_identity(vncproxy_get_suexec_identity, NULL, NULL, APR_HOOK_MIDDLE);
#endif

#ifdef USE_MODPROXY
	/* mod_proxy.h declared hooks */
	proxy_hook_scheme_handler(vncproxy_scheme_handler, NULL, proxyHandlerSucc, APR_HOOK_FIRST);
	proxy_hook_canon_handler(vncproxy_canon_handler, NULL, proxyHandlerSucc, APR_HOOK_FIRST);
#endif /* USE_MODPROXY */

#ifdef ALL_HOOKS
	*proxy_hook_create_req(vncproxy_create_req, NULL, NULL, APR_HOOK_MIDDLE);
	*proxy_hook_fixups(vncproxy_fixups, NULL, NULL, APR_HOOK_MIDDLE);
	*proxy_hook_pre_request(vncproxy_pre_request, NULL, NULL, APR_HOOK_MIDDLE);
	*proxy_hook_post_request(vncproxy__post_request, NULL, NULL, APR_HOOK_MIDDLE);
	*proxy_hook_request_status(vncproxy_request_status, NULL, NULL, APR_HOOK_MIDDLE);
#endif

	/* mod_dav.h declared hooks */
#ifdef ALL_HOOKS
	*dav_hook_gather_propsets()
		* dav_hook_find_liveprop()
		* dav_hook_insert_all_liveprops()
		* dav_hook_status_hook()
#endif

		/* ap_config.h declared hooks */
#ifdef ALL_HOOKS
		* ap_hook_ret()
#endif

		/* util_filter.h */
		ap_register_input_filter("VNCPROXY_INPUT_FILTER", vncproxy_input_filter, NULL, AP_FTYPE_PROTOCOL);
#ifdef ALL_HOOKS
	ap_register_output_filter("VNCPROXY_OUTPUT_FILTER", vncproxy_output_filter, NULL, AP_FTYPE_RESOURCE);
#endif

	/* Some other things that can hook into apache or its modules */
	/* ap_register_provider */
	/* ap_register_include_provider */
	/* ap_register_rewrite_mapfunc */
	/* ap_register_log_hander */
	/* ap_register_request_note */
	/* ap_register_extra_mpm_process */
}

/*--------------------------------------------------------------------------*/
/*                                                                          */
/* Finally, the list of callback routines and data structures that provide  */
/* the static hooks into our module from the other parts of the server.     */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*
 * Module definition for configuration.  If a particular callback is not
 * needed, replace its routine name below with the word NULL.
 */
module AP_MODULE_DECLARE_DATA vncproxy_module = {
	STANDARD20_MODULE_STUFF,
	vncproxy_create_dir_config, /* per-directory config creator */
	vncproxy_merge_dir_config, /* dir config merger */
	vncproxy_create_server_config, /* server config creator */
	vncproxy_merge_server_config, /* server config merger */
	vncproxy_cmds, /* command table */
	vncproxy_register_hooks, /* set up other request processing hooks */
};
