#!/bin/python
#
# @package      hubzero-invokeapp
# @file         invoke_app_test.py
# @author       Derrick Kearney <dsk@purdue.edu>
# @copyright    Copyright (c) 2013-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2013-2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.

import re
import os
import sys
import unittest
import subprocess


INVOKE_APP_PATH = "/usr/bin/invoke_app"
PARAMETERS_PATH = "parameters.hz"
DATA = {
    'datafile1' : 'this is datafile1',
    'datafile2' : 'this is datafile2'
}


def pretest_setup():

    # setup our datafiles
    for fname,text in DATA.items():
        with open(fname,'w') as f:
            f.write(text)


def posttest_cleanup():

    for fname in DATA.keys():
        fpath = os.path.join(os.getcwd(),fname)
        os.remove(fpath)


def run_invoke_app(command_list,parameters_text=None):

    environment = os.environ

    if parameters_text != None:
        with open(PARAMETERS_PATH,'w') as f:
            f.write(parameters_text)
        environment['TOOL_PARAMETERS'] = PARAMETERS_PATH

    # run the invoke_app command
    p = subprocess.Popen(
            command_list,
            shell=False,
            env=environment,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE)
    (stdout,stderr) = p.communicate()

    return (stdout,stderr)


def get_tool_output(stdout,stderr):

    toolout = stdout.rstrip()

    toolerr =""
    matches = re.search("exec'ing[^\n]+\n(.*)",stderr,re.DOTALL)
    if matches:
        toolerr = matches.group(1)

    return (toolout,toolerr)


class invoke_app_tests(unittest.TestCase):

    def setUp(self):
        pass

    def test_1_command_no_templates(self):
        """launching invoke_app with one -C command (not template) should run the command
           ex: invoke_app -C "echo hi"
           should produce: hi
        """

        # build our invoke_app command
        command_list = [INVOKE_APP_PATH,
                        '-C','echo hi']

        # run invoke_app
        stdout,stderr = run_invoke_app(command_list)
        toolout,toolerr = get_tool_output(stdout,stderr)

        # parse the output
        expected_out = "hi"
        self.assertTrue(toolout == expected_out,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_out,toolout))

        expected_err = ""
        self.assertTrue(toolerr == expected_err,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_err,toolerr))


    def test_3_commands_no_templates(self):
        """launching invoke_app with multiple non-templated -C commands
           should run the last command.
           ex: invoke_app -C "echo hi" -C "echo bye" -C "echo yeah"
           should produce: yeah
        """

        # build our invoke_app command
        command_list = [INVOKE_APP_PATH,
                        '-C','echo hi',
                        '-C','echo bye',
                        '-C','echo yeah']

        # run invoke_app
        stdout,stderr = run_invoke_app(command_list)
        toolout,toolerr = get_tool_output(stdout,stderr)

        # parse the output
        expected_out = "yeah"
        self.assertTrue(toolout == expected_out,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_out,toolout))

        expected_err = ""
        self.assertTrue(toolerr == expected_err,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_err,toolerr))


    def test_1_template_0_default_run_template_1(self):
        """launching invoke_app with one -C template command should
           launch toolparams to run the command
           ex: invoke_app -C "cat @@file(datafile1)"
           should launch: toolparams 'cat @@file(datafile1)'
           and produce: this is datafile1
        """

        # create our parameters file
        parameters_text = '\n'.join([
            "file(datafile1):%s" % os.path.join(os.getcwd(),'datafile1'),
        ])

        # build our invoke_app command
        command_list = [INVOKE_APP_PATH,
                        '-C','cat @@file(datafile1)']

        # run invoke_app
        stdout,stderr = run_invoke_app(command_list,parameters_text)
        toolout,toolerr = get_tool_output(stdout,stderr)

        # parse the output
        expected_out = DATA['datafile1']
        self.assertTrue(toolout == expected_out,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_out,toolout))

        expected_err = ""
        self.assertTrue(toolerr == expected_err,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_err,toolerr))


    def test_1_template_1_default_run_template_1(self):
        """launching invoke_app with one -C template command
           and one -C non-template command should launch
           toolparams to run the command. when the parameters file
           has a valid reference to file(datafile1), the templated
           command should be run.
           ex: invoke_app -C "cat @@file(datafile1)" -C "echo hi"
           should launch: toolparams 'cat @@file(datafile1)' -default 'echo hi'
           and produce: this is datafile1
        """

        # create our parameters file
        parameters_text = '\n'.join([
            "file(datafile1):%s" % os.path.join(os.getcwd(),'datafile1'),
        ])

        # build our invoke_app command
        command_list = [INVOKE_APP_PATH,
                        '-C','cat @@file(datafile1)',
                        '-C','echo hi']

        # run invoke_app
        stdout,stderr = run_invoke_app(command_list,parameters_text)
        toolout,toolerr = get_tool_output(stdout,stderr)

        # parse the output
        expected_out = DATA['datafile1']
        self.assertTrue(toolout == expected_out,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_out,toolout))

        expected_err = ""
        self.assertTrue(toolerr == expected_err,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_err,toolerr))


    def test_1_template_1_default_run_default(self):
        """launching invoke_app with one -C template command
           and one -C non-template command should launch
           toolparams to run the command. when the parameters file
           does not have a valid reference to file(datafile1), the
           non-templated command should be run.
           ex: invoke_app -C "cat @@file(datafile1)" -C "echo hi"
           should launch: toolparams 'cat @@file(datafile1)' -default 'echo hi'
           and produce: hi
        """

        # create our parameters file
        parameters_text = '\n'.join([
        ])

        # build our invoke_app command
        command_list = [INVOKE_APP_PATH,
                        '-C','cat @@file(datafile1)',
                        '-C','echo hi']

        # run invoke_app
        stdout,stderr = run_invoke_app(command_list,parameters_text)
        toolout,toolerr = get_tool_output(stdout,stderr)

        # parse the output
        expected_out = "hi"
        self.assertTrue(toolout == expected_out,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_out,toolout))

        expected_err = ""
        self.assertTrue(toolerr == expected_err,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_err,toolerr))


    def test_1_default_1_template_run_template_1(self):
        """launching invoke_app with one -C non-template command
           and one -C template command should launch
           toolparams to run the command. when the parameters file
           has a valid reference to file(datafile1), the templated
           command should be run.
           ex: invoke_app -C "echo hi" -C "cat @@file(datafile1)"
           should launch: toolparams 'cat @@file(datafile1)' -default 'echo hi'
           and produce: this is datafile1
        """

        # create our parameters file
        parameters_text = '\n'.join([
            "file(datafile1):%s" % os.path.join(os.getcwd(),'datafile1'),
        ])

        # build our invoke_app command
        command_list = [INVOKE_APP_PATH,
                        '-C','echo hi',
                        '-C','cat @@file(datafile1)']

        # run invoke_app
        stdout,stderr = run_invoke_app(command_list,parameters_text)
        toolout,toolerr = get_tool_output(stdout,stderr)

        # parse the output
        expected_out = DATA['datafile1']
        self.assertTrue(toolout == expected_out,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_out,toolout))

        expected_err = ""
        self.assertTrue(toolerr == expected_err,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_err,toolerr))


    def test_1_default_1_template_run_default(self):
        """launching invoke_app with one -C non-template command
           and one -C template command should launch
           toolparams to run the command. when the parameters file
           does not have a valid reference to file(datafile1), the
           non-templated command should be run.
           ex: invoke_app -C "echo hi" -C "cat @@file(datafile1)"
           should launch: toolparams 'cat @@file(datafile1)' -default 'echo hi'
           and produce: hi
        """

        # create our parameters file
        parameters_text = '\n'.join([
        ])

        # build our invoke_app command
        command_list = [INVOKE_APP_PATH,
                        '-C','echo hi',
                        '-C','cat @@file(datafile1)']

        # run invoke_app
        stdout,stderr = run_invoke_app(command_list,parameters_text)
        toolout,toolerr = get_tool_output(stdout,stderr)

        # parse the output
        expected_out = "hi"
        self.assertTrue(toolout == expected_out,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_out,toolout))

        expected_err = ""
        self.assertTrue(toolerr == expected_err,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_err,toolerr))


    def test_2_templates_0_default_run_template_1(self):
        """launching invoke_app with two -C template commands
           and zero -C non-template command should launch
           toolparams to run the command. when the parameters file
           has a valid reference to file(datafile1), the
           appropriate templated command should be run.
           ex: invoke_app -C "cat @@file(datafile1)" -C "cat @@file(datafile2)"
           should launch: toolparams 'cat @@file(datafile1)' -or 'cat @@file(datafile2)'
           and produce: this is datafile1
        """

        # create our parameters file
        parameters_text = '\n'.join([
            "file(datafile1):%s" % os.path.join(os.getcwd(),'datafile1'),
        ])

        # build our invoke_app command
        command_list = [INVOKE_APP_PATH,
                        '-C','cat @@file(datafile1)',
                        '-C','cat @@file(datafile2)']

        # run invoke_app
        stdout,stderr = run_invoke_app(command_list,parameters_text)
        toolout,toolerr = get_tool_output(stdout,stderr)

        # parse the output
        expected_out = DATA['datafile1']
        self.assertTrue(toolout == expected_out,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_out,toolout))

        expected_err = ""
        self.assertTrue(toolerr == expected_err,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_err,toolerr))


    def test_2_templates_0_default_run_template_2(self):
        """launching invoke_app with two -C template commands
           and zero -C non-template command should launch
           toolparams to run the command. when the parameters file
           has a valid reference to file(datafile2), the
           appropriate templated command should be run.
           ex: invoke_app -C "cat @@file(datafile1)" -C "cat @@file(datafile2)"
           should launch: toolparams 'cat @@file(datafile1)' -or 'cat @@file(datafile2)'
           and produce: this is datafile2
        """

        # create our parameters file
        parameters_text = '\n'.join([
            "file(datafile2):%s" % os.path.join(os.getcwd(),'datafile2'),
        ])

        # build our invoke_app command
        command_list = [INVOKE_APP_PATH,
                        '-C','cat @@file(datafile1)',
                        '-C','cat @@file(datafile2)']

        # run invoke_app
        stdout,stderr = run_invoke_app(command_list,parameters_text)
        toolout,toolerr = get_tool_output(stdout,stderr)

        # parse the output
        expected_out = DATA['datafile2']
        self.assertTrue(toolout == expected_out,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_out,toolout))

        expected_err = ""
        self.assertTrue(toolerr == expected_err,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_err,toolerr))


    def test_2_templates_1_default_run_template_1(self):
        """launching invoke_app with two -C template commands
           and one -C non-template command should launch
           toolparams to run the command. when the parameters file
           has a valid reference to file(datafile1), the
           appropriate templated command should be run.
           ex: invoke_app -C "cat @@file(datafile1)" -C "cat @@file(datafile2)" -C "echo hi"
           should launch: toolparams 'cat @@file(datafile1)' -or 'cat @@file(datafile2)' -default 'echo hi'
           and produce: this is datafile1
        """

        # create our parameters file
        parameters_text = '\n'.join([
            "file(datafile1):%s" % os.path.join(os.getcwd(),'datafile1'),
        ])

        # build our invoke_app command
        command_list = [INVOKE_APP_PATH,
                        '-C','cat @@file(datafile1)',
                        '-C','cat @@file(datafile2)',
                        '-C','echo hi']

        # run invoke_app
        stdout,stderr = run_invoke_app(command_list,parameters_text)
        toolout,toolerr = get_tool_output(stdout,stderr)

        # parse the output
        expected_out = DATA['datafile1']
        self.assertTrue(toolout == expected_out,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_out,toolout))

        expected_err = ""
        self.assertTrue(toolerr == expected_err,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_err,toolerr))


    def test_2_templates_1_default_run_template_2(self):
        """launching invoke_app with two -C template commands
           and one -C non-template command should launch
           toolparams to run the command. when the parameters file
           has a valid reference to file(datafile2), the
           appropriate templated command should be run.
           ex: invoke_app -C "cat @@file(datafile1)" -C "cat @@file(datafile2)" -C "echo hi"
           should launch: toolparams 'cat @@file(datafile1)' -or 'cat @@file(datafile2)' -default 'echo hi'
           and produce: this is datafile2
        """

        # create our parameters file
        parameters_text = '\n'.join([
            "file(datafile2):%s" % os.path.join(os.getcwd(),'datafile2'),
        ])

        # build our invoke_app command
        command_list = [INVOKE_APP_PATH,
                        '-C','cat @@file(datafile1)',
                        '-C','cat @@file(datafile2)',
                        '-C','echo hi']

        # run invoke_app
        stdout,stderr = run_invoke_app(command_list,parameters_text)
        toolout,toolerr = get_tool_output(stdout,stderr)

        # parse the output
        expected_out = DATA['datafile2']
        self.assertTrue(toolout == expected_out,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_out,toolout))

        expected_err = ""
        self.assertTrue(toolerr == expected_err,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_err,toolerr))


    def test_2_templates_1_default_run_default(self):
        """launching invoke_app with two -C template commands
           and one -C non-template command should launch
           toolparams to run the command. when the parameters file
           does not have a valid reference, the
           appropriate non-templated command should be run.
           ex: invoke_app -C "cat @@file(datafile1)" -C "cat @@file(datafile2)" -C "echo hi"
           should launch: toolparams 'cat @@file(datafile1)' -or 'cat @@file(datafile2)' -default 'echo hi'
           and produce: hi
        """

        # create our parameters file
        parameters_text = '\n'.join([
        ])

        # build our invoke_app command
        command_list = [INVOKE_APP_PATH,
                        '-C','cat @@file(datafile1)',
                        '-C','cat @@file(datafile2)',
                        '-C','echo hi']

        # run invoke_app
        stdout,stderr = run_invoke_app(command_list,parameters_text)
        toolout,toolerr = get_tool_output(stdout,stderr)

        # parse the output
        expected_out = 'hi'
        self.assertTrue(toolout == expected_out,
            "expected \"%s\"\nreceived \"%s\"\nstdout:\n%s\nstderr:\n%s" \
            % (expected_out,toolout,stdout,stderr))

        expected_err = ""
        self.assertTrue(toolerr == expected_err,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_err,toolerr))


    def test_1_template_1_default_1_template_run_template_1(self):
        """launching invoke_app with one -C template command
           and one -C non-template command and a second
           -C template comamnd should launch toolparams
           to run the command. when the parameters file
           has a valid reference to file(datafile1), the
           appropriate templated command should be run.
           ex: invoke_app -C "cat @@file(datafile1)" -C "echo hi" -C "cat @@file(datafile2)"
           should launch: toolparams 'cat @@file(datafile1)' -or 'cat @@file(datafile2)' -default 'echo hi'
           and produce: this is datafile1
        """

        # create our parameters file
        parameters_text = '\n'.join([
            "file(datafile1):%s" % os.path.join(os.getcwd(),'datafile1'),
        ])

        # build our invoke_app command
        command_list = [INVOKE_APP_PATH,
                        '-C','cat @@file(datafile1)',
                        '-C','echo hi',
                        '-C','cat @@file(datafile2)']

        # run invoke_app
        stdout,stderr = run_invoke_app(command_list,parameters_text)
        toolout,toolerr = get_tool_output(stdout,stderr)

        # parse the output
        expected_out = DATA['datafile1']
        self.assertTrue(toolout == expected_out,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_out,toolout))

        expected_err = ""
        self.assertTrue(toolerr == expected_err,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_err,toolerr))


    def test_1_template_1_default_1_template_run_template_2(self):
        """launching invoke_app with one -C template commands
           and one -C non-template command and a second
           -C template command should launch toolparams
           to run the command. when the parameters file
           has a valid reference to file(datafile2), the
           appropriate templated command should be run.
           ex: invoke_app -C "cat @@file(datafile1)" -C "echo hi" -C "cat @@file(datafile2)"
           should launch: toolparams 'cat @@file(datafile1)' -or 'cat @@file(datafile2)' -default 'echo hi'
           and produce: this is datafile2
        """

        # create our parameters file
        parameters_text = '\n'.join([
            "file(datafile2):%s" % os.path.join(os.getcwd(),'datafile2'),
        ])

        # build our invoke_app command
        command_list = [INVOKE_APP_PATH,
                        '-C','cat @@file(datafile1)',
                        '-C','echo hi',
                        '-C','cat @@file(datafile2)']

        # run invoke_app
        stdout,stderr = run_invoke_app(command_list,parameters_text)
        toolout,toolerr = get_tool_output(stdout,stderr)

        # parse the output
        expected_out = DATA['datafile2']
        self.assertTrue(toolout == expected_out,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_out,toolout))

        expected_err = ""
        self.assertTrue(toolerr == expected_err,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_err,toolerr))


    def test_1_template_1_default_1_template_run_default(self):
        """launching invoke_app with one -C template command
           and one -C non-template command and a second
           -C template command should launch toolparams
           to run the command. when the parameters file
           does not have a valid reference, the
           appropriate non-templated command should be run.
           ex: invoke_app -C "cat @@file(datafile1)" -C "echo hi" -C "cat @@file(datafile2)"
           should launch: toolparams 'cat @@file(datafile1)' -or 'cat @@file(datafile2)' -default 'echo hi'
           and produce: hi
        """

        # create our parameters file
        parameters_text = '\n'.join([
        ])

        # build our invoke_app command
        command_list = [INVOKE_APP_PATH,
                        '-C','cat @@file(datafile1)',
                        '-C','echo hi',
                        '-C','cat @@file(datafile2)']

        # run invoke_app
        stdout,stderr = run_invoke_app(command_list,parameters_text)
        toolout,toolerr = get_tool_output(stdout,stderr)

        # parse the output
        expected_out = 'hi'
        self.assertTrue(toolout == expected_out,
            "expected \"%s\"\nreceived \"%s\"\nstdout:\n%s\nstderr:\n%s" \
            % (expected_out,toolout,stdout,stderr))

        expected_err = ""
        self.assertTrue(toolerr == expected_err,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_err,toolerr))


    def test_1_default_2_templates_run_template_1(self):
        """launching invoke_app with one -C non-template command
           and two -C template commands should launch toolparams
           to run the command. when the parameters file
           has a valid reference to file(datafile1), the
           appropriate templated command should be run.
           ex: invoke_app -C "echo hi" -C "cat @@file(datafile1)" -C "cat @@file(datafile2)"
           should launch: toolparams 'cat @@file(datafile1)' -or 'cat @@file(datafile2)' -default 'echo hi'
           and produce: this is datafile1
        """

        # create our parameters file
        parameters_text = '\n'.join([
            "file(datafile1):%s" % os.path.join(os.getcwd(),'datafile1'),
        ])

        # build our invoke_app command
        command_list = [INVOKE_APP_PATH,
                        '-C','echo hi',
                        '-C','cat @@file(datafile1)',
                        '-C','cat @@file(datafile2)']

        # run invoke_app
        stdout,stderr = run_invoke_app(command_list,parameters_text)
        toolout,toolerr = get_tool_output(stdout,stderr)

        # parse the output
        expected_out = DATA['datafile1']
        self.assertTrue(toolout == expected_out,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_out,toolout))

        expected_err = ""
        self.assertTrue(toolerr == expected_err,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_err,toolerr))


    def test_1_default_2_templates_run_template_2(self):
        """launching invoke_app with one -C non-template command
           and two -C template commands should launch toolparams
           to run the command. when the parameters file
           has a valid reference to file(datafile2), the
           appropriate templated command should be run.
           ex: invoke_app -C "echo hi" -C "cat @@file(datafile1)" -C "cat @@file(datafile2)"
           should launch: toolparams 'cat @@file(datafile1)' -or 'cat @@file(datafile2)' -default 'echo hi'
           and produce: this is datafile2
        """

        # create our parameters file
        parameters_text = '\n'.join([
            "file(datafile2):%s" % os.path.join(os.getcwd(),'datafile2'),
        ])

        # build our invoke_app command
        command_list = [INVOKE_APP_PATH,
                        '-C','echo hi',
                        '-C','cat @@file(datafile1)',
                        '-C','cat @@file(datafile2)']

        # run invoke_app
        stdout,stderr = run_invoke_app(command_list,parameters_text)
        toolout,toolerr = get_tool_output(stdout,stderr)

        # parse the output
        expected_out = DATA['datafile2']
        self.assertTrue(toolout == expected_out,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_out,toolout))

        expected_err = ""
        self.assertTrue(toolerr == expected_err,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_err,toolerr))


    def test_1_default_2_template_run_default(self):
        """launching invoke_app with one -C non-template command
           and two -C template command should launch toolparams
           to run the command. when the parameters file
           does not have a valid reference, the
           appropriate non-templated command should be run.
           ex: invoke_app -C "echo hi" -C "cat @@file(datafile1)" -C "cat @@file(datafile2)"
           should launch: toolparams 'cat @@file(datafile1)' -or 'cat @@file(datafile2)' -default 'echo hi'
           and produce: hi
        """

        # create our parameters file
        parameters_text = '\n'.join([
        ])

        # build our invoke_app command
        command_list = [INVOKE_APP_PATH,
                        '-C','echo hi',
                        '-C','cat @@file(datafile1)',
                        '-C','cat @@file(datafile2)']

        # run invoke_app
        stdout,stderr = run_invoke_app(command_list,parameters_text)
        toolout,toolerr = get_tool_output(stdout,stderr)

        # parse the output
        expected_out = 'hi'
        self.assertTrue(toolout == expected_out,
            "expected \"%s\"\nreceived \"%s\"\nstdout:\n%s\nstderr:\n%s" \
            % (expected_out,toolout,stdout,stderr))

        expected_err = ""
        self.assertTrue(toolerr == expected_err,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_err,toolerr))


    def test_2_templates_1_default_run_index_1(self):
        """launching invoke_app with two -C template commands
           and one -C non-template command should launch
           toolparams to run the command. when the parameters file
           has a valid positional reference, the
           appropriate templated command should be run.
           ex: invoke_app -C "cat @@file(#1)" -C "cat @@file(datafile2)" -C "echo hi"
           should launch: toolparams 'cat @@file(#1)' -or 'cat @@file(datafile2)' -default 'echo hi'
           and produce: this is datafile1
        """

        # create our parameters file
        parameters_text = '\n'.join([
            "file(datafile1):%s" % os.path.join(os.getcwd(),'datafile1'),
        ])

        # build our invoke_app command
        command_list = [INVOKE_APP_PATH,
                        '-C','cat @@file(#1)',
                        '-C','cat @@file(datafile2)',
                        '-C','echo hi']

        # run invoke_app
        stdout,stderr = run_invoke_app(command_list,parameters_text)
        toolout,toolerr = get_tool_output(stdout,stderr)

        # parse the output
        expected_out = DATA['datafile1']
        self.assertTrue(toolout == expected_out,
            "expected \"%s\"\nreceived \"%s\"\nstdout:\n%s\nstderr:\n%s" \
            % (expected_out,toolout,stdout,stderr))

        expected_err = ""
        self.assertTrue(toolerr == expected_err,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_err,toolerr))


#    def test_positional_2_templates_1_default_run_index_1_1(self):
#        """launching invoke_app with two -C template commands
#           and one -C non-template command should launch
#           toolparams to run the command. when the parameters file
#           has two valid positional references, the first matching
#           templated command should be run.
#           ex: invoke_app -C "cat @@file(#1)" -C "cat @@file(#1) @@file(#2)" -C "echo hi"
#           should launch: toolparams 'cat @@file(#1)' -or 'cat @@file(datafile2)' -default 'echo hi'
#           and produce: this is datafile1
#        """
#
#        # create our parameters file
#        parameters_text = '\n'.join([
#            "file(datafile1):%s" % os.path.join(os.getcwd(),'datafile1'),
#            "file(datafile2):%s" % os.path.join(os.getcwd(),'datafile2'),
#        ])
#
#        # build our invoke_app command
#        command_list = [INVOKE_APP_PATH,
#                        '-C','cat @@file(#1)',
#                        '-C','cat @@file(#1) @@file(#2)',
#                        '-C','echo hi']
#
#        # run invoke_app
#        stdout,stderr = run_invoke_app(command_list,parameters_text)
#        toolout,toolerr = get_tool_output(stdout,stderr)
#
#        # parse the output
#        expected_out = DATA['datafile1']
#        self.assertTrue(toolout == expected_out,
#            "expected \"%s\"\nreceived \"%s\"\nstdout:\n%s\nstderr:\n%s" \
#            % (expected_out,toolout,stdout,stderr))
#
#        expected_err = ""
#        self.assertTrue(toolerr == expected_err,
#            "expected \"%s\"\nreceived \"%s\"" \
#            % (expected_err,toolerr))


    def test_positional_2_templates_1_default_run_index_1_2(self):
        """launching invoke_app with two -C template commands
           and one -C non-template command should launch
           toolparams to run the command. when the parameters file
           has two valid positional references, the first matching
           templated command should be run.
           ex: invoke_app -C "cat @@file(#1) @@file(#2)" -C "cat @@file(#1)" -C "echo hi"
           should launch: toolparams 'cat @@file(#1) @@file(#2)' -or 'cat @@file(#1)' -default 'echo hi'
           and produce:
           this is datafile1
           this is datafile2
        """

        # create our parameters file
        parameters_text = '\n'.join([
            "file(datafile1):%s" % os.path.join(os.getcwd(),'datafile1'),
            "file(datafile2):%s" % os.path.join(os.getcwd(),'datafile2'),
        ])

        # build our invoke_app command
        command_list = [INVOKE_APP_PATH,
                        '-C','cat @@file(#1) @@file(#2)',
                        '-C','cat @@file(#1)',
                        '-C','echo hi']

        # run invoke_app
        stdout,stderr = run_invoke_app(command_list,parameters_text)
        toolout,toolerr = get_tool_output(stdout,stderr)

        # parse the output
        expected_out = "%s%s" % (DATA['datafile1'],DATA['datafile2'])
        self.assertTrue(toolout == expected_out,
            "expected \"%s\"\nreceived \"%s\"\nstdout:\n%s\nstderr:\n%s" \
            % (expected_out,toolout,stdout,stderr))

        expected_err = ""
        self.assertTrue(toolerr == expected_err,
            "expected \"%s\"\nreceived \"%s\"" \
            % (expected_err,toolerr))


    def tearDown(self):
        if os.path.isfile(PARAMETERS_PATH):
            os.remove(PARAMETERS_PATH)

if __name__ == '__main__':
    pretest_setup()
    tr = unittest.TextTestRunner(stream=sys.stdout,verbosity=0)
    unittest.main(testRunner=tr)
    # unittest.main(testRunner=tr,exit=False)
    posttest_cleanup()

