# @package      hubzero-mw2-common
# @file         log.py
# @copyright    Copyright (c) 2016-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Based on previous work by Richard L. Kennell and Nicholas Kisseberth
#
# Copyright (c) 2016-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#

"""
 Logging Functions
 Capture stdout and stderr to a file.
 Save stdout and make it available for intentional output
 For debugging, use stderr instead of a real file.

 Requirement (from Nicholas Kisseberth): show time and date for each log entry
 Entries are interleaved for different processes and calls: show process id.
 Requirement (from Rick) make log entries as short as possible.

 Rationale for not making this a class: it would need to be passed around everywhere.
 Backgrounding is done here because open file descriptors can interfere with dissociating
 a process.
"""
import os
import sys
import time
import cgi
from errors import ChildError, MaxwellError
from constants import VERBOSE

logfile = None
_user_print = None  # point to htmlprint or to ttyprint, depending on whether we're debugging
log_id = ""
saved_out = None

def save_out():
  """Save a copy of std_out to be able to send out replies like "OK".  It has to be
  called before logging is setup, otherwise the fd will have been overwritten.
  However, calling this prevents things like maxwell_service's startvnc command from
  completing.  It actually does its job but the calling program hangs forever.  See
  discard_out().
  """
  global saved_out
  saved_out = os.fdopen(os.dup(sys.stdout.fileno()), 'w')

def discard_out():
  """trigger garbage collection on the file object and so allow dissociated processes
  to complete dissociation.  If this isn't done, any background process will prevent grandparents
  from terminating.  It can be observed as a linearization of all maxwell activity, so that the
  background display creation slows down the display of an application."""
  global saved_out
  try:
    saved_out.close()
  except (IOError, ValueError, AttributeError):
    pass # already closed
  saved_out = None

  # typically, file descriptors 3-22 are also open and prevent an apache restart
  # when background processes are hanging around (i.e., when sessions are running).
  # So, we need to close them.  Doing it after calling saved_out.close() prevents errors.
  for fd in range(3,1024):
    try:
      os.close(fd)
    except:
      pass

def ttyprint(msg):
  """Intentionally printing output to the original stdout, without HTML formatting"""
  saved_out.write(msg + "\n")
  saved_out.flush() # probably redundant, stdout flushes when a "\n" is encountered

def _htmlprint(msg):
  """Format message for display to user (HTML), unless we're debugging with a TTY.
  This is used in com_tools (Joomla component in the CMS for middleware administration) for example.
  """
  if saved_out is not None:
    saved_out.write("<b>%s%s</b><br />\n" % (cgi.escape(log_id), cgi.escape(msg)))
    saved_out.flush() # probably redundant, stdout flushes when a "\n" is encountered

def setup_log(logfile_name, new_log_id):
  """Set up errors to go to the log file.
  Printed text is shown to the user.  Everything on stderr is logged.
  Return the file descriptor to be used to avoid using the global statement."""
  global logfile, _user_print, log_id

  if new_log_id is not None:
    log_id = new_log_id + ":"

  if os.isatty(1) and os.isatty(2):
    sys.stderr.write("tty detected, will print errors to tty\n")
    # for debugging, manual invocation, do not redirect stderr
    sys.stdout = sys.stderr
    logfile = sys.stderr
    # output without special formatting
    _user_print = ttyprint
    return
  else:
    open_log(logfile_name)
    _user_print = _htmlprint

def log_setid(new_log_id):
  log_id = new_log_id + ":"

def open_log(logfile_name):
  """Open the log file and copy the file descriptor into stdout and stderr.
  This captures all output from called programs.  This functionality is
  separated from setup_log to make testing easier.  It bypasses the tty check so
  it is not intended to be called directly."""
  global logfile
  try:
    # os.O_NOFOLLOW flag: symlinks to logs are used sometimes (need to get rid of that)
    fd = os.open(logfile_name, os.O_CREAT|os.O_APPEND|os.O_WRONLY|os.O_NOFOLLOW, 0640)
  except (OSError, IOError), e:
    sys.stderr.write("Unable to open log file: %s" % e)
    sys.exit(1)
  # force all output to go to the file
  os.dup2(fd, 1)
  os.dup2(fd, 2)
  os.close(fd)
  logfile = sys.stderr

def log_exc(exc):
  import traceback
  cla, exc, trbk = sys.exc_info()
  excName = cla.__name__
  try:
    excArgs = exc.__dict__["args"]
  except KeyError:
    excArgs = ""
  excTb = traceback.format_tb(trbk, 10)
  #excTb.replace("\\n", "\n")
  log("%s\n%s\n%s" % (excName, excArgs, excTb))

def log(msg):
  """ local time (with DST if set),  e.g., '07/05 13:06:58'"""
  timestamp = time.strftime("%m/%d %H:%M:%S")
  if VERBOSE:
    logfile.write("[%s]%d: %s%s\n" % (timestamp, os.getpid(), log_id, msg))
  else:
    logfile.write("[%s] %s\n" % (timestamp, msg))
  logfile.flush()

def user_print(msg):
  """Format message for display to user (HTML), unless we're debugging with a TTY"""
  _user_print(msg)

def print_n_log(msg):
  """When the same message is to be both shown to the user and logged, use
    this function to avoid writing the message twice in the code."""
  if saved_out is not None:
    _user_print(msg)
  log(msg)

def dns_resolve(hostname):
  """Query A records"""
  # try reading /etc/resolv.conf instead
  search_domain = '.nanohub.org'
  try:
    # silly thing ignores "search" specifications in /etc/resolv.conf
    # will only work with FQDNs
    import DNS
  except ImportError:
    log('Could not import DNS module from PyDNS.')
    return 'unknown'
  try:
    DNS.ParseResolvConf()
    r=DNS.Request(qtype='A')
    if hostname.find(".") < 0:
      # no dots in name.  Search which domain?
      res = r.req(hostname + search_domain)
    else:
      res = r.req(hostname)
    for ans in res.answers:
      if ans['typename'] == 'A':
        return ans['data']
    log("query on %s returned %s" % (hostname, res.answers))
    return "unknown"
  except StandardError:
    # log and continue, this is a non-essential op
    # do not show to user, so log only
    log("Problem with domain name service.")
    return "unknown"

def dns_reverse(ipaddr):
  """Query PTR records, that is the remote DNS server to which this IP address belongs
  Note that the reply can be malicious.  Used for logging purposes.  Import is done here
  because the DNS module is not installed on all hosts."""
  try:
    import DNS
  except ImportError:
    log("Could not import DNS module.")
    return "unknown"
  # import only when this function is called;
  # this allows the program to run on hosts that don't have DNS installed. (if it doesn't call this)
  try:
    arr=ipaddr.split('.')
    arr.reverse()
    revaddr = '.'.join(arr) + '.in-addr.arpa'
    DNS.ParseResolvConf()
    r=DNS.Request(qtype='PTR')
    res = r.req(revaddr)
    for ans in res.answers:
      if ans['typename'] == 'PTR':
        return ans['data']
    return "unknown"
  except StandardError:
    # log and continue, this is a non-essential op
    # do not show to user, so log only
    log("Problem with domain name service.")
    return "unknown"

#=============================================================================
# Process Dissociation Support
#=============================================================================

def background():
  """Fork a dissociated process.  Return 1 for child.  Return 0 for original parent.
  This involves forking twice;  the intermediate process must not return."""

  sys.stdout.flush()
  sys.stderr.flush()

  # Dissociate from the parent so that it doesn't wait.
  # Use the double-fork trick.
  try:
    pid = os.fork()
    if pid > 0:
      return False  # Indicate we're the first parent.
  except OSError, ex:
    raise MaxwellError("fork #1 failed: %d (%s)" % (ex.errno, ex.strerror))

  # This is the child.  Dissociate from the parent.
  discard_out() # close file descriptors
  os.chdir("/") # so we don't prevent unmounting the filesystem
  try:
    # Become a process leader of a new process group, to avoid getting some signals
    os.setsid()
    os.dup2(2,1) # everything goes to stderr
  except OSError, ex:
    raise ChildError("process management failed: %d (%s)" % (ex.errno, ex.strerror))

  # avoid zombie process
  try:
    pid = os.fork()
    if pid > 0:
      os._exit(0)  # this process isn't needed and shouldn't return!
  except OSError, ex:
    raise ChildError("fork #2 failed: %d (%s)" % (ex.errno, ex.strerror))

  # At this point, this is a child of a child, and immune to "zombification"
  return True  # Indicate we're the desired child.

def dissociate():
  """Become a dissociated child.  Original process exits."""
  if not background():
    sys.exit(0)

def null_input():
  try:
    fd = os.open("/dev/null", os.O_RDONLY)
    os.dup2(fd,0)
  except OSError, ex:
    raise MaxwellError("Could not open /dev/null for input, %s", ex)
