# @package      hubzero-mw2-common
# @file         support.py
# @copyright    Copyright (c) 2016-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Based on previous work by Richard L. Kennell and Nicholas Kisseberth
#
# Copyright (c) 2016-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#

import os
import sys
import stat
import time
from errors import PrivateError
from constants import RUN_DIR

"""
Library of miscellaneous functions needed by various packages, placed here to avoid
code duplication.
"""

def check_rundir():
  """ Check that the running directory exists and is safe, and cd into it. """
  try:
    lock_stat = os.lstat(RUN_DIR)
  except OSError:
    # does not exist
    try:
      os.umask(0077)
      os.makedirs(RUN_DIR)
      lock_stat = os.lstat(RUN_DIR)
    except OSError:
      print "Unable to create %s." % RUN_DIR
      sys.exit(1)

  # check that it has the expected permissions and ownership
  # check that we are the owner of this dir and that others can't write
  if lock_stat[stat.ST_MODE] & stat.S_IWOTH:
    print "%s directory has unsafe permissions" % RUN_DIR
    sys.exit(1)

  usr_id = lock_stat[stat.ST_UID]
  if usr_id != os.geteuid() and usr_id != 0:
    print "%s directory has incorrect owner: %s" % (RUN_DIR, usr_id)
    sys.exit(1)

  # We always run here:
  os.chdir(RUN_DIR)


def release_dirlock(lock_path):
  if os.path.exists(lock_path):
    os.rmdir(lock_path)

def get_dirlock(lock_path, expiration=50, retries=6, sleeptime=1):
  """ get directory lock at specified path.
  If directory already exists, check how old it is.
    If directory is older than expiration (in seconds), delete it.
    else if there are retries left, sleep
    decrement retries
    loop.
  else create it.
  Suggest that retries*sleeptime > expiration."""
  attempt = 0
  while attempt < retries:
    try:
      os.mkdir(lock_path, 0600)
    except OSError:
      attempt += 1
      # File exists, so check how long it's been
      try:
        lock_stat = os.stat(lock_path)
      except OSError: # file has been deleted in the meantime
        continue # try again
      ts = lock_stat[stat.ST_MTIME]
      if ts + expiration > time.time():
        if sleeptime > 0:
          time.sleep(sleeptime)
        else:
          raise PrivateError("Unable to obtain lock '%s'" % lock_path)
      else:
        # delete old lock and try again
        # if we can't delete it, exception will bubble up
        os.rmdir(lock_path)
      continue
    break
  if attempt >= retries:
    raise PrivateError("Too many retries to obtain lock '%s'" % lock_path)

def genpasswd(length,letters=None):
  """Generate a cleartext password using urandom, with even probability of each character.
  this is done by discarding random bytes that don't fit into an even multiple of the number
  of allowed characters.  It runs as quickly as the previous version that had a bias for the
  selection of some characters."""
  if letters is None:
    # no quotes (SQL) and spaces please
    letters="""abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789-_+=~;:,.<>?/!@#$%^&*()[]{}"""
  y = len(letters)
  # get numbers in a batch, it's faster than one by one
  r = os.urandom(length*2)
  i = 0 # how many random characters we've generated so far
  j = 0 # index to next unused random number in array
  output = ""
  maxdiv = (256 / y) * y
  while i < length:
    # reject results that would skew the freq. distribution of characters
    if ord(r[j]) < maxdiv:
      output += letters[ord(r[j]) % y]
      i += 1
    j += 1
    if j >= length*2:
      # we've run out of random numbers, get some more
      r = os.urandom(length*2)
      j = 0
  return output

