# @package      hubzero-mw2-common
# @file         container.py
# @copyright    Copyright (c) 2016-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# OpenVZ code based on previous work by Richard L. Kennell and Nicholas Kisseberth
# Docker container support for HUBzero middleware
#
# Copyright (c) 2016-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#

"""
container.py: class to create, manipulate, and stop containers, and processes within
Designed for simfs support by OpenVZ
Used on execution hosts by the middleware service.  This class should encapsulate OpenVZ idiosyncrasies
and in theory could be replaced by another container.py file to handle other virtualization mechanisms
Some OpenVZ synonyms:
veid = containerID = ctid
The display number in the database is the ctid
"""
import subprocess
import socket
import os
import stat
import sys
import grp
import time
import threading
import re

from errors import MaxwellError
from constants import CONTAINER_K, VERBOSE, RUN_DIR, SERVICE_LOG
from log import log, log_exc, setup_log
from user_account import make_User_account, User_account, User_account_JSON, User_account_anonymous
DEBUG = True

def make_Container(disp, machine_number, overrides={}):
  """Return a Container class or subclass instance depending on the OS and configuration"""
  try:
    if overrides["class"] == "ContainerVZ7":
      # print "creating OpenVZ container %d" % disp
      return ContainerVZ7(disp, machine_number, overrides)
    elif overrides["class"] == "ContainerDocker":
      # print "creating Docker container"
      return ContainerDocker(disp, machine_number, overrides)
    elif overrides["class"] == "ContainerAWS":
      return ContainerAWS(disp, machine_number, overrides)
    else:
      # print "creating default OpenVZ container"
      return Container(disp, machine_number, overrides)
  except KeyError:
    pass
  # try to open /etc/redhat-release, if it contains "Virtuozzo Linux release 7" use the ContainerVZ7 class
  # else return an old container
  try:
    with open("/etc/redhat-release") as file:
      release = file.read()
      if release.find("Virtuozzo Linux release 7") == 0:
        return ContainerVZ7(disp, machine_number, overrides)
  except IOError:
    # e.g., file does not exist
    pass
  return Container(disp, machine_number, overrides)

def log_subprocess(p, info=None):
  """Communicate after Popen, and log any output"""
  (stdout, stderr) = p.communicate(info)
  if len(stderr) > 0:
    if p.returncode == 0:
      log("success:" + stderr.rstrip())
    else:
      log("stderr: " + stderr.rstrip())
  if len(stdout) > 0:
    log("stdout: " + stdout.rstrip())

class Container():
  """A container: Virtual Private Server (VPS) uniquely identified by a veid.
  Note that new documentation uses "CTID" instead of veid: ConTainer's IDentifer (CTID)
  According to the OpenVZ Users Guide, CTIDs 0-100 are reserved and should not be used.
  OpenVZ only currently uses CTID 0 but recommends reserving CTID's 0-100 for non-use

  Paths used by OpenVZ  (see OpenVZ User's Guide Version 2.7.0-8 by SWsoft):

  ct_private  (e.g., "/vz/private/veid")
  This is a path to the Container private area where Virtuozzo Containers 4.0 keeps its private data.

  ct_root (e.g., "/vz/root/veid")
  This is a path to the Container root folder where the Container private area is mounted.

  vz_root
  This is a path to the Virtuozzo folder where Virtuozzo Containers program files are located.

  depends on script INTERNAL_PATH +"mergeauth"

"""

  def __init__(self, disp, machine_number, overrides={}):
    self.k = CONTAINER_K
    self.k.update(overrides)
    self.disp = disp
    self.vncpass = None
    self.mounts = []
    # self.ionhelper: support for instanton cache user to run the actual
    # simulations to provide results to both cache and real user.
    if 'IONHELPER' in self.k:
      self.ionhelper = True
    else:
      self.ionhelper = False
    if disp < 1:
      # container ID 0 is the execution host itself so can't be used
      raise MaxwellError("Container ID must be at least 1")
    if disp > int(self.k["MAX_CONTAINERS"]) or disp < 1:
      raise MaxwellError("Container ID must be less than %d" % int(self.k["MAX_CONTAINERS"]))

    self.veid = disp + self.k["VZOFFSET"]
    self.vz_private_path = "%s/private/%d" % (self.k["VZ_PATH"], self.veid)
    self.vz_root_path = "%s/root/%d" % (self.k["VZ_PATH"], self.veid)

    # IPv4 address of the container = f(PRIVATE_NET, containerID, machine_number)
    # where machine_number defaults to something based on the IP address of the execution host
    # so that containers have different IP addresses across a fleet of execution hosts
    #
    # To avoid the Invalid MIT-MAGIC-COOKIE error, the IP address of containers must be unique
    # over an entire hub (not just over an execution host).
    # xauth list shows the cookies a user has.  A cookie in use can get overwritten by a new
    # one if there's an IP address collision
    #
    # PRIVATE_NET IPv4 has (at least) 2 bytes free.  If a fleet has no more than 64 execution hosts
    # the address space for each is 256/64 * 256 = 1024 -2 (for broadcast and "subnet")
    # so there's a maximum of 1022 containers/execution host
    #
    # automatic "machine_number" assignment
    # assumption: IPv4 is used, and the last byte of the IP address of execution hosts mod 64 is unique
    # (not necessarily true, one could be x.y.z.1 and another x.y.z.65)
    # if there's a collision, you'll need to manually set "machine_number" for one of them
    # or, set "machine_number" for all your execution hosts
    #
    if disp > 1022:
      # count starts at 1; 1023th container could be broadcast address
      log("container ID %d too high, xauth collisions possible" % disp)
    if (machine_number == 0):
      try:
        self.machine_number = int(socket.gethostbyname(socket.getfqdn()).split('.')[3])
      except StandardError:
        raise MaxwellError("machine_number not set and unable to derive one from IP address.")
      if (self.machine_number == 0):
        raise MaxwellError("unable to set machine_number")
    else:
      self.machine_number = machine_number
    if DEBUG:
      try:
        # virtual SSH doesn't setup a log file but uses this class to get the IP address
        # consider setting up a log file in /usr/bin/virtualssh_client
        log("DEBUG: machine number is %d" % machine_number)
      except AttributeError:
        pass

    # last byte = disp mod 256
    # second byte = int(disp/256) + (machine_number mod 64) *4
    # broadcast address if machine_number mod 64 = 63, disp/255 = 3 and last byte = 255
    digit = disp/256 + (self.machine_number % 64) *4
    self.veaddr = self.k["PRIVATE_NET"] % (digit % 256, disp % 256)

  def groups(self):
    """ find last user in /etc/passwd, which is the user owning the tool session, and get the groups of that account.
    HELPER accounts like "ionhelper" (see also self.k['HELPER']) have their /etc/passwd entries before the user's."""
    cmd = ["tail", "-n", '1', "/var/lib/vz/root/%s/etc/passwd" % veid]
    userline = subprocess.Popen(cmd, stdout=subprocess.PIPE).communicate()[0]
    userparts = userline.split(':')
    return make_User_account(userparts[0], self.k).groups()

  def __printvzstats(self, err):
    """Print statistics for an OpenVZ VPS."""
    f = open("/proc/vz/vestat")
    for line in f:
      arr = line.split()
      if len(arr) < 5:
        continue
      if arr[0] == "VEID":
        continue
      if DEBUG:
        log("DEBUG: Checking /proc/vz/vestat veid = %s" % arr[0])
      try:
        if int(arr[0]) == self.veid:
          # Since VEs are pre-created, this is NOT the real time.
          # Let the middleware host compute this time.
          #log("real %f\n" % (int(arr[4])/1000.0))
          err.write("user %f\n" % (int(arr[1])/1000.0))
          err.write("sys %f\n" % (int(arr[3])/1000.0))
          break
      except ValueError:
        log("can't convert to integer, continuing")
    f.close()
    flag_print = False
    for line in open("/proc/user_beancounters"):
      if line.find(":") > 0:
        parts = line.split(":")
        if parts[0].strip() == "Version":
          continue
        if flag_print:
          break
        if int(parts[0].strip()) == self.disp:
          flag_print = True
          err.write("resource                     held              maxheld              barrier                limit              failcnt\n")
          err.write(parts[1].strip() + "\n")
      else:
        if flag_print:
          err.write(line.strip() + "\n")

  def __etc_passwd(self, account, err):
    """Add user info to /etc/passwd, /etc/shadow and /etc/group, also modify /etc/sudoers.
    """
    etc_passwd = open(self.vz_root_path + "/etc/passwd", "a")
    etc_shadow = open(self.vz_root_path + "/etc/shadow", "a")
    etc_group = open(self.vz_root_path + "/etc/group", "a")

    if 'apps' in account.groups():
      # add the apps user and edit the sudoers file to allow su to apps
      apps_user = make_User_account('apps', self.k)
      err.write(apps_user.passwd_entry() + "\n")
      etc_passwd.write(apps_user.passwd_entry() + "\n")
      etc_shadow.write(apps_user.shadow_entry() + "\n")
      etc_sudoers = open(self.vz_root_path + "/etc/sudoers", 'a', 0440)
      etc_sudoers.write("%apps           ALL=NOPASSWD:/bin/su - apps\n")
      etc_sudoers.close()
      apps_groups = apps_user.groups()
    else:
      apps_groups = []

    # ionhelper user for instant-on cache management
    # assumes /var/ion/... exists in the template
    if self.ionhelper:
      err.write('ionhelper:x:199:199::/var/ion/:/bin/false\n')
      etc_passwd.write('ionhelper:x:199:199::/var/ion/:/bin/false\n')
      etc_shadow.write('ionhelper:*:17821:0:99999:7:::\n')
    elif 'HELPER' in self.k:
      helper_pwd = '%s:x:%s:%s::%s:/bin/false\n' % (self.k['HELPER'], self.k['HELPER_UID'], self.k['HELPER_GID'], self.k['HELPER_HOME'])
      err.write(helper_pwd)
      etc_passwd.write(helper_pwd)
      etc_shadow.write('%s:*:17821:0:99999:7:::\n' % self.k['HELPER'])

    # write the /etc/passwd entry of the user last so we can look that up in firewall_readd.py
    err.write(account.passwd_entry())
    err.write("\n")
    etc_passwd.write(account.passwd_entry())
    etc_passwd.write("\n")
    etc_shadow.write(account.shadow_entry())
    etc_shadow.write("\n")
    # Add CMS groups to the /etc/group file in the container/VEID...
    # normally CMS gids are above 500
    # gids below 500 could be system groups and should be left alone, except for fuse.  See later
    for g in account.group_pairs():
      # expecting to loop over something like this:  [['group10', 10], ['group11', 11]]
      gname = g[0]
      gid = g[1]
      if gid > 500:
        # copy all group info as is if gid > 500
        if gname in apps_groups:
          #  support su to apps.  Create all the groups that user apps belongs to, and add user apps and user helper if applicable
          if 'HELPER' in self.k:
            etc_group.write("%s:x:%d:%s,%s,%s\n" % (gname, gid, account.user, 'apps', self.k['HELPER']))
          else:
            etc_group.write("%s:x:%d:%s,%s,%s\n" % (gname, gid, account.user, 'apps', 'ionhelper'))
        else:
          if 'HELPER' in self.k:
            etc_group.write("%s:x:%d:%s,%s\n" % (gname, gid, account.user, self.k['HELPER']))
          else:
            etc_group.write("%s:x:%d:%s,%s\n" % (gname, gid, account.user, 'ionhelper'))

    if self.ionhelper:
      # add ionhelper group and put user in it
      etc_group.write("%s:x:%d:%s\n" % ('ionhelper', 199, account.user))
    if 'HELPER' in self.k:
      # add HELPER group and put user in it
      etc_group.write("%s:x:%s:%s\n" % (self.k['HELPER'], self.k['HELPER_GID'], account.user))
    etc_group.close()
    etc_passwd.close()
    etc_shadow.close()

  def update_resources(self, session):
    """
    Add contents to resources file for anonymous sessions
    Experimental anonymous session support.  Use at your own risk
    """
    homedir = self.k["HOME_DIR"]+"/"+'anonymous'
    rpath = "%s/%s/data/sessions/%s/resources" % (self.vz_root_path, homedir, session)
    try:
      rfile = open(rpath,"a+")
      # Read data from command line and write to file, until an empty string is found
      while 1:
        line = sys.stdin.readline()
        if line == "":
          break
        rfile.write(line)

      rfile.close()
    except OSError:
      raise MaxwellError("Unable to append to resource file.")

  def create_anonymous(self, session, params):
    """Add an "anonymous" user in the container"""
    homedir = self.k["HOME_DIR"]+"/"+'anonymous'
    args = ['/usr/sbin/vzctl', 'exec2', str(self.veid), 'adduser', '--uid', '1234']
    args += ['--disabled-password', '--home', homedir, '--gecos', '"anonymous user"', 'anonymous']
    subprocess.check_call(args)

    rdir = "%s%s/data/sessions/%s" % (self.vz_root_path, homedir, session)
    if VERBOSE:
      log("creating " + rdir)
    os.makedirs(rdir)
    os.chown(rdir,  1234, 1234)
    os.chown("%s%s/data/sessions" % (self.vz_root_path, homedir),  1234, 1234)
    os.chown("%s%s/data/" % (self.vz_root_path, homedir),  1234, 1234)
    rfile = open(rdir+"/resources", "w")
    rfile.write("sessionid %s\n" % session)
    rfile.write("results_directory %s/data/results/%s\n" % (homedir, session))
    os.fchown(rfile.fileno(), 1234, 1234)
    rfile.close()
    if VERBOSE:
      log("setup anonymous session directory and resources in session %s" % (session))
    if params is not None and params != "":
      import urllib2
      params_path = rdir + "/parameters.hz"
      pfile = open(params_path, "w")
      pfile.write(urllib2.unquote(params).decode("utf8"))

  def mount_paths(self, mountlist):
    """Given a list of paths, mount them in the container.  Called by maxwell_service when receiving the command mount_paths.
    This functionality is needed for the MyGeohub hydroshare connectivity via /srv/irods/external. Need to implement it for Docker too!"""
    for mount_pt in mountlist:
      log("mounting %s" % (mount_pt))
      # mount_pt must already exist
      if not os.path.exists(mount_pt):
        log("trying to mount '%s' but it doesn't exist" % mount_pt)
        continue
      if not os.path.exists(self.vz_root_path + mount_pt):
        args = ['/bin/mkdir', '-m', '0700', '-p', self.vz_root_path + mount_pt]
        p = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        p.communicate()
        if p.returncode != 0:
          raise MaxwellError("Could not create '%s'" % (self.vz_root_path + mount_pt))
      self.__root_mount(mount_pt, 'rw,noatime')

  def __setup_helper(self, helper, helper_home, helper_uid, helper_gid, session_id, runner_home, runner_name, err):
    """
    Helper account: local trusted system account supporting sudo operations on behalf of a tool session's owner
    setup resources for helper accounts which allow users "sudo" operations to be done to a trusted account
    helper account home must be local to container for security reasons, because user can write to the drivers directory
    the session user's home directory may be on NFS, don't use root to read from it
    """
    # create session directory in helper account's home
    sessiondir_helper = "%s/%s/data/sessions/%s" % (self.vz_root_path, helper_home, session_id)
    if DEBUG:
      log("creating session directory in helper account: %s" % sessiondir_helper)
    os.makedirs(sessiondir_helper, 0750)

    # copy/edit "resources" file from user to helper account's session directory
    # rewrite the results_directory line
    rpath_user = "%s/data/sessions/%s/resources" % (runner_home, session_id)
    rpath_helper = "%s/%s/data/sessions/%s/resources" % (self.vz_root_path, helper_home, session_id)
    # run copy as user to access NFS share
    # more or less like:
    # su pmeunier -s /bin/sh -c "cat /home/nanohub/pmeunier/data/sessions/20522/resources" \
    # > /vz/root/2/var/ion/data/sessions/20522/resources

    # open file for writing, unbuffered
    resfile = open(rpath_helper, 'w', 0)
    # rewrite the first 2 lines
    resfile.write("sessionid %s\n" % session_id)
    resfile.write("results_directory %s/runs\n" % helper_home)
    # read from original starting at line 3, as the user owning the tool session
    p = subprocess.Popen(['su', runner_name, '-s', '/bin/sh', '-c', 'tail -n +3 ' + rpath_user], stdout = resfile, stderr = err)
    p.communicate()

    drivers_dir = self.vz_root_path + helper_home + '/drivers'
    try:
      os.makedirs(drivers_dir, 0770)
    except OSError:
      # it's fine if it already exists, but set permissions
      os.chmod(drivers_dir, 0770)
    os.chmod(self.vz_root_path + '/' + helper_home, 0770)
    # finally make sure everything is owned by helper account
    os.system('chown -R %s:%s %s/%s' % (helper_uid, helper_gid, self.vz_root_path, helper_home))

  def __mergeauth_X11(self, username, err):
    """
    Add X11 information for container to helper's .Xauthority file
    Use the mergeauth script wrapper to xauth
    """
    count = 0
    # helper needs to access the display
    args = ['/usr/sbin/vzctl', 'exec2', str(self.veid), 'su', username, '-s', '/bin/sh', '-c', self.k["INTERNAL_PATH"] + 'mergeauth']
    if DEBUG:
      log("DEBUG: xauth: %s\n" % " ".join(args))
    while True:
      p = subprocess.Popen(args, stderr = err, stdout = err)
      p.communicate()
      if p.returncode == 0:
        break
      time.sleep(0.5)
      count = count+1
      if count > self.k["XAUTH_RETRIES"]:
        err.write("Unable to extract xauth cookie for %d\n" % self.veid)
        raise MaxwellError() # cleanup rules

  def __child_unix_command(self, user, session_id, timeout, command, params, sesslog_path):
    """child
      1. Setup the environment inside the container: permissions, password, group files,
      2. Setup the firewall rules on the host
      3. Setup X server authentication.  Call xauth so we're allowed to connect to the X server
      4. Invoke the command within the container
      5. Calculate time stats
      6. Restore the firewall rules
    """
    err = open(sesslog_path + ".err", 'a', 0)
    out = open(sesslog_path + ".out", 'a', 0)
    err.write("Starting command '%s' for '%s' with timeout '%s'\n" % (command, user, timeout))

    if user == "anonymous":
      if self.k["ANONYMOUS"]:
        # do not mount /home.
        # do not make any LDAP calls.
        account = User_account_anonymous(user)
      else:
        raise MaxwellError("Anonymous user session not supported")
    else:
      account = make_User_account(user, self.k)
      # mount script doesn't mount /home anymore
      self.__root_mount(account.homedir, "rw,nodev,nosuid,noatime")
      # Setup the environment inside the container: permissions, password, group files
      self.__etc_passwd(account, err)
      if VERBOSE:
        err.write("VERBOSE: wrote /etc/passwd\n")

      if self.ionhelper:
        # ionhelper in /var/ion
        # __setup_helper(helper, helper_home, helper_uid, helper_gid, session_id, runner_home, runner_name)
        self.__setup_helper('ion', '/var/ion', '199', '199', session_id, account.homedir, user, err)
      if 'HELPER' in self.k:
        self.__setup_helper(self.k['HELPER'], self.k['HELPER_HOME'], self.k['HELPER_UID'], self.k['HELPER_GID'], session_id, account.homedir, user, err)

    # Get a list of the supplementary groups...
    groups = account.groups()

    # groups like "fuse" may pre-exist inside containers with a different gid than on the hub
    # just appending a new line to /etc/group could create conflicting definitions
    # example: vhub.org.def:@define MW_CONTAINER_GROUPS '"fuse", "public"'
    # If  called with two non-option arguments, adduser will add an existing user to an existing group.
    # Assumes a Debian Linux container, will fail with RedHat because the command is named useradd instead
    if DEBUG:
      err.write("DEBUG: calling adduser\n")
    for defgroup in self.k["DEFAULT_GROUPS"]:
      p = subprocess.Popen(['/usr/sbin/vzctl', 'exec2', str(self.veid), 'adduser', user, defgroup], stdout = err, stderr = err)
      p.communicate()
      if self.ionhelper:
        p = subprocess.Popen(['/usr/sbin/vzctl', 'exec2', str(self.veid), 'adduser', 'ionhelper', defgroup], stdout = err, stderr = err)
        p.communicate()
      if 'HELPER' in self.k:
        p = subprocess.Popen(['/usr/sbin/vzctl', 'exec2', str(self.veid), 'adduser', self.k['HELPER'], defgroup], stdout = err, stderr = err)
        p.communicate()

    # 1b. /apps bind mount conditional on apps group membership
    # This requires xvnc to not be in /apps, but in /usr/sbin
    if 'apps' in groups:
      mount_opt = 'rw,acl,noatime'
      self.__root_mount("/apps", mount_opt)
      if user == 'apps':
        err.write("user is apps, not mounting home directory twice\n")
      else:
        err.write("%s is in apps group\n" % user)
        # mount home directory of user 'apps' to support su to apps
        apps_account = make_User_account('apps', self.k)
        self.__root_mount(apps_account.homedir, "rw,nodev,nosuid,noatime")
    else:
      if 'APPS_SUBDIR_MOUNTING' in self.k:
        if self.k["APPS_SUBDIR_MOUNTING"]:
          # Experimental, use at your own risk
          err.write("%s is NOT in apps group, figuring out which directories to mount\n" % user)
          self.__mount_subdirs_ro(err, user, "/apps", 'ro,acl,noatime')
        else:
          mount_opt = 'ro,acl,noatime'
          self.__root_mount("/apps", mount_opt)
      else:
        mount_opt = 'ro,acl,noatime'
        self.__root_mount("/apps", mount_opt)

    if 'APPS_SUBDIR_MOUNTING' in self.k:
      # Experimental, use at your own risk
      # mount /data subdirs
      if self.k["APPS_SUBDIR_MOUNTING"]:
        if os.path.exists('/data'):
          for s in os.listdir('/data'):
            self.__mount_subdirs(err, user, os.path.join('/data',s))

    # Username-based mounts
    # uses bind mounts from an already mounted filesystem, which has a directory for each user
    # the mounted directory is the username
    if self.k["USER_MOUNT"]:
      for mount_pt in self.k["USER_MOUNT_POINTS"]:
        log("mounting %s" % (mount_pt))
        # mount_pt must already exist
        if not os.path.exists(mount_pt):
          err.write("Mount point '%s' does not exist" % mount_pt)
          continue
        source_mount = mount_pt + user
        # check if source exists
        if not os.path.exists(source_mount):
          # create it as the user, not root
          args = ['/bin/su', user, '-c', "mkdir -m 0700 " + source_mount]
          p = subprocess.Popen(args, stderr = err, stdout = err)
          p.communicate()
          if p.returncode != 0:
            if VERBOSE:
              err.write("Warning: '%s' did not exist, could not create it as user '%s', so will not be mounted\n" % (source_mount, user))
            continue
        if not os.path.exists(self.vz_root_path + source_mount):
          args = ['/bin/mkdir', '-m', '0700', '-p', self.vz_root_path + source_mount]
          p = subprocess.Popen(args, stderr = err, stdout = err)
          p.communicate()
          if p.returncode != 0:
            raise MaxwellError("Could not create '%s'" % (self.vz_root_path + source_mount))
        self.__root_mount(source_mount, 'rw,noatime')

    # Mount projects based on user membership
    if self.k["PROJECT_MOUNT"]:
      for g in groups:
        if g[0:3] == "pr-":
          source_mount = self.k["PROJECT_PATH"] + g[3:]
          if not os.path.exists(source_mount):
            continue
          if not os.path.exists(self.vz_root_path + source_mount):
            args = ['/bin/mkdir', '-m', '0700', '-p', self.vz_root_path + source_mount]
            p = subprocess.Popen(args, stderr = err, stdout = err)
            p.communicate()
            if p.returncode != 0:
              raise MaxwellError("Could not create '%s'" % (self.vz_root_path + source_mount))
          self.__root_mount(source_mount, 'rw,noatime')

    # Mount public project file areas
    # https://mygeohub.org/support/ticket/1226
    # IMPORTANT:  path for this can be entirely different from path for regular project areas
    # compare "/srv/mygeohub/projects/" vs "/srv/irods/"
    # this section uses PROJECT_PUBLIC_PATH instead of PROJECT_PATH
    # mount read-only for users who do not belong to the corresponding projects
    # mount read-write for users who belong
    # public project file areas are:  /srv/mygeohub/projects/*/files/public
    # algorithm
    #   for each project group in which the user belongs
    #     if there's a public area, mount it r/w
    #   find projects that have "public" directories
    #     if already mounted, continue
    #     else mount read-only
    if self.k["PROJECT_PUBLIC_MOUNT"]:
      # mount read/write public areas for project members
      for g in groups:
        if g[0:3] == "pr-":
          source_mount = self.k["PROJECT_PUBLIC_PATH"] + g[3:] + "/files/public"
          if not os.path.exists(source_mount):
            continue
          if not os.path.exists(self.vz_root_path + source_mount):
            args = ['/bin/mkdir', '-m', '0700', '-p', self.vz_root_path + source_mount]
            p = subprocess.Popen(args, stderr = err, stdout = err)
            p.communicate()
            if p.returncode != 0:
              raise MaxwellError("Could not create '%s'" % (self.vz_root_path + source_mount))
          self.__root_mount(source_mount, 'rw,noatime')
      import glob
      listing = glob.glob(self.k["PROJECT_PUBLIC_PATH"]+ '/*/files/public')
      for pubpath in listing:
        mntpt = self.vz_root_path + pubpath
        if os.path.isdir(mntpt):
          continue
        args = ['/bin/mkdir', '-m', '0700', '-p', mntpt]
        p = subprocess.Popen(args, stderr = err, stdout = err)
        p.communicate()
        if p.returncode != 0:
          raise MaxwellError("Could not create '%s'" % (mntpt))
        self.__root_mount(pubpath, 'ro,noatime')

    # Use local storage for the session directory
    # implemented for cdmhub, to use SSDs on the execution host
    if self.k["LOCAL_SESSIONDIR"]:
      import shutil
      localhome_dir="/home/sessions/" + user
      local_dir = "/home/sessions/%s/%s" % (user, session_id)
      remote_dir = account.homedir + "/data/sessions/%s" % session_id
      # create local user home so operations can be made under the user account and not root
      # mv fails if the session directory is not empty
      # but this can happen only during testing because the same sessnum is used.
      # Better to not attempt rm -rf otherwise
      if False:
        args = ['/bin/rm', '-rf', localhome_dir]
        p = subprocess.Popen(args, stderr = err, stdout = err)
        p.communicate()
      args = ['mkdir', '-p', '-m', '0750', localhome_dir]
      p = subprocess.Popen(args, stderr = err, stdout = err)
      p.communicate()
      if p.returncode != 0:
        raise MaxwellError("Could not create '%s'" % (localhome_dir))
      os.chown(localhome_dir, account.uid, account.gid)

      # move the session directory to fast local storage
      args = ['/bin/su', user, '-c', 'mv %s %s' % (remote_dir, localhome_dir)]
      p = subprocess.Popen(args, stderr = err, stdout = err)
      p.communicate()
      if p.returncode != 0:
        raise MaxwellError("Could not move '%s' to '%s'" % (remote_dir, localhome_dir))

      # mount it
      # mount: only root can do that
      # mounting as root will follow symlinks so would be unsafe, can mount over important mount points like /apps
      # create a symlink from remote_dir to local_dir, as the user
      # so the openvz mount script has to mount /home/sessions inside the container!
      args = ['/bin/su', user, '-c', "ln -s %s %s" % (local_dir, remote_dir) ]
      p = subprocess.Popen(args, stderr = err, stdout = err)
      p.communicate()
      if p.returncode != 0:
        raise MaxwellError("Could not symlink %s to '%s'" % (remote_dir, local_dir))

    # Deprecated -- SSHFS-based user mounts -- Deprecated
    if self.k["SSHFS_MOUNT"]:
      # create an SSH connection for each container
      for mount_pair in self.k["SSHFS_MOUNT_POINTS"]:
        # array of remote, local info
        remote_path = mount_pair[0] + user
        container_path = self.vz_root_path + mount_pair[1] + user
        manage_path = mount_pair[1] + user
        log("mounting %s at %s" % (remote_path, container_path))
        if not os.path.exists(manage_path):
          # create it as the user, not root
          args = ['/bin/su', user, '-c', "mkdir -m 0700 " + manage_path]
          p = subprocess.Popen(args, stderr = err, stdout = err)
          p.communicate()
          if p.returncode != 0:
            # possible race condition if user starts two sessions quickly for the first time
            raise MaxwellError("Could not create '%s'" % (manage_path))
        if not os.path.exists(container_path):
          args = ['/bin/mkdir', '-m', '0700', '-p', container_path]
          p = subprocess.Popen(args, stderr = err, stdout = err)
          p.communicate()
          if p.returncode != 0:
            raise MaxwellError("Could not create '%s'" % (self.vz_root_path + source_mount))
        args = ['/usr/bin/ssh', '-o', 'intr', '-o', 'sync_read', '-o', 'IdentityFile=%s' % self.k["SSHFS_MOUNT_KEY"], '-o', 'allow_other', remote_path, container_path]
        p = subprocess.Popen(args, stderr = err, stdout = err)
        p.communicate()

    # 2. Setup the firewall rules on the host
    if VERBOSE:
      err.write("VERBOSE: setting up firewall rules\n")
    self.firewall_by_group(groups, 'add')

    # Wrap the following in a "try" to reverse the iptables state in case of an exception
    try:
      # 3. Setup X server authentication
      self.__mergeauth_X11(user, err)
      if self.ionhelper:
        self.__mergeauth_X11('ionhelper', err)
      if 'HELPER' in self.k:
        self.__mergeauth_X11(self.k['HELPER'], err)

      # 4. Invoke the command within the container
      # Actual application start!
      # In the command below, use "time" to get the runtime of the command.
      # The user and sys cputimes will be inaccurate, but will be overridden
      # by printvzstats().  We use it to get the clock time of the command.
      #
      # Note: time is a built-in shell command, there is no separate binary installed
      # so the whole thing has to be passed to a shell for interpretation...
      # when shell=True, args needs to be a string for "time" to be interpreted as the built-in
      #
      # Also, the environment needs to be passed inside the container.  Setting env= in the
      # subprocess call only sets the environment for the vzctl command.
      #
      # In addition, the whole "su..." command has to be passed inside quotes, otherwise it fails.
      #  Perhaps some of the switches get interpreted by an earlier command than intended? time?
      env_cmd = " ".join(account.env(session_id, timeout, params) + ["DISPLAY=\"%s:0.0\"" % (self.veaddr)])
      try:
        env_cmd += " " + self.k["EXTRA_ENV_CMD"]
      except KeyError:
        pass
      args = ['/usr/sbin/vzctl', 'exec2', str(self.veid), 'su', user, '-s', '/bin/dash', '-c',
         '\"cd; %s %s\"' % (env_cmd, command)]
      if 'HELPER' in self.k:
        # expecting command like '/apps/jupyter/r16/middleware/invoke'
        toolname = command.split('/')[2]
        if toolname in self.k['HELPER_TOOLS']:
          if DEBUG:
            log("DEBUG: toolname %s matches %s\n" % (toolname, " ".join(self.k['HELPER_TOOLS'])))
          env_cmd = " ".join(make_User_account(self.k['HELPER'], self.k).env(session_id, timeout, params) + ["DISPLAY=\"%s:0.0\"" % (self.veaddr)])
          try:
            env_cmd += " " + self.k["EXTRA_ENV_CMD"]
          except KeyError:
            pass
          args = ['/usr/sbin/vzctl', 'exec2', str(self.veid), 'su', self.k['HELPER'], '-s', '/bin/dash', '-c',
         '\"cd; %s %s\"' % (env_cmd, command)]
      if DEBUG:
        log("DEBUG: command is %s\n" % " ".join(args))
      start_time = time.time()
      # subprocess.call(args)
      # Python docs: "The data read is buffered in memory, so do not use this method if the data size is large or unlimited."
      # Problem:  If the tool is misbehaving and has GBs of output, then root processes start
      # consuming GBs of memory!  Todo: find alternative to calling subprocess.communicate while capturing output
      p = subprocess.Popen(args, stderr = err, stdout = out)
      p.communicate()
      end_time = time.time()

      # 5. Calculate time stats
      if VERBOSE:
        err.write("Processing stats\n")
      err.write("real\t%f\n" % (end_time - start_time))
      self.__printvzstats(err)
      # everything went OK
      err.write("Exit_Status: 0\n")
      err.close()
    except StandardError, exc:
      # cleanup iptable rules
      err.write("tool session failed due to exception:'%s'\n" % exc)
      err.write("Exit_Status: 2\n")
      err.close()

    # 6. Leave firewall cleanup to maxwell_service, to be done after the container has stopped

    os._exit(0)

  def invoke_unix_command(self, user, session_id, timeout, command, params, sesslog_path):
    """Start a tool in the container.
     Child will invoke the command.
     Parent will log the exit status.  When we'll return, other things will happen (notify).
     When we are called, the log file has been closed and we're a dissociated process.
     Stdout and stderr have been redirected to files, so we use that for logging.
    user: string
    session_id: string (int+letter)
    timeout: int
    command: string
    """
    try:
      pid = os.fork()
    except OSError, ose:
      log("unable to fork: '%s', exiting" % ose)
      sys.exit(1)

    if pid == 0:
      self.__child_unix_command(user, session_id, timeout, command, params, sesslog_path)
    # parent
    try:
      log("Waiting for %d" % pid)
      os.waitpid(pid, 0)
    except OSError:
      pass
    return 0

  def screenshot(self, user, sessionid):
    """Support display of session screenshots for app UI.  On error, do not produce an exception."""
    account = make_User_account(user, self.k)
    destination = "%s/data/sessions/%s/screenshot.png" % (account.homedir, sessionid)
    if os.path.isdir("%s/data/sessions/%s" % (account.homedir, sessionid)):
      vz_env = account.env(sessionid, 8000, False)
      env_cmd = " ".join(vz_env + ["DISPLAY=\"%s:0.0\"" % (self.veaddr)])
      command = "/usr/bin/screenshot %s" % destination
      args = ['/usr/sbin/vzctl', 'exec2', str(self.veid), 'su', user, '-s', '/bin/dash', '-c',
        '\"cd; %s %s\"' % (env_cmd, command)]
      p = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
      log_subprocess(p)

  def __root_mount(self, point, perm):
    """Mount a directory to make it available to OpenVZ containers."""
    mntpt = self.vz_root_path + point
    if os.path.isdir(mntpt):
      try:
        os.rmdir(mntpt)
      except OSError, exc:
        log("exception:'%s'\n" % exc)
        raise MaxwellError("'%s' already exists and is probably already mounted.  Giving up."
          % mntpt)
    try:
      saved_umask = os.umask(0)
      os.makedirs(mntpt, 0755)
    finally:
      os.umask(saved_umask)
    if VERBOSE:
      log("Created %s" % mntpt)

    # -n: Mount without writing in /etc/mtab.
    args = ["/bin/mount", "-n", "--bind", point, '-o', perm, mntpt]
    p = subprocess.Popen(args, stderr=subprocess.PIPE, stdout=subprocess.PIPE)
    log_subprocess(p)
    if p.returncode != 0:
      try:
        os.rmdir(mntpt)
      except OSError:
        pass
      raise MaxwellError("Could not mount %s in '%d'" % (point, self.veid))

  def __mount_subdirs(self, err, user, topdir):
    """Bind mount subdirectories of topdir if they are accessible to user.  Determine if they should be read-only or r/w"""
    if os.path.isdir(topdir):
      for o in os.listdir(topdir):
        if o == 'lost+found':
          continue
        fp=os.path.join(topdir,o)
        if os.path.isdir(fp):
          accessible = False
          # try to ls the directory, as the user
          args = ['/bin/su', user, '-c', "ls " + fp]
          with open(os.devnull, 'w') as devnull:
            returncode = subprocess.call(['/bin/su', user, '-c', "ls " + fp], stdout=devnull, stderr=devnull)
          if returncode == 0:
            accessible = True
            permissions = 'ro,noatime'
          # try to touch a file as the user, with "su"
          with open(os.devnull, 'w') as devnull:
            returncode = subprocess.call(['/bin/su', user, '-c', "touch %s/.test" % fp], stdout=devnull, stderr=devnull)
          if returncode == 0:
            # we can write
            accessible = True
            permissions = 'rw,noatime'
          if accessible:
            err.write("mounting %s with permissions %s; " %(fp, permissions))
            self.__root_mount(fp, permissions)
          else:
            err.write("cannot mount %s" %(fp))

  def __mount_subdirs_ro(self, err, user, topdir, permissions):
    """Bind mount subdirectories of topdir with provided permissions if they are readable by the user.
    The permissions are intended to contain 'ro', as it only tests read access.  Faster than __mount_subdirs"""
    if os.path.isdir(topdir):
      for o in os.listdir(topdir):
        if o == 'lost+found':
          continue
        fp=os.path.join(topdir,o)
        if os.path.isdir(fp):
          accessible = False
          # try to ls the directory, as the user
          args = ['/bin/su', user, '-c', "ls " + fp]
          with open(os.devnull, 'w') as devnull:
            returncode = subprocess.call(['/bin/su', user, '-c', "ls " + fp], stdout=devnull, stderr=devnull)
          if returncode == 0:
            accessible = True
          if accessible:
            err.write("mounting %s with permissions %s; " %(fp, permissions))
            self.__root_mount(fp, permissions)
          else:
            err.write("cannot mount %s" %(fp))

  def firewall_by_group(self, groups, operation='add'):
    """groups is an array of groups the user belongs to
    FW_GROUP_MAP is an array of ["group_name", net_cidr, portmin, portmax]
    where net_cidr is something like 128.46.19.160/32
    """
    rule_start = [self.k["FW_CHAIN"], '-i', 'venet0', '-s', self.veaddr]
    for g in self.k["FW_GROUP_MAP"]:
      if g[0] in groups:
        (net_cidr, portmin, portmax) = g[1:]
        if net_cidr == "":
          # block access to hubzero networks due to firewall rule exceptions
          hostpart = ['!', '-d', self.k["MW_PROTECTED_NETWORK"]]
        else:
          # hostpart = ['-d', socket.gethostbyname(host)]
          hostpart = ['-d', net_cidr]
        if portmin == 0:
          # all ports
          portpart = []
        else:
          if portmax == 0:
            # single port
            portpart = ['-p', 'tcp', '--dport', '%s' % portmin]
          else:
            portpart = ['-p', 'tcp', '--dport', '%s:%s' % (portmin, portmax)]
        fwd_rule = rule_start + hostpart + portpart + ['-j', 'ACCEPT']
        if DEBUG:
          log('DEBUG: ' + operation + ' ' + ' '.join(fwd_rule))
        if operation == 'add':
          i=0
          while i < 3:
            # retry adding firewall rules.  Log final failure and keep going.
            try:
              # insert due to RETURN or DROP rules
              subprocess.check_call(['/sbin/iptables', '--wait', '100', '-A'] + fwd_rule)
              i = 3
            except subprocess.CalledProcessError:
              log('Warning: unable to add firewall rule ' + ' '.join(fwd_rule))
              i += 1
              if i < 3:
                time.sleep(1)
        else:
          # delete rule if present
          # iptables -C uses the logic for -D (delete) so it returns 0 if the rule exists, 1 if it doesn't exist
          args = ['/sbin/iptables', '--wait', '100', '-t', 'filter', '-C'] + fwd_rule
          p = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
          p.communicate()
          if p.returncode ==0:
            # retry deleting firewall rules, but eventually ignore the error
            i=0
            while i < 10:
              try:
                subprocess.check_call(['/sbin/iptables', '--wait', '100', '-D'] + fwd_rule)
                i = 10
              except subprocess.CalledProcessError:
                log('Warning: unable to delete firewall rule ' + ' '.join(fwd_rule))
                i += 1
                if i < 10:
                  time.sleep(1)

  def firewall_cleanup(self):
    """Get all rules for the AUTO_FORWARD chain.  Delete the ones that include the IP address of this container.
    This is more robust than calling firewall_by_group with the delete operation, because group memberships
    could have changed since the session start."""

    p = subprocess.Popen(['/sbin/iptables', '--wait', '100', '-S', self.k["FW_CHAIN"]], stderr=subprocess.PIPE, stdout=subprocess.PIPE)
    (stdout, stderr) = p.communicate()
    if p.returncode != 0:
      log("Unable to list rules in %s chain" % self.k["FW_CHAIN"])
      return
    lines= stdout.split('\n')
    for line in lines:
      if line.find(self.veaddr + '/32') != -1:
        args = ['/sbin/iptables', '--wait', '100', '-D'] + line.split()[1:]
        i = 0
        while i < 10:
          try:
            subprocess.check_call(args)
            i = 10
          except subprocess.CalledProcessError:
            log('Warning: unable to delete firewall rule: ' + ' '.join(args))
            i += 1
            if i < 10:
              time.sleep(1)

  def set_ipaddress(self):
    """Set IP address of container"""
    status = os.system("vzctl set %d --ipadd %s" % (self.veid, self.veaddr))
    if status != 0:
      raise MaxwellError("Bad status for vzctl set %d --ipadd %s: %d" %
        (self.veid,self.veaddr,status))

  def umount(self):
    """unmount container directories"""
    self.__openVZ_umount(self.vz_root_path)
    self.__openVZ_umount(self.vz_private_path)

  def __openVZ_umount(self, fs_path):
    """If given path exists, call ctid.umount for that container
    The ctid.umount script is part of the shutdown process of a container.  This indicates
    an unclean shutdown.
    """
    if os.path.exists(fs_path):
      # tell openVZ to unmount that container's file system
      # internally, that umount script doesn't use abolute paths so we need
      # to set the PATH
      v_env = {"VEID" : str(self.veid), "PATH": "/bin:/usr/bin"}
      args = ["%s/%s" % (self.k["VZ_CONF_PATH"], self.k["OVZ_SESSION_UMOUNT"])]
      p = subprocess.Popen(args, stderr=subprocess.PIPE, stdout=subprocess.PIPE, env = v_env)
      log_subprocess(p)
      if p.returncode != 0:
        raise MaxwellError("Could not unmount container '%d'" % (self.veid))
      if self.k["APPS_READONLY"]:
        # we mount /apps rw or ro depending on group membership of user
        try:
          os.rmdir(fs_path + '/apps')
        except OSError:
          pass
      try:
        os.rmdir(fs_path)
      except OSError:
        pass
      if os.path.exists(fs_path):
        if os.path.exists('/usr/sbin/lsof'):
          p = subprocess.Popen(['/usr/sbin/lsof', fs_path], stderr=subprocess.PIPE, stdout=subprocess.PIPE)
          (stdout, stderr) = p.communicate()
          if p.returncode == 0:
            log("lsof %s: %s" % (fs_path, stdout))
          else:
            log("Unable to call lsof on %s" % fs_path)
        raise MaxwellError("'%s' still exists" % fs_path)

  def create_xstartup(self):
    """xstartup is a file we create to make VNC happy.  RUN_DIR is something like '/usr/lib/mw'.
    VNC is started inside containers, and /usr is mounted inside."""
    x_path = RUN_DIR + "/xstartup"
    try:
      lock_stat = os.lstat(x_path)
    except OSError:
      # does not exist
      try:
        xstartup = os.open(x_path, os.O_CREAT | os.O_WRONLY | os.O_NOFOLLOW, 0700)
        os.write(xstartup, "#!/bin/bash\n")
        os.close(xstartup)
        lock_stat = os.lstat(x_path)
      except OSError:
        raise MaxwellError("Unable to create '%s'." % x_path)

    # check that it has the expected permissions and ownership
    # check that we are the owner and that others can't write
    if lock_stat[stat.ST_MODE] & stat.S_IWOTH:
      raise MaxwellError("'%s' has unsafe permissions.  Remove write permissions for others"
        % x_path)

    usr_id = lock_stat[stat.ST_UID]
    if usr_id != os.geteuid():
      raise MaxwellError("'%s' has incorrect owner: %s" % (x_path, usr_id))

  def read_passwd(self):
    """VNC password is 8 bytes, we want the version encrypted for VNC, not the
    encoded version for web use
    """
    self.vncpass = sys.stdin.read(8)
    return

  def stunnel(self):
    """We handle tunnels and forwards here, to make the inside of containers visible to the outside.
    Containers are mapped to port ranges using the dispnum (a.k.a. CTID a.k.a. veid).
    """
    in_port = self.veid + self.k["STUNNEL_PORTS"] # e.g., 4000 + display
    remote = '%s:%d' % (self.veaddr, self.k["PORTBASE"])
    # kill anything listening on the stunnel port; should have been killed when stopping container
    # to avoid race condition with TCP_WAIT state.
    p = subprocess.Popen(['fuser', '-n', 'tcp', '%d' % in_port, '-k', '-9'], stderr=subprocess.PIPE, stdout=subprocess.PIPE)
    # retrieve but ignore error message which is generated if there was no process to kill
    (stdout, stderr) = p.communicate()
    if p.returncode == 0:
      # a process was killed, wait a bit for TCP_WAIT state to clear
      time.sleep(1)
    if self.k["TUNNEL_MODE"][:5] == 'socat':
      args = ["socat"]
      args.append("OPENSSL-LISTEN:%d,cert=%s,fork,verify=0" % (in_port, self.k["PEM_PATH"]))
      args.append("tcp4:%s" % remote)
      if len(self.k["TUNNEL_MODE"]) > 5:
        args += ["-d", "-d", "-d"]
      if DEBUG:
        log('DEBUG: ' + " ".join(args))
      process = subprocess.Popen(
        args,
        stdout = subprocess.PIPE,
        stderr = subprocess.PIPE
      )
      (stdout, stderr) = process.communicate()
      if process.returncode != 0:
        log("Can't start ssl socat (it's probably already running): %s" % stderr)

      if VERBOSE:
        log("forwarder started: %s" % stdout)

    else:
      if self.k["TUNNEL_MODE"] == 'stunnel4':
        # stunnel4 will read configuration from a pipe.
        # reading from stdin causes the first connection to fail
        # we're going to create a pipe and fork because Popen is too limited
        # file descriptors r, w for reading and writing
        r, w = os.pipe()
        processid = os.fork()
        if processid:
          # This is the parent process
          # note: the parent can't exec otherwise the webserver will think the container is done starting
          os.close(r)
          w = os.fdopen(w, 'w')
          w.write("cert = %s\n" % self.k["PEM_PATH"])
          w.write("accept = %d\n" % in_port)
          w.write("connect = %s\n" % remote)
          w.write("debug = 3\n")
          # RedHat has FIPS version, this errors out on Debian.
          w.write("fips=no\n")
          w.write("output=/var/log/stunnel\n")
          w.write("[stunnel3]\n")
          w.close()
          (pid, status) = os.wait()
          if status == 0 and VERBOSE:
            log("VERBOSE: stunnel4 os.wait pid=%d, status=%d." %(pid, status))
          else:
            raise MaxwellError("stunnel error! pid=%d, status=%d." %(pid, status))
        else:
          # This is the child process, will read from r
          os.close(w)
          os.execl("/usr/bin/stunnel", "/usr/bin/stunnel", "-fd", "%d" % r)
          raise MaxwellError("unable to execute /usr/bin/stunnel")
      else:
        # stunnel3 for Debian
        # status = os.system("stunnel -d %d -r %s:%d -p %s" %
        #         (4000+self.disp, self.veaddr, 5000,  self.k["PEM_PATH"]))
        # -d: daemon mode
        # -r [host:]port    connect to remote service
        # , '-D', '7' to increase debug level to 7
        args = ["stunnel", '-D', '4', '-d', str(in_port), '-r', remote, '-p',  self.k["PEM_PATH"]]
        log(" ".join(args))
        subprocess.check_call(args)

      if VERBOSE:
        log("VERBOSE: stunnel started for %d" % self.veid)

    # Start a forwarder to make the display look external.
    # This is only for backward-compatibility.
    #os.system("socat tcp4-listen:%d,fork,reuseaddr,linger=0 tcp4:%s:5000 > /dev/null 2>&1 < /dev/null &" % (5000+ self.veid, self.veaddr))

  def delete_confs(self):
    """Get rid of these links if they exist."""
    for ext in ['conf', 'mount', 'umount']:
      try:
        os.unlink("%s/%d.%s" % (self.k["VZ_CONF_PATH"], self.veid, ext))
      except EnvironmentError:
        if DEBUG:
          log("DEBUG: File %s/%d.%s was already deleted or missing" %
            (self.k["VZ_CONF_PATH"], self.veid, ext))

    # stop quotas if they are already running
    # vzquota off 194
    # vzquota drop 194
    # this operation can fail if the operations are still ongoing; that's fine.
    subprocess.call(['/usr/sbin/vzquota', 'off', '%d' % self.veid], stderr=open('/dev/null', 'w'))
    # drop safely removes the quota file -- this file can cause problems,
    # e.g., when container template is changed
    subprocess.call(['/usr/sbin/vzquota', 'drop', '%d' % self.veid], stderr=open('/dev/null', 'w'))

  def create_confs(self):
    """In directory /etc/vz/conf, create the symlinks to the mount, unmount scripts and
    the configuration file.  The mount script is called when starting the VE (container).
    Ignore errors if symlinks already exist."""
    try:
      os.symlink("%s/%s" % (self.k["VZ_CONF_PATH"], self.k["OVZ_SESSION_CONF"]),
        "%s/%d.conf" % (self.k["VZ_CONF_PATH"], self.veid))
    except EnvironmentError:
      pass
    try:
      os.symlink("%s/%s" % (self.k["VZ_CONF_PATH"], self.k["OVZ_SESSION_MOUNT"]),
        "%s/%d.mount" % (self.k["VZ_CONF_PATH"], self.veid))
    except EnvironmentError:
      pass
    try:
      os.symlink("%s/%s" % (self.k["VZ_CONF_PATH"], self.k["OVZ_SESSION_UMOUNT"]),
        "%s/%d.umount" % (self.k["VZ_CONF_PATH"], self.veid))
    except EnvironmentError:
      pass

  def start_filexfer(self):
    """Start a socat forwarder for filexfer.  We never kill it.
       If we can't start one, that means there's already one running."""
    port = self.veid + self.k["FILEXFER_PORTS"]
    os.system("socat tcp4-listen:%d,fork,reuseaddr,linger=0 tcp4:%s:%d > /dev/null 2>&1 &"
      % (port, self.veaddr, port))

  def start(self, geom):
    """ start a container.
    Have /usr be symlink at the beginning (from setup_template), then remove it to put a mount
    Setup a lock directory that will be erased by the start process when it's done
    this functionality appears to be duplicated by the .mount scripts in /etc/vz/conf.
    We wait for the lock to be removed, to indicate that the mount script has finished.  This is
    not an access lock.
    """
    lock_dir = "%s/lock/mount.%d.lock" % (self.k["VZ_PATH"], self.veid)
    if not os.path.exists(lock_dir):
      os.mkdir(lock_dir)
      if VERBOSE:
        log("Created %s" % lock_dir)
    else:
      if VERBOSE:
        log("Already existed: %s" % lock_dir)
    start_time = time.time()
    if DEBUG:
      # extremely verbose
      args = ["vzctl", "--verbose", "start", str(self.veid)]
    else:
      args = ["vzctl", "start", str(self.veid)]
    p = subprocess.Popen(args, stderr=subprocess.PIPE, stdout=subprocess.PIPE)
    log_subprocess(p)
    if p.returncode != 0:
      raise MaxwellError("Can't start container '%d'" % (self.veid))
    # now sleep until the lock is gone
    while os.path.exists(lock_dir):
      time.sleep(1)
      if time.time() - start_time > 60:
        raise MaxwellError("Timed out waiting for container to start")
    end_time = time.time()
    # log how long we waited
    log ("vzctl start time: %f seconds" % (end_time - start_time))

    # replace the symlink with a mount
    os.unlink(self.vz_private_path + "/usr")

    # If vz/root mount is done after the call to start, without a symlink in place, we get:
    # bash: line 318: awk: command not found
    # ERROR: Can't change file /etc/hosts

    # If vz/root mount is done before the call to start, we get:
    # error 32
    # mount: special device /vz/root/257/.root/usr does not exist
    #
    # if we try private instead of root, we get:
    # mount: special device /vz/private/261/.root/usr does not exist
    #
    # symlink can't be left alone due to bug in gcc;  mounting is needed.
    #
    # check mount point exists or create it
    usr_mnt = self.vz_root_path + "/usr"
    if not os.path.exists(usr_mnt):
      os.mkdir(usr_mnt)

    # mount --bind olddir newdir
    # -n: Mount without writing in /etc/mtab.
    args = ["/bin/mount", "-n", "--bind", self.vz_root_path + "/.root/usr", usr_mnt]
    p = subprocess.Popen(args, stderr=subprocess.PIPE, stdout=subprocess.PIPE)
    log_subprocess(p)
    if p.returncode != 0:
      raise MaxwellError("Can't bind mount .root/usr in '%d'" % (self.veid))

    if VERBOSE:
      log("vzctl exec2 %d %s %s 0 %s" %\
          (self.veid, self.k["INTERNAL_PATH"]+'startxvnc', self.veaddr, geom))
    args = ["vzctl", "exec2", str(self.veid)]
    args += [self.k["INTERNAL_PATH"] + 'startxvnc', self.veaddr, '0', geom]
    process = subprocess.Popen(
      args,
      stdin = subprocess.PIPE,
      stdout = subprocess.PIPE,
      stderr = subprocess.PIPE
    )
    (stdout, stderr) = process.communicate(self.vncpass)
    if process.returncode != 0:
      raise MaxwellError("Unable to start internal Xvnc server: %s%s" %(stdout, stderr))
    elif VERBOSE:
      log(stdout)
      end_time3 = time.time()
      # log how long we waited
      log ("startxvnc call took: %f seconds" % (end_time3 - end_time))

  def resize(self, geometry):
    """Change XVNC geometry on the fly, after the container has started."""
    (width, height) = geometry.split('x')
    args = ["vzctl", "exec2", str(self.veid), self.k["INTERNAL_PATH"] + 'hzvncresize']
    args += ['-a', '/var/run/Xvnc/passwd-%s:0' % self.veaddr, width, height ]
    process = subprocess.Popen(
      args,
      stdin = subprocess.PIPE,
      stdout = subprocess.PIPE,
      stderr = subprocess.PIPE
    )
    (stdout, stderr) = process.communicate()
    if process.returncode != 0:
      raise MaxwellError("Unable to change Xvnc geometry: %s%s" %(stdout, stderr))

  def setup_template(self):
    """Setup symlinks and mount points for OpenVZ container.
        usr is a temporary symlink.  It needs to be a symlink initially and later
        we replace it with a mount point.  The mount point is needed
        because some versions of gcc don't work with a symlink.
        Doing the mount point without first doing the symlink sometimes generates these:
      # bash: line 318: awk: command not found
      #  ERROR: Can't change file /etc/hosts
    """
    os.makedirs(self.vz_root_path)
    os.chmod(self.vz_root_path, 0755)
    if VERBOSE:
      log("created directory " + self.vz_root_path)
    os.makedirs(self.vz_private_path)
    os.chmod(self.vz_private_path, 0755)
    if VERBOSE:
      log("created directory " + self.vz_private_path)
    # for link in ['bin', 'lib', 'sbin', 'lib64', 'usr']: equivalent to "template copy -a" method
    # also link 'emul', 'lib32' to support 32-bit binaries
    for link in ['bin', 'lib', 'sbin', 'emul', 'lib32', 'lib64', 'usr', 'opt']:
      path = self.vz_private_path + "/" + link
      # treating usr as a mount point can generate these errors:
      # bash: line 318: awk: command not found
      #  ERROR: Can't change file /etc/hosts
      if os.path.lexists(path):
        log("%s already exists!" % path)
      else:
        os.symlink(".root/%s" % link, path)

    for vzdir in ['.root', 'home', 'mnt', 'proc', 'sys']:
      os.mkdir(self.vz_private_path + "/" + vzdir)
      os.chmod(self.vz_private_path + "/" + vzdir, 0755)

    if VERBOSE:
      log("template setup")

  def __delete_root(self):
    """Delete container root path.  If it doesn't exist, it will fail the directory test"""
    if os.path.isdir(self.vz_root_path):
      os.rmdir(self.vz_root_path)
      # what if directory isn't empty?

  def __log_status(self):
    """log the status of the container if in VERBOSE mode"""
    if VERBOSE:
      log(self.get_status())

  def get_status(self):
    """Obtain the status of this container"""
    args = ['vzctl', 'status', str(self.veid)]
    p = subprocess.Popen(args, stderr=subprocess.PIPE, stdout=subprocess.PIPE)
    (stdout, stderr) = p.communicate()
    if p.returncode != 0:
      raise MaxwellError("Can't get status of container '%d': %s" % (self.veid, stderr))
    return str(stdout)

  def __halt(self):
    """ Hard halt for all processes in the container.  Does not wait for anything."""
    args = ['vzctl', 'exec', str(self.veid), 'halt', '-nf']
    process = subprocess.Popen(
      args,
      stdout = subprocess.PIPE,
      stderr = subprocess.PIPE
    )
    (stdout, stderr) = process.communicate()
    if process.returncode == 14:
      # Container configuration file vps.conf(5) not found
      # try to fix it otherwise VE can't be shut down!
      self.create_confs()
      subprocess.check_call(args)
    if process.returncode != 0:
      log("Unable to halt VE: %s%s" %(stdout, stderr))

  def wait_unlock(self):
    """Allow the caller to know when OpenVZ is done starting or stopping a container"""
    attempt = 0
    while os.path.exists("/vz/lock/%d" % self.veid):
      time.sleep(10)
      attempt += 1
      if attempt > 50:
        raise MaxwellError("Timeout waiting for lock to be released on VE: %d" %(self.veid))

  def stop(self):
    """# Stops  and  unmounts  a  container.
    Use '--fast' option so OpenVZ leaves mounts alone for our .umount script to handle.
    Otherwise OpenVZ makes /vz/root/VEID read-only"""
    args = ['vzctl', 'stop', str(self.veid), '--fast']
    process = subprocess.Popen(args, stderr=subprocess.PIPE, stdout=subprocess.PIPE)
    (stdout, stderr) = process.communicate()
    if process.returncode == 14:
      # Container configuration file vps.conf(5) not found
      # try to fix it otherwise VE can't be shut down!
      self.create_confs()
      p = subprocess.Popen(args, stderr=subprocess.PIPE, stdout=subprocess.PIPE)
      log_subprocess(p)
      return
    if process.returncode == 3:
      # Error in waitpid(12345): No child processes
      # try to umount instead
      self.umount()
      if VERBOSE:
        log("ignoring error 3 in 'vzctl stop' : %s%s exit code %d"
        % (stdout, stderr, process.returncode))
    elif process.returncode != 0:
      raise MaxwellError("'vzctl stop' output: %s%s exit code %d"
        % (stdout, stderr, process.returncode))
    else:
      log(stdout)
      # delete the directory only if successful
      # We needed to wait until now because "vzctl stop" expects it.
      if os.path.isdir(self.vz_root_path):
        os.rmdir(self.vz_root_path)

  def __vzproccount(self):
    """Read /proc/vz/veinfo to get the number of processes in the container.
    Each line presents a running Container in the <CT_ID> <reserved> <number_of_processes> <IP_address> ... format:
    """
    try:
      f = open("/proc/vz/veinfo")
    except EnvironmentError:
      log("vzproccount: can't open veinfo.")
      # possibly stopped
      return 0
    while 1:
      line = f.readline()
      if line == "":
        if False:
          log("End of file /proc/vz/veinfo.")
        return 0
      arr = line.split()
      # expecting something like "         29     0     2      10.26.0.29"
      if len(arr) != 4:
        continue
      if arr[0] == str(self.veid):
        #log("vzproccount is %s" % arr[2])
        try:
          return int(arr[2])
        except ValueError:
          return 0

  def stop_submit_local(self):
    """check for submit --local.  If it's there, give it SIGINT and wait.
    Called by maxwell_service before calling killall."""
    # kill stunnel and log any errors as it should still be running
    in_port = self.veid + self.k["STUNNEL_PORTS"] # e.g., 4000 + display
    p = subprocess.Popen(['fuser', '-n', 'tcp', '%d' % in_port, '-k', '-9'], stderr=subprocess.PIPE, stdout=subprocess.PIPE)
    log_subprocess(p)
    check_submit_args = ['vzctl', 'exec', str(self.veid), '/usr/bin/pgrep', '-f', '\"submit --local\"']
    p = subprocess.Popen(check_submit_args, stderr=subprocess.PIPE, stdout=subprocess.PIPE)
    (stdout, stderr) = p.communicate()
    rc = p.returncode
    if DEBUG:
      if len(stderr) > 0:
        log("DEBUG: error finding submit pid: " + stderr.rstrip())
      if len(stdout) > 0:
        log("DEBUG: submit pid was: " + stdout.rstrip())
    if rc == 0:
      log("Signal submit --local to exit")
      pkill_args = ['vzctl', 'exec', str(self.veid), 'pkill', '-15', '-f', '\"submit --local\"']
      p = subprocess.Popen(pkill_args, stderr=subprocess.PIPE, stdout=subprocess.PIPE)
      (stdout, stderr) = p.communicate()
      attempt = 0
      p = subprocess.Popen(check_submit_args, stderr=subprocess.PIPE, stdout=subprocess.PIPE)
      # discard output
      p.communicate()
      rc = p.returncode
      while rc == 0 and (attempt < 10):
        attempt += 1
        time.sleep(20) # give time for submit to exit
        p = subprocess.Popen(check_submit_args, stderr=subprocess.PIPE, stdout=subprocess.PIPE)
        # discard output
        p.communicate()
        rc = p.returncode
      if attempt > 9:
        log("submit --local didn't exit!")
    elif DEBUG:
      log("DEBUG stop_submit_local: exit code %d.  Running processes:" % rc)
      args= ['vzctl', 'exec', str(self.veid), 'ps aux']
      p = subprocess.Popen(args, stderr=subprocess.PIPE, stdout=subprocess.PIPE)
      log_subprocess(p)

  def killall(self):
    """Kill processes other than init running in the VPS.
    Start with milder signals.  Wait as long as the number of processes goes down.
    More recent kernels count kthreadd and khelper processes that are unkillable.

    Called by maxwell_service in 2 cases: after a tool is done running, and when receiving the stopvnc command."""
    status = self.get_status()
    if status.find("running") == -1:
      if DEBUG:
        log("DEBUG: killall: container was not running")
      # container isn't running, make sure root and private areas are unmounted.
      self.umount()

    for sig in [1, 2, 15, 9]:
      pcount = self.__vzproccount()
      # send all signals to avoid issues with unmounting
      if pcount == 0:
        break
      log("Killing %d processes in veid %d with signal %d" % (self.__vzproccount(), self.veid, sig))
      # -1 indicates all processes except the kill process itself and init.
      args = ['vzctl', 'exec', str(self.veid), 'kill -%d -1' % sig]
      p = subprocess.Popen(args, stderr=subprocess.PIPE, stdout=subprocess.PIPE)
      log_subprocess(p)
      # Wait as long as the number of processes keeps going down
      ccount = 0
      tries = 0
      while ccount < pcount and tries < 100:
        time.sleep(2) # give time for processes to exit, otherwise we try again too early
        pcount = ccount
        ccount = self.__vzproccount()
        tries += 1
        # log("attempt %d, signal %d, count is %d" % (tries, sig, ccount))
      if tries == 100:
        log("timeout waiting for processes to exit from signal %d" % (sig))
    # wait because in some edge cases the container shuts down
    attempt = 0
    while self.__vzproccount() > 1 and (attempt < 5):
      if DEBUG:
        log("DEBUG: killall: final wait")
      attempt += 1
      time.sleep(1)

class ContainerVZ7(Container):
  """Modified for OpenVZ 7 compatibility"""
  def __init__(self, disp, machine_number, overrides={}):
    # OpenVZ 7: Seems to have 2 folder hierarchies it can work with.
    # one mounts /vz/private/veid/fs to /vz/root/veid, and has more folders and files under /vz/private/veid/
    # the other maintains compatibility with /vz/private/veid being mounted directly
    # in this class we use the compatibility option
    # if we were converting to the new format we'd do:
    # self.vz_private_path = "%s/private/%d/fs" % (self.k["VZ_PATH"], self.veid)
    # but that's not enough.  Not sure what is the trigger between vzctl using one or the other
    Container.__init__(self, disp, machine_number, overrides)

  def create_xstartup(self):
    """LINT.
    This function created an "xstartup" file for VNC.  Does not appear needed anymore.
    """
    pass

  def delete_confs(self):
    """Get rid of these links if they exist."""
    for ext in ['conf', 'mount', 'umount']:
      try:
        os.unlink("%s/%d.%s" % (self.k["VZ_CONF_PATH"], self.veid, ext))
      except EnvironmentError:
        if DEBUG:
          log("File %s/%d.%s was already deleted or missing" %
            (self.k["VZ_CONF_PATH"], self.veid, ext))
    # OpenVZ 7 has withdrawn filesystem quota support from legacy simfs, vzquota calls removed

  def create_confs(self):
    """In directory /etc/vz/conf, create a symlink to the configuration file.
    Under OpenVZ 7, we don't create symlinks for the mount and .umount scripts
    because vps.mount and vps.umount are called, regardless of container ID. """
    try:
      os.symlink("%s/%s" % (self.k["VZ_CONF_PATH"], self.k["OVZ_SESSION_CONF"]),
        "%s/%d.conf" % (self.k["VZ_CONF_PATH"], self.veid))
    except EnvironmentError:
      raise MaxwellError("Unable to create OpenVZ symlinks")

class ContainerDocker(Container):
  """Modified to use Docker.
  We run 2 docker containers per tool session.
  One runs services like VNC, the other runs the tool.  The advantage of doing it this way is we can uncouple the VNC server from the tool -- i.e., use different templates, possibly running different OS versions.  The services could use more up-to-date packages, so vulnerability patching could be done without needing to upgrade tools.  Container names would be <self.veid>-services and <self.veid>-tool.  The call to start a standby session will start the <self.veid>-services container.  We'll keep only one standby session when using Docker.
  """

  def __init__(self, disp, machine_number, overrides={}):
    # machine_number: ignored (lint)
    self.k = CONTAINER_K
    self.k.update(overrides)
    self.disp = disp
    self.veid = disp
    self.vncpass = None
    # mounts:  array to record list of bind mounts to make when starting container
    self.mounts = []
    if 'IONHELPER' in self.k:
      self.ionhelper = True
    else:
      self.ionhelper = False
    if disp < 1:
      raise MaxwellError("Container ID must be at least 1")
    #
    # we use /22 networks
    if disp > 1021:
      # container count starts at 1;  we want container #1 to use the .1 address
      # 1023th IP is broadcast address (3*256 + 255)
      # 1022th IP is the Docker bridge's IP (3*256 + 254) because
      raise MaxwellError("container ID %d too high, maximum is 1021" % disp)

    # tools network doesn't offer any services, by default doesn't connect to internet
    self.toolnet_name = 'toolnet'

    # service network provides X server at known IP address for stunnel to connect to it
    # without IP collision in .Xauthority file
    self.servicenet_name = 'servicenet'
    # service net CIDR provided by self.k["PRIVATE_NET"] and machine_number calculation

    self.toolnet_CIDR = self.k["DOCKERTOOL_NET"] + '/22'
    gwbytes = self.k["DOCKERTOOL_NET"].split('.')
    gateway = gwbytes[0:2] + [str(int(gwbytes[2]) + 3), '254']
    self.toolnet_gateway = ".".join(gateway)
    IPa = gwbytes[0:2] + [str(int(gwbytes[2]) + disp/256), str(disp %256)]
    self.tool_container_IP = ".".join(IPa)

    self.servicenet_CIDR = self.k["DOCKERSERVICE_NET"] + '/22'
    gwbytes = self.k["DOCKERSERVICE_NET"].split('.')
    gateway = gwbytes[0:2] + [str(int(gwbytes[2]) + 3), '254']
    self.servicenet_gateway = ".".join(gateway)
    IPa = gwbytes[0:2] + [str(int(gwbytes[2]) + disp/256), str(disp %256)]
    self.services_container_IP = ".".join(IPa)

    # for inherited stunnel function
    self.veaddr = self.services_container_IP

  def resize(self, geometry):
    """Change XVNC geometry on the fly, after the container has started."""
    (width, height) = geometry.split('x')
    args = ["docker", "exec"]
    args += ['-e', 'DISPLAY=%s:0' % self.services_container_IP]
    args += ['%d.services' % self.veid]
    args += [self.k["INTERNAL_PATH"] + 'hzvncresize', '-a', '/var/run/Xvnc/passwd-%s:0' % self.services_container_IP, width, height ]
    process = subprocess.Popen(
      args,
      stdin = subprocess.PIPE,
      stdout = subprocess.PIPE,
      stderr = subprocess.PIPE
    )
    (stdout, stderr) = process.communicate()
    if process.returncode != 0:
      if DEBUG:
        log(" ".join(args))
      raise MaxwellError("Unable to change Xvnc geometry: %s%s" %(stdout, stderr))
    elif VERBOSE:
      log(stdout)

  def screenshot(self, user, sessionid):
    """Support display of session screenshots for app UI.  On error, do not produce an exception."""
    account = make_User_account(user, self.k)
    destination = "%s/data/sessions/%s/screenshot.png" % (account.homedir, sessionid)
    if os.path.isdir("%s/data/sessions/%s" % (account.homedir, sessionid)):
      args = ["docker", "exec"]
      args += ['-e', 'DISPLAY=%s:0' % self.services_container_IP]
      args += ['--user', user, '%d.tool' % self.veid]
      args += ["/usr/bin/screenshot", destination]
      p = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
      log_subprocess(p)

  def create_xstartup(self):
    """LINT.  This function created an "xstartup" file for VNC.  Does not appear needed anymore.
    """
    pass

  def mount_paths(self, mountlist):
    """Given a list of paths, mount them in the container.  Called by maxwell_service when receiving the command mount_paths.
    This functionality is needed for the MyGeohub hydroshare connectivity via /srv/irods/external.
    Docker implementation: store the information on the mount points until we can start the container.
    This container instance is preserved during the call to maxwell_service, which received all
     the information it needed with the "startxapp" command."""
    for mount_pt in mountlist:
      log("registering mount point %s" % (mount_pt))
      # mount_pt must already exist
      if not os.path.exists(mount_pt):
        log("trying to mount '%s' but it doesn't exist" % mount_pt)
        continue
      # docker: noatime isn't a valid mount option
      self.mounts += [[mount_pt, '']]

  def groups(self):
    """ find last user in /etc/passwd, and get the groups"""
    cmd = ['docker', 'exec', '%d.tool' % self.veid, "tail", "-n", '1', "/etc/passwd"]
    userline = subprocess.Popen(cmd, stdout=subprocess.PIPE).communicate()[0]
    userparts = userline.split(':')
    return make_User_account(userparts[0], self.k).groups()

  def __printDockerStats(self, err):
    """At end of session, get cpu stats and write it to sessnum.err.
    Docker provides cpu time info in cgroups filesystem
    collect it and write out using a format similar to the one used for OpenVZ (see __printvzstats):
    Docker units are in 1/100 of seconds so divide by 100.0 instead of 1000.0
    """
    #
    # tool container
    #
    # get full length Docker container ID
    # "docker ps -aqf 'name=24.tool'" returns abbreviated name, we need the full name for the filesystem path
    tool_uuid = ContainerDocker.__get_container_UUID('%d.tool' % self.veid)
    (tool_user_time, tool_system_time) = ContainerDocker.__parse_cpuacct(tool_uuid)
    err.write("tool user time %f\n" % (tool_user_time/100.0))
    err.write("tool sys time %f\n" % (tool_system_time/100.0))
    #
    # Services container
    #
    service_uuid = ContainerDocker.__get_container_UUID('%d.services' % self.veid)
    (service_user_time, service_system_time) = ContainerDocker.__parse_cpuacct(service_uuid)
    err.write("service user time %f\n" % (service_user_time/100.0))
    err.write("service sys time %f\n" % (service_system_time/100.0))
    #
    # Overall
    #
    err.write("Total time for tools and services\n")
    err.write("user %f\n" % ((tool_user_time + service_user_time)/100.0))
    err.write("sys %f\n" % ((tool_system_time + service_system_time)/100.0))

  def update_resources(self, session):
    """Add contents to resources file for anonymous sessions"""
    homedir = self.k["HOME_DIR"]+"/"+'anonymous'
    # needs to be converted for Docker
    pass

  def create_anonymous(self, session, params):
    """Add an "anonymous" user in the container"""
    homedir = self.k["HOME_DIR"]+"/"+'anonymous'
    args = ['/usr/sbin/vzctl', 'exec2', str(self.veid), 'adduser', '--uid', '1234']
    args += ['--disabled-password', '--home', homedir, '--gecos', '"anonymous user"', 'anonymous']
    # needs to be converted for Docker
    pass

  def set_ipaddress(self):
    """IP address of container already set at container start time"""
    pass

  def __root_mount(self, point, perm):
    """Build list of mount points to use when creating container.  "perm" is intended to be
    used as is in the mount command option '-o' """
    self.mounts += [[point, perm]]

  def get_status(self):
    """Obtain the status of the services Docker container
    command:
    docker ps --filter "name=^13.services" --format "{{.Names}}"
    expected output if container exists:
    13.services
    return code of "docker ps" is 0 regardless of whether container exists or not
    The caret in the filter is needed otherwise it will also return '113.services'.
    return values from this function fake OpenVZ output.
    """
    args = ['/usr/bin/docker', 'ps', '--filter', 'name=^%d.services' % self.veid, '--format', '{{.Names}}']
    p = subprocess.Popen(args, stderr=subprocess.PIPE, stdout=subprocess.PIPE)
    (stdout, stderr) = p.communicate()
    if p.returncode != 0:
      raise MaxwellError("Can't get status of container '%d': %s" % (self.veid, stderr))
    if stdout.find('%d.services' % self.veid) == 0:
      return 'running'
    else:
      conf_flag_path = "/var/log/mw-service/conf_flag_%d" % self.veid
      if os.path.exists(conf_flag_path):
        return 'down'
      else:
        return 'deleted'

  def umount(self):
    """Remove old Docker volumes.
    We try to start all containers with the --rm option so they are automatically removed (cleaned up) after execution.
    Note: this function is called also during startup in case of an unclean situation."""
    # remove old VNC password volume
    args = ["docker", "volume", 'rm', str(self.veid)+ '.Xvnc']
    p = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    p.communicate()
    if p.returncode == 0 and DEBUG:
      log("DEBUG: Deleted old %d.Xvnc volume" % self.veid)

  def setup_template(self):
    """Docker:
    i. Create volume to store X server authentication information.
    ii. Check that the container images exist and if not, build them
    """

    FNULL = open(os.devnull, 'w')

    # docker volume create 13.Xvnc
    args = ["docker", "volume", 'create', str(self.veid)+ '.Xvnc']
    p = subprocess.Popen(args, stdout=FNULL, stderr=FNULL)
    p.communicate()
    if p.returncode != 0:
      raise MaxwellError("Unable to create %d.Xvnc volume" % self.veid)
    elif VERBOSE:
      log("VERBOSE: Created %d.Xvnc volume" % self.veid)

    # docker build tigervnc:latest
    # check that the reply has a line with "tigervnc" or build it
    args = ['docker', 'inspect', 'hubzero_services']
    p = subprocess.Popen(args, stdout=FNULL, stderr=FNULL)
    p.communicate()
    if p.returncode != 0:
      # cd /etc/mw-service/docker;  docker build -f Dockerfile.TigerVNC --tag tigervnc .
      args = ['docker', 'build', '-f', 'Dockerfile.hubzero_services', '--tag', 'hubzero_services', '.']
      p = subprocess.Popen(args, cwd='/etc/mw-service/docker/hubzero_services', stdout=subprocess.PIPE, stderr=subprocess.PIPE)
      p.communicate()
      if p.returncode != 0:
        raise MaxwellError("Unable to build hubzero_services image")
      else:
        log("Built hubzero_services image")
    elif DEBUG:
      log("DEBUG: no action necessary for hubzero_services image")

    # docker image vncpasswd:latest
    # check that the reply has a line with "vncpasswd" or build it
    # cd /etc/mw-service/docker/hubzero_vncpasswd;  docker build -f Dockerfile.hubzero_vncpasswd --tag vncpasswd .
    args = ['docker', 'inspect', 'hubzero_vncpasswd']
    p = subprocess.Popen(args, stdout=FNULL, stderr=FNULL)
    p.communicate()
    if p.returncode != 0:
      args = ['docker', 'build', '-f', 'Dockerfile.hubzero_vncpasswd', '--tag', 'hubzero_vncpasswd', '.']
      p = subprocess.Popen(args, cwd='/etc/mw-service/docker/hubzero_vncpasswd', stdout=subprocess.PIPE, stderr=subprocess.PIPE)
      p.communicate()
      if p.returncode != 0:
        raise MaxwellError("Unable to build hubzero_vncpasswd image")
      else:
        log("Built hubzero_vncpasswd image")
    elif DEBUG:
      log("DEBUG: no action necessary for hubzero_vncpasswd image")

    # docker image for tools
    # check that the reply has a line with the tool image's name, or build it
    # cd /etc/mw-service/docker;  docker build -f Dockerfile.vncpasswd --tag vncpasswd .
    args = ['docker', 'inspect', 'hubzero_tools']
    p = subprocess.Popen(args, stdout=FNULL, stderr=FNULL)
    p.communicate()
    if p.returncode != 0:
      args = ['docker', 'build', '-f', 'Dockerfile.hubzero_tools', '--tag', 'hubzero_tools', '.']
      p = subprocess.Popen(args, cwd='/etc/mw-service/docker/hubzero_tools', stdout=subprocess.PIPE, stderr=subprocess.PIPE)
      p.communicate()
      if p.returncode != 0:
        raise MaxwellError("Unable to build hubzero_tools image")
      else:
        log("Built hubzero_tools image")
    elif DEBUG:
      log("DEBUG: no action necessary for hubzero_tools image")

    # Toolnet has masquerade, servicenet doesn't
    ContainerDocker.__check_or_create_network(self.toolnet_name, self.toolnet_CIDR, self.toolnet_gateway, True)
    ContainerDocker.__check_or_create_network(self.servicenet_name, self.servicenet_CIDR, self.servicenet_gateway, False)

    # Allow X server access
    # connections from tool to service container port 6000
    # rely on the VNC password for security, as any tool could try to use any X server.
    # possible hardening: create container-specific rules instead.
    # need to insert the rules because the last rule of chain DOCKER-USER is a RETURN/DROP
    # -I DOCKER-USER -i br_toolnet -o br_servicenet -p tcp -m tcp --dport 6000 -j ACCEPT
    ContainerDocker.__check_or_add_iptables('DOCKER-USER -i br_toolnet -o br_servicenet -p tcp -m tcp --dport 6000 -j ACCEPT', '-I')

    # hzvncproxyd-ws connections to Xvnc server
    # support both models, where the proxy is running in a docker0 container and where it's running directly on the host
    # 1. Allow host to service containers port 5000 (Xvnc server)
    # because there's a bridge and host has a NIC (gateway) directly on the service network, it goes through the OUTPUT chain
    # insert because there's a DROP statement on the OUTPUT chain
    ContainerDocker.__check_or_add_iptables('OUTPUT -s %s -o br_servicenet -p tcp -m tcp --dport 5000 -j ACCEPT' \
      % (self.servicenet_gateway), '-I')
    # 2. Allow docker0 network to connect to port 5000 on service containers
    ContainerDocker.__check_or_add_iptables('DOCKER-USER -i docker0 -o br_servicenet -p tcp -m tcp --dport 5000 -j ACCEPT', '-I')

    # Some tools need to connect to the internet for various reasons
    # use conntrack to support MASQUERADE from tool container
    # insert because there's a RETURN statement at the end
    ContainerDocker.__check_or_add_iptables('DOCKER-USER -o br_toolnet -m conntrack --ctstate RELATED,ESTABLISHED -j ACCEPT', '-I')
    # all open tool containers would also have: ContainerDocker.__check_or_add_iptables('DOCKER-USER -i br_toolnet -j ACCEPT', '-I')

    # POSTROUTING MASQUERADE for tool network
    prt = 'POSTROUTING -s %s -o %s -p %s -j MASQUERADE'
    ContainerDocker.__check_or_add_iptables((prt %(self.toolnet_CIDR, self.k["PUBLIC_INT"], 'tcp')) + ' --to-ports 10000-50000', '-A', 'nat')
    ContainerDocker.__check_or_add_iptables((prt %(self.toolnet_CIDR, self.k["PUBLIC_INT"], 'udp')) + ' --to-ports 10000-50000', '-A', 'nat')
    ContainerDocker.__check_or_add_iptables((prt %(self.toolnet_CIDR, self.k["PUBLIC_INT"], 'icmp')), '-A', 'nat')

  def create_confs(self):
    """Use a directory lock as a flag indicating that we're preparing to use, or have used, a container/display"""
    conf_flag_path = "/var/log/mw-service/conf_flag_%d" % self.veid
    if not os.path.exists(conf_flag_path):
      try:
        os.mkdir(conf_flag_path)
      except OSError:
        pass

  def delete_confs(self):
    """remove directory lock, to indicate that container/display was cleaned up, e.g., firewall rules and at jobs were removed.
    This is useful to quickly verify if invoke_unix_command has done its cleanup at the end of a tool session."""
    conf_flag_path = "/var/log/mw-service/conf_flag_%d" % self.veid
    if os.path.exists(conf_flag_path):
      try:
        os.rmdir(conf_flag_path)
      except OSError:
        pass

  def start(self, geom):
    """X server and other services. Start a container called veid.services
    (veid = ctid = display+offset) at the IP address
    we determined during class initialization to avoid xauth conflicts.
    Use the self.servicenet_name network.
    1. Setup the VNC password storage area
    2. Start TigerVNC using that password and geom
    """
    start_time = time.time()

    # kill any old service container that might still be running
    args = ["docker", "kill", '%d.services' % self.veid]
    p = subprocess.Popen(args, stdin = subprocess.PIPE, stderr=subprocess.PIPE, stdout=subprocess.PIPE)
    (stdout, stderr) = p.communicate()

    # make sure there is no tools container yet -- kill it if it exists
    args = ['docker', 'kill', str(self.veid)+ '.tool']
    process = subprocess.Popen(args, stdout = subprocess.PIPE, stderr = subprocess.PIPE)
    process.communicate()

    # store the password
    args = ["docker", "run", '-i', '--name', '%d.services' % self.veid ]
    # mount Xvnc volume
    args += ['-v', '%d.Xvnc:/var/run/Xvnc' % self.veid]
    # autoclean container after execution
    args += ['--rm']
    # vncpasswd image has ENTRYPOINT ["dd", "count=1", "bs=8"];  built from Dockerfile.vncpasswd
    args += ['hubzero_vncpasswd', 'of=/var/run/Xvnc/passwd-%s:0' % self.services_container_IP]
    p = subprocess.Popen(args, stdin = subprocess.PIPE, stderr=subprocess.PIPE, stdout=subprocess.PIPE)
    (stdout, stderr) = p.communicate(self.vncpass)
    if p.returncode != 0:
      raise MaxwellError("Unable to store VNC password: %s%s" %(stdout, stderr))

    # Start the VNC server
    args = ["docker", "run", '-d', '--name', '%d.services' % self.veid ]
    args += ['--cpu-shares', '65536']
    # mount Xvnc volume
    args += ['-v', '%d.Xvnc:/var/run/Xvnc' % self.veid]
    # autoclean container after execution
    args += ['--rm']

    # DOCKER_XSHM: Share X11 domain socket and enable shared memory transport for X11 (display 0)
    #              This is a work in progress as xauth still needs to be setup for this
    #              to actually be used. *njk*
    if 'DOCKER_XSHM' in self.k and self.k['DOCKER_XSHM']:
        args += ['-v', '%d.X11:/tmp/.X11-unix' % self.veid]
        args += ['--ipc=shareable']

    # connect to servicenet
    args += ['--network', self.servicenet_name, '--ip', self.services_container_IP]
    # tigervnc image has ENTRYPOINT ["/usr/bin/startxvnc"];  this script is different from the OpenVZ one
    # execs Xtigervnc after building fontpath and creating Xauthority-related files
    args += ['hubzero_services', self.services_container_IP, '0', geom]

    # WIP: The idea here is to be able to use standard VNC servers rather than only customized
    #      hubzero ones. Further work on this requires changes to the client to use standard
    #      resize functions and detect when hubzero extensions are available. The goal is to
    #      use turbovnc which has enhanced support for VirtualGL. This proof of concept worked
    #      with the client brute force modified to not use any hubzero extensions. *njk*
    if 0:
        args = ["docker"]
        args += ['run']
        args += ['-d']
        args += ['--name', '%d.services' % self.veid ]
        args += ['--rm']
        args += ['-v', '%d.Xvnc:/var/run/Xvnc' % self.veid]
        args += ['-v', '%d.X11:/tmp/.X11-unix' % self.veid]
        args += ['--ipc=shareable']
        args += ['--network', self.servicenet_name]
        args += ['--ip', self.services_container_IP]
        args += ['-e', 'XVNC_DISPLAY=0']
        args += ['-e', 'XVNC_GEOMETRY=%s' % geom]
        args += ['-e', 'XVNC_RFBPORT=%d' % (5000 + self.veid) ]
        args += ['-e', 'XVNC_RFBAUTH=/var/run/Xvnc/passwd-%s:%d' % (self.services_container_IP, 0)]
        args += ['-e', 'XVNC_FONTPATH=']
        args += ['-e', 'XVNC_DESKTOP=hub']
        args += ['-e', 'XVNC_DEPTH=24']
        args += ['-e', 'XVNC_WAIT=3000']
        args += ['-e', 'XVNC_COOKIE=']
        args += ['-e', 'XVNC_PN=accept']
        args += ['-e', 'XVNC_LISTEN=tcp']
        args += ['-e', 'XVNC_HOSTNAME=%s' % self.services_container_IP ]
        args += ['turbovnc:1.60']

    p = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    (stdout, stderr) = p.communicate(self.vncpass)
    if p.returncode != 0:
      raise MaxwellError("Unable to start VNC server: %s%s" %(stdout, stderr))

    # WIP: This was some additional debugging useful when launching alternative vnc
    #      servers (see above). Keeping here for future reference. *njk*
    if 0:
        log("DEBUG: launched VNC server: %s%s" %(stdout, stderr))
        args = ["docker"]
        args += ['logs']
        args += ['%d.services' % self.veid]
        p = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        (stdout, stderr) = p.communicate(self.vncpass)
        log("DEBUG: launched VNC server log: %s%s" %(stdout, stderr))

  def start_filexfer(self):
    """Start a socat forwarder for filexfer.  We never kill it.
       If we can't start one, that means there's already one running.
       Docker difference:  connect to tool container, not service container.
       Consider instead adding a firewall rule like:
       iptables -t nat -A PREROUTING -i extif -p tcp --dport port --to-destination self.tool_container_IP:port
    """
    port = self.veid + self.k["FILEXFER_PORTS"]
    os.system("socat tcp4-listen:%d,fork,reuseaddr,linger=0 tcp4:%s:%d > /dev/null 2>&1 &"
      % (port, self.tool_container_IP, port))

  def __is_running(self, name):
    """
    name: e.g., ctid.services or ctid.tools
    return True if container with that tag is running
    """
    args = ['/usr/bin/docker', 'ps', '--filter', 'name=^%s' % name, '--format', '{{.Names}}']
    p = subprocess.Popen(args, stderr=subprocess.PIPE, stdout=subprocess.PIPE)
    (stdout, stderr) = p.communicate()
    if p.returncode != 0:
      raise MaxwellError("Can't get status of Docker container %s: %s" % (name, stderr))
    return stdout.find(name) == 0

  def stop_submit_local(self):
    """check for submit --local.  If it's there, give it SIGTERM and wait.  Note that submit handles SIGINT and SIGTERM identically.
    Called by maxwell_service."""
    if not self.__is_running('%d.tool' % self.veid):
      return
    # HZ classic:  kill stunnel so browser window returns quickly to dashboard
    # onescienceplace: no stunnel needed, hzvncproxy runs directly on the execution host
    in_port = self.veid + self.k["STUNNEL_PORTS"] # e.g., 4000 + display
    p = subprocess.Popen(['fuser', '-n', 'tcp', '%d' % in_port, '-k', '-9'], stderr=subprocess.PIPE, stdout=subprocess.PIPE)
    (stdout, stderr) = p.communicate()
    if DEBUG:
      log("DEBUG: killing stunnel exit code %d output %s:%s" % (p.returncode, stdout.rstrip('\n'), stderr.rstrip('\n')))
    elif p.returncode != 0:
      log("Error killing stunnel: %d output %s:%s" % (p.returncode, stdout.rstrip('\n'), stderr.rstrip('\n')))

    # check if submit --local is running.  Use "docker top" with options of regular Linux "ps" command to filter on "submit" command
    # the pid field is required otherwise the error "Error response from daemon: Couldn't find PID field in ps output" is printed
    # ps option --no-headers also prints the error "Error response from daemon: Couldn't find PID field in ps output"
    # ps option --no-trunc doesn't work ("Error response from daemon: ps: error: unknown gnu long option")
    # Expected: "Error response from daemon: ps: exit status 1" if there's no submit command running, otherwise exit status 0
    # "docker top" may or may not return an error when there are no results.  Handle both cases.
    check_submit_args = ['docker', 'container', 'top', str(self.veid) + '.tool', '-o pid,command']
    pcheck = subprocess.Popen(check_submit_args, stdout=subprocess.PIPE,stderr = subprocess.PIPE)
    (stdout, stderr) = pcheck.communicate()
    if pcheck.returncode == 0:
      # parse ps output
      found_submit_local = False
      for line in stdout.splitlines():
        # skip header line
        if line.find('PID') != -1:
          continue
        # distinguish between submit invocations, look for --local argument on same line
        if line.find('submit') != -1 and line.find('--local') != -1:
          # found it!
          found_submit_local = True
          break
      if not found_submit_local:
        # no error and no submit
        if DEBUG:
          log("DEBUG: submit --local not found, and no error in docker top")
        return
      if DEBUG:
        log("DEBUG: sending SIGTERM to submit --local")
      signal_args = ['docker', 'exec', str(self.veid) + '.tool', 'pkill', '-15', '-f', 'submit --local']
      rc = subprocess.call(signal_args)
      if rc != 0:
        log("Error sending SIGTERM to submit --local")
        return
      attempt = 0
      rc = 0
      while rc == 0 and (attempt < 10):
        attempt += 1
        if DEBUG:
          log("DEBUG: waiting for submit --local to exit")
        time.sleep(20) # give time for submit to exit
        # don't use subprocess.call to avoid printing garbage in logs
        pwait = subprocess.Popen(check_submit_args, stdout = None, stderr = None)
        pwait.communicate()
        rc = pwait.returncode
      if attempt > 9:
        log("submit --local didn't exit!")
    elif DEBUG:
      # no submit found, we're done
      log("DEBUG: submit is not running: exit code %d" % pcheck.returncode)
      args= ['docker', 'container', 'top', str(self.veid) + '.tool', 'f -N -u root']
      p = subprocess.Popen(args, stderr=subprocess.PIPE,stdout=subprocess.PIPE)
      (stdout, stderr) = p.communicate()
      log('DEBUG: ps output of tool container ' + str(self.veid) + ':\n' + stdout.rstrip('\n'))

  def __proccount(self):
    # get process count
    args= ['docker', 'stats', '--no-stream', '--format', '{{.PIDs}}', str(self.veid) + '.tool']
    p = subprocess.Popen(args, stderr=subprocess.PIPE,stdout=subprocess.PIPE)
    (stdout, stderr) = p.communicate()
    try:
      return int(stdout)
    except ValueError, e:
      return 0

  def killall(self):
    """Kill the docker exec call that started the tool.  Wait for the process to be gone.
    Killing the entire container would prevent collecting stats; we want the container to keep running.
    The container was started with a different command, and so will keep running if we just stop the tool.
    In practice this amounts to killing all non-root processes, which is easier and more reliable than other methods.
    Calling kill from within the container behaves wonkily.  Killing the "docker exec" command directly is also
    not reliable.
    """

    if VERBOSE:
      log("VERBOSE: killing tool in container " + str(self.veid)+ '.tool')

    # identify the processes to be killed, then kill them
    args = ['docker', 'container', 'top', str(self.veid) + '.tool', '-N', '-u', 'root']

    if self.__is_running('%d.tool' % self.veid):
      for sig in [15, 9]:
        p = subprocess.Popen(args, stdout=subprocess.PIPE,stderr = subprocess.PIPE)
        (stdout, stderr) = p.communicate()
        pids = []
        for line in stdout.splitlines():
          pid = line.split()[0]
          if pid == "PID":
            continue
          pids += [str(pid)]
        if len(pids) == 0:
          break
        if VERBOSE:
          log("killing with signal %d processes %s" % (sig, " ".join(pids)))
        args = ['kill', '-s', str(sig)] + pids
        p = subprocess.Popen(args, stdout=subprocess.PIPE,stderr = subprocess.PIPE)
        p.communicate()
        time.sleep(10)
    # don't stop the services container here because that could cause race conditions, and also we likely still want to collect stats about it

  def __docker_kill(self, name):
    args = ['docker', 'kill', name]
    p = subprocess.Popen(args, stdout = subprocess.PIPE, stderr = subprocess.PIPE)
    (stdout, stderr) = p.communicate()
    if p.returncode != 0:
      log("Unable to stop Docker container '%s': %s%s exit code %d"
        % (name, stdout, stderr, p.returncode))
    elif VERBOSE:
      log("VERBOSE: killed Docker container " + name)

  def stop(self):
    """
    Stop containers used for a display.
    kill tool container first, X server last so the tools don't log errors because the X server went away first.
    Do not collect stats.
    """
    # tools
    if self.__is_running('%d.tool' % self.veid):
      self.__docker_kill('%d.tool' % self.veid)

    # services
    if self.__is_running('%d.services' % self.veid):
      self.__docker_kill('%d.services' % self.veid)

    # remove VNC password volume
    self.umount()

  def firewall_by_group(self, groups, operation='add'):
    """Differs from OpenVZ: interface name, IP address
    Allow tool container to contact some external hosts.
    add rules in DOCKER-USER chain, which gets packets from FORWARD
    groups is an array of groups the user belongs to
    FW_GROUP_MAP is an array of ["group_name", net_cidr, portmin, portmax]
    where net_cidr is something like 128.46.19.160/32
    """
    rule_start = [self.k["FW_CHAIN"], '-i', 'br_toolnet', '-s', self.tool_container_IP + '/32']
    for g in self.k["FW_GROUP_MAP"]:
      if g[0] in groups:
        (net_cidr, portmin, portmax) = g[1:]
        if net_cidr == "":
          # block access to hubzero networks due to firewall rule exceptions
          hostpart = ['!', '-d', self.k["MW_PROTECTED_NETWORK"]]
        else:
          # hostpart = ['-d', socket.gethostbyname(host)]
          hostpart = ['-d', net_cidr]
        if portmin == 0:
          # all ports
          portpart = []
        else:
          if portmax == 0:
            # single port
            portpart = ['-p', 'tcp', '--dport', '%s' % portmin]
          else:
            portpart = ['-p', 'tcp', '--dport', '%s:%s' % (portmin, portmax)]
        fwd_rule = rule_start + hostpart + portpart + ['-j', 'ACCEPT']
        if DEBUG:
          log('DEBUG: ' + operation + ' ' + ' '.join(fwd_rule))
        if operation == 'add':
          i=0
          while i < 3:
            # retry adding firewall rules.  Log final failure and keep going.
            try:
              # insert due to RETURN or DROP rules
              subprocess.check_call(['/sbin/iptables', '--wait', '100', '-I'] + fwd_rule)
              i = 3
            except subprocess.CalledProcessError:
              log('Warning: unable to add firewall rule ' + ' '.join(fwd_rule))
              i += 1
              if i < 3:
                time.sleep(1)
        else:
          # retry deleting firewall rules, but eventually ignore the error
          i=0
          while i < 10:
            try:
              subprocess.check_call(['/sbin/iptables', '--wait', '100', '-D'] + fwd_rule)
              i = 10
            except subprocess.CalledProcessError:
              log('Warning: unable to delete firewall rule ' + ' '.join(fwd_rule))
              i += 1
              if i < 10:
                time.sleep(1)

  def firewall_cleanup(self):
    """Get all rules for the AUTO_FORWARD chain.  Delete the ones that include the IP address of this container.
    This is more robust than calling firewall_by_group with the delete operation, because group memberships
    could have changed since the session start."""

    p = subprocess.Popen(['/sbin/iptables', '--wait', '100', '-S', 'DOCKER-USER'], stderr=subprocess.PIPE, stdout=subprocess.PIPE)
    (stdout, stderr) = p.communicate()
    if p.returncode != 0:
      log("Unable to list rules in DOCKER-USER chain")
      return
    lines= stdout.split('\n')
    for line in lines:
      if line.find(self.tool_container_IP + '/32') != -1:
        args = ['/sbin/iptables', '--wait', '100', '-D'] + line.split()[1:]
        i = 0
        while i < 10:
          try:
            subprocess.check_call(args)
            i = 10
          except subprocess.CalledProcessError:
            log('Warning: unable to delete firewall rule: ' + ' '.join(args))
            i += 1
            if i < 10:
              time.sleep(1)

  def check_stunnel(self):
    """If tunnel doesn't exist, restart it by calling the stunnel function"""
    in_port = self.veid + self.k["STUNNEL_PORTS"] # e.g., 4000 + display
    if os.path.exists('/usr/sbin/lsof'):
      p = subprocess.Popen(['/usr/sbin/lsof', '-i', ':%d' % in_port], stderr=subprocess.PIPE, stdout=subprocess.PIPE)
      (stdout, stderr) = p.communicate()
      if p.returncode == 0:
        return
      else:
        log("socat missing on port %d" % in_port)
        self.stunnel()

  def invoke_unix_command(self, user, session_id, timeout, command, params, sesslog_path):
    """Start a tool container named veid.tools.
    We need the container to keep running after the tool stops, so we can gather cpu stats.
    Start container with a detached sleep command.
    Start tool with a separate docker exec command
    Log to session_id.err file, session_id.out file is to contain tool output only.
    Log the exit status of the docker exec command.
    Get container CPU stats.  Notify web server/client that the tool ended.
    user: string
    session_id: string (int+letter)
    timeout: int
    command: string
    """
    start_time = time.time()

    account = make_User_account(user, self.k)

    if self.__is_running('%d.tool' % self.veid):
      # Killing this errand tool container could trigger a chain of events killing the services container as well, which we need.
      log("VERBOSE: dirty start.  Killing tool container that was already running and shouldn't have been")
      self.__docker_kill('%d.tool' % self.veid)

    # make sure that socat is still running for the service container
    self.check_stunnel()

    # Copy Xauthority info and run the tool.
    # use --init option to be able to send signals to submit --local and others.
    args = ['/usr/bin/docker', 'run', '--init', '--name', '%d.tool' % self.veid]
    args += ['--storage-opt']
    if 'DOCKER_QUOTA' in self.k:
      args += ['size=%s' % self.k['DOCKER_QUOTA']]
    else:
      args += ['size=5g']
    if 'DOCKER_RUNTIME' in self.k:
      args += ['--runtime=%s' % self.k['DOCKER_RUNTIME']]

    # DOCKER_GPUS: Give docker containers access to nvidia gpus via nvidia-container-toolkit *njk*
    if 'DOCKER_GPUS' in self.k and self.k['DOCKER_GPUS']:
      args += ['--gpus','all']

    # DOCKER_XSHM: Share X11 domain socket with service container and enable shared memory transport for X11
    #              This is a work in progress as xauth still needs to be setup for this
    #              to actually be used. *njk*
    if 'DOCKER_XSHM' in self.k and self.k['DOCKER_XSHM']:
      args += ['--ipc=container:%d.services' % self.veid ]
      args += ['-v','%d.X11:/tmp/.X11-unix' % self.veid]

    # DOCKER_VGL: Enable VirtualGL support by sharing the X11 (display 1) domain socket
    #             from the X11 server running on the host and presetting some VirtualGL
    #             environment variables to set some vglrun defaults. For VirtualGL to work
    #             DOCKER_GPUS would also need to be defined. *njk*
    if 'DOCKER_VGL' in self.k and self.k['DOCKER_VGL']:
      args += ['-v','/tmp/.X11-unix/X1:/tmp/.X11-unix/X1']
      args += ['-e', 'VGL_DISPLAY=:1']
      args += ['-e', 'VGL_COMPRESS=proxy']

    # cleanup: don't keep image around after it's done
    # detached execution so container keeps running after tool is done
    args += ['--rm', '-d']
    # example add-host options:
    # args += ['--add-host=datacenterhub.org:10.111.11.100']
    # args += ['--add-host=dev.datacenterhub.org:10.111.13.100']
    if 'DOCKER_ADD_HOSTS' in self.k:
      for h in self.k['DOCKER_ADD_HOSTS']:
        args += ['--add-host=' + h ]
    # mount Xvnc volume for weber tools that verify the vnc password and so need to read /var/run/Xvnc/passwd-*
    args += ['-v', '%d.Xvnc:/var/run/Xvnc' % self.veid]
    # connect to toolnet
    args += ['--network', self.toolnet_name, '--ip', self.tool_container_IP]
    args += ['-e', 'DISPLAY=%s:0' % self.services_container_IP]
    # /etc/passwd
    args += ['-e', 'etcpasswd=' + account.passwd_entry() + '\n']
    # /etc/shadow
    args += ['-e', 'etcshadow=' + user + ':!:17052:0:99999:7:::\n']
    # set hostname displayed at prompt inside container
    if 'DOCKER_HOSTNAME' in self.k:
      args += ['-h', self.k['DOCKER_HOSTNAME'] + "_%s_%d" % (session_id, self.veid)]
    else:
      args += ['-h', "docker_%s_%d" % (session_id, self.veid)]

    # Do we mount /apps?
    # need to know labels on image we'll use. #1: determine image
    #
    # search for image based on name
    # expecting command like '/apps/jupyter/r16/middleware/invoke'
    toolname = command.split('/')[2]
    revision = command.split('/')[3]
    # toolname should be like 'jupyter'
    image_test = ['docker', 'image', 'ls', toolname + '_' + revision,  '-q']
    p = subprocess.Popen(image_test, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    (stdout, stderr) = p.communicate()
    if len(stdout) > 0:
      tool_image = toolname + '_' + revision
    else:
      image_test = ['docker', 'image', 'ls', toolname,  '-q']
      p = subprocess.Popen(image_test, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
      (stdout, stderr) = p.communicate()
      if len(stdout) > 0:
        tool_image = toolname
      elif 'DOCKER_IMAGE_TOOL' in self.k:
        tool_image = self.k["DOCKER_IMAGE_TOOL"]
      else:
        tool_image = 'hubzero_tools'

    # Check image labels to know if we need to mount /apps or not
    # default: mount it
    # mountapps is expected to have a value assigned
    # docker inspect -f '{{.Config.Labels}}' tool_image
    image_labels_cmd = ['docker', 'inspect', '-f', '{{.Config.Labels.mountapps}}', tool_image]
    p = subprocess.Popen(image_labels_cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    (ma_out, ma_err) = p.communicate()
    mountapps = stdout != '<no value>\n'
    if ma_out.find('<no value>') == 0:
      mountapps = True
    elif ma_out.find('false') == 0:
      mountapps = False
    else:
      mountapps = True
    log("DEBUG: label is %s" % ma_out)
    if 'apps' in account.groups():
      # add the apps user and edit the sudoers file to allow su to apps
      apps_user = make_User_account('apps', self.k)
      apps_groups = apps_user.groups()
      args += ['-e', 'etcpasswdapps=' + apps_user.passwd_entry() + "\n"]
      args += ['-e', 'etcshadowapps='+apps_user.shadow_entry() + "\n"]
      args += ['-e', 'etcsudoers=%apps           ALL=NOPASSWD:/bin/su - apps\n']
      if mountapps:
        # mount /apps read-write
        args += ['-v', account.home_prefix + '/apps:/apps']
        log("VERBOSE: mounting /apps r/w")
      # mount apps home directory
      args += ['-v', apps_user.ext_homedir() + ':' + apps_user.homedir]
    else:
      apps_groups = []
      if mountapps:
        # mount /apps read-only
        args += ['-v', account.home_prefix + '/apps:/apps:ro']
        log("VERBOSE: mounting /apps r/o")
    if self.ionhelper:
      # the ionhelper account must exist for the Xauthority step, it is needed by the setup_accounts script
      # this includes the sudoers rules.
      args += ['-e', 'etcpasswdhelper=ionhelper:x:199:199::/var/ion/:/bin/false\n']
      args += ['-e', 'etcshadowhelper=ionhelper:*:17821:0:99999:7:::\n']
      # add the user to the ionhelper group
      args += ['-e', "group_ionhelper=ionhelper:x:%d:%s\n" % (199, user)]
      # provide information to copy resource file
      # path to copy from, path to copy to, session id
      sessiondir_helper = '/var/ion/data/sessions/%s' % (session_id)
      args += ['-e', "sessiondir_helper=" + sessiondir_helper]
      # Copy resources file for helper (copy happens in setup_accounts script)
      # resources file path for user is rpath_user
      rpath_user = "%s/data/sessions/%s/resources" % (account.homedir, session_id)
      args += ['-e', 'rpath_user=' + rpath_user]
      # resources file path for helper is rpath_helper
      rpath_helper = "/var/ion/data/sessions/%s/resources" % (session_id)
      args += ['-e', 'rpath_helper=' + rpath_helper]
      args += ['-e', 'sessionid=' + session_id]
    elif 'HELPER' in self.k:
      args += ['-e', 'etcpasswdhelper=%s:x:%s:%s::%s:/bin/false\n' % (self.k['HELPER'], self.k['HELPER_UID'], self.k['HELPER_GID'], self.k['HELPER_HOME'])]
      args += ['-e', 'etcshadowhelper=%s:*:17821:0:99999:7:::\n' % self.k['HELPER']]
      # add the user to the helper group.  note self.k['HELPER_GID'] is actually a string so use %s
      args += ['-e', "group_%s=%s:x:%s:%s,%s\n" % (self.k['HELPER'], self.k['HELPER'], self.k['HELPER_GID'], user, self.k['HELPER'])]
      # provide information to copy resource file
      # path to copy from, path to copy to, session id
      sessiondir_helper = "%s/data/sessions/%s" % (self.k['HELPER_HOME'], session_id)
      args += ['-e', "sessiondir_helper=" + sessiondir_helper]
      # Copy resources file for helper (copy happens in setup_accounts script)
      # resources file path for user is rpath_user
      rpath_user = "%s/data/sessions/%s/resources" % (self.k['HELPER_HOME'], session_id)
      args += ['-e', 'rpath_user=' + rpath_user]
      # resources file path for helper is rpath_helper
      rpath_helper = "%s/data/sessions/%s/resources" % (self.k['HELPER_HOME'], session_id)
      args += ['-e', 'rpath_helper=' + rpath_helper]
      args += ['-e', 'sessionid=' + session_id]

    # problem passing group names as environment variables on the command line:
    # dashes in group names are interpreted by docker as options!
    # how to escape them?
    # first attempt: replace them with something else, like underscores
    # perhaps use a very rare combination like 3 underscores, and in the "setup_accounts" script that
    # creates entries in /etc/group, restore to a dash
    # see hubzero-mw2-exec-service package
    for g in account.group_pairs():
      gname = g[0]
      gid = g[1]
      if gid > 500:
        # copy group info if gid > 500
        if gname in apps_groups:
          # support su to apps
          # Create all the groups that user apps belongs to, and add user apps and user helper if applicable

          if 'HELPER' in self.k:
            args += ['-e', "group_%s=%s:x:%d:%s,%s,%s\n" % (gname.replace('-', '_'), gname, gid, user, 'apps', self.k['HELPER'])]
          else:
            args += ['-e', "group_%s=%s:x:%d:%s,%s,%s\n" % (gname.replace('-', '_'), gname, gid, user, 'apps', 'ionhelper')]
        else:
          if 'HELPER' in self.k:
            args += ['-e', "group_%s=%s:x:%d:%s,%s\n" % (gname.replace('-', '_'), gname, gid, user, self.k['HELPER'])]
          else:
            args += ['-e', "group_%s=%s:x:%d:%s,%s\n" % (gname.replace('-', '_'), gname, gid, user, 'ionhelper')]

    for defgroup in self.k["DEFAULT_GROUPS"]:
      groupinfo = grp.getgrnam(defgroup)
      gname = groupinfo[0]
      gid = groupinfo[2]
      args += ['-e', "group_%s=%s:x:%d:%s\n" % (defgroup.replace('-', '_'), defgroup, gid, user)]

    # mount user's home directory
    args += ['-v', account.ext_homedir() + ':' + account.homedir]
    # extra mount points
    if 'EXTRA_MOUNT_PATHS' in self.k:
      for mnt_pt in self.k['EXTRA_MOUNT_PATHS']:
        args += ['-v', mnt_pt[0] + ':' + mnt_pt[1]]

    # Mount projects based on user membership
    # project path can be different from public project file areas
    # this does not translate uids and gids, and is not appropriate for the hub's project storage that is owned by the Apache user and apache group
    # it is used for /srv/idata or /srv/irods where directories are owned by the project groups
    # PROJECT_PATHS: an array of directory paths ending with /
    if "PROJECT_MOUNT" in self.k and self.k["PROJECT_MOUNT"]:
      if "PROJECT_PATHS" in self.k:
        for p in self.k["PROJECT_PATHS"]:
          for g in account.group_pairs():
            gname = g[0]
            gid = g[1]
            if gname[0:3] == "pr-":
              source_mount = p + gname[3:]
              if not os.path.exists(source_mount):
                continue
              args += ['-v', source_mount + ':' + source_mount]
      elif "PROJECT_PATH" in self.k:
        for g in account.group_pairs():
          gname = g[0]
          gid = g[1]
          if gname[0:3] == "pr-":
            source_mount = self.k["PROJECT_PATH"] + gname[3:]
            if not os.path.exists(source_mount):
              continue
            args += ['-v', source_mount + ':' + source_mount]
    # Username-based mounts
    # uses bind mounts from an already mounted filesystem, which has a directory for each user
    # the mounted directory is the username
    if self.k["USER_MOUNT"]:
      for mount_pt in self.k["USER_MOUNT_POINTS"]:
        # mount_pt must already exist
        if not os.path.exists(mount_pt):
          log("Mount point '%s' does not exist" % mount_pt)
          continue
        source_mount = mount_pt + user
        # check if source exists
        if not os.path.exists(source_mount):
          # create it as the user, not root
          mk_args = ['/bin/su', user, '-c', "mkdir -m 0700 " + source_mount]
          p = subprocess.Popen(mk_args)
          p.communicate()
          if p.returncode != 0:
            if VERBOSE:
              log("Warning: '%s' did not exist, could not create it as user '%s', so will not be mounted\n" % (source_mount, user))
            continue
        args += ['-v', source_mount + ':' + source_mount]

    # Mount public project file areas
    # can be a different path from user membership-based mounts
    if "PROJECT_PUBLIC_MOUNT" in self.k and self.k["PROJECT_PUBLIC_MOUNT"]:
      # mount read/write public areas for project members
      # doesn't work if owned by Apache user...
      #for g in account.group_pairs():
      #  gname = g[0]
      #  gid = g[1]
      #  if gname[0:3] == "pr-":
      #    source_mount = self.k["PROJECT_PUBLIC_PATH"] + gname[3:] + "/files/public"
      #    if not os.path.exists(source_mount):
      #      continue
      #    args += ['-v', source_mount + ':' + source_mount]
      import glob
      if "PROJECT_PUBLIC_PATHS" in self.k:
        for p in self.k["PROJECT_PUBLIC_PATHS"]:
          listing = glob.glob(p + '*/files/public')
          for pubpath in listing:
            if pubpath + ':' + pubpath not in args:
              args += ['-v', pubpath + ':' + pubpath + ':ro']
      if "PROJECT_PUBLIC_PATH" in self.k:
        listing = glob.glob(self.k["PROJECT_PUBLIC_PATH"] + '*/files/public')
        for pubpath in listing:
          if pubpath + ':' + pubpath not in args:
            args += ['-v', pubpath + ':' + pubpath + ':ro']
    for mount in self.mounts:
      if not os.path.exists(mount[0]):
        continue
      args += ['-v', mount[0] + ':' + mount[0] + ':' + mount[1]]

    # USER environment variable used for xauth operation
    # Can't use the --user option because we need root to setup account and group information
    args += ['-e', 'USER='+user]
    # set current working directory.  Replaces the "cd" operation done by the OpenVZ middleware
    args += ['-w', account.homedir]

    # memory limits
    if 'DOCKER_MEMORY' in self.k:
      args += ['-m', self.k["DOCKER_MEMORY"]]
    else:
      args += ['-m', '1G']

    # swap out idle containers
    args += ['--memory-swappiness', '95']
    # memory-swap must be >= memory
    if 'DOCKER_MEM_SWAP' in self.k:
      args += ['--memory-swap', self.k["DOCKER_MEM_SWAP"]]
    # don't use this Docker option, it results in unresponsive containers: args += ['--oom-kill-disable']

    if 'DOCKER_MEM_RESERVATION' in self.k:
      args += ['--memory-reservation', self.k["DOCKER_MEM_RESERVATION"]]

    if 'DOCKER_MEM_KERNEL' in self.k:
      args += ['--kernel-memory', self.k["DOCKER_MEM_KERNEL"]]

    # CPU quota.  0.5: at most 50% of a CPU core every second.
    if 'DOCKER_CPUS' in self.k:
      args += ['--cpus', self.k["DOCKER_CPUS"]]

    # CPU shares, relative number >=2
    # default is 1024, tool containers should have a low number
    #
    # Don't use DOCKER_CPU_SHARES if enabling GPU as docker will be unable to reset the value
    # without removing the GPU privleges granted in GPU mode (see comments where 'at' command
    # is setup below). *njk*

    if ('DOCKER_CPU_SHARES' in self.k) and (not 'DOCKER_GPUS' in self.k or not self.k['DOCKER_GPUS']):
      args += ['--cpu-shares', self.k["DOCKER_CPU_SHARES"]]
    else:
      args += ['--cpu-shares', '1024']
    # image must have ENTRYPOINT that updates account and group info from environment variables, then sleeps

    args += [tool_image]
    if DEBUG:
      log("DEBUG: command is %s" % args)
    # output from this command shouldn't go in the sessnum.err and sessnum.out files
    p = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    (stdout, stderr) = p.communicate()
    if p.returncode != 0:
      # docker kill containers in this display
      self.stop()
      # delete volume for services auth
      self.umount()
      # delete lock directory (difference between "deleted" and "down" states)
      self.delete_confs()
      raise MaxwellError("Can't start tool container '%d.tool' because %s" % (self.veid, stderr))
    if DEBUG:
      log("DEBUG: container UUID is %s" % stdout.strip())
    container_time = time.time()
    if DEBUG:
      log("DEBUG: time to start tool container was %f seconds" % (container_time - start_time))

    # Setup the firewall rules on the host after calling the first "docker run" because we don't know the IP address before that
    self.firewall_by_group(account.groups(), 'add')
    firewall_time = time.time()
    if DEBUG:
      log("DEBUG:time to setup firewall rules was %f seconds" % (firewall_time - container_time))

    # run tool separately
    # if an error is returned, it's difficult to know if it's because the tool failed to run or
    # if the session ended normally but stopping the X server caused an error.
    # create logs for the tool
    err = open(sesslog_path + ".err", 'a', 0)
    out = open(sesslog_path + ".out", 'a', 0)
    if VERBOSE:
      err.write("VERBOSE: Starting command '%s' for '%s' with timeout '%s'\n" % (command, user, timeout))

    # Verify that the setup script has finished setting up user accounts
    args = ['/usr/bin/docker', 'exec', '%d.tool' % self.veid, 'getent', 'passwd', user]
    i = 0
    rc = 1
    while i < 100 and rc != 0:
      p = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=None)
      p.communicate()
      rc = p.returncode
      if rc != 0:
        time.sleep(1)
        i += 1
    if i >= 100:
      # Restore the firewall
      self.firewall_cleanup()
      # docker kill containers in this display
      self.stop()
      # delete volume for services auth
      self.umount()
      # delete lock directory (difference between "deleted" and "down" states)
      self.delete_confs()
      raise MaxwellError("User account getent test failed")

    # Finally start the tool
    args = ['/usr/bin/docker', 'exec', ]
    if 'HELPER' in self.k:
      # expecting command like '/apps/jupyter/r16/middleware/invoke'
      toolname = command.split('/')[2]
      if toolname in self.k['HELPER_TOOLS']:
        # run this command as user (no su needed on our part, so no issues with escaping characters)
        args += ['--user', self.k['HELPER']]
        # set current working directory.  Replaces the "cd" operation done by the OpenVZ middleware
        args += ['-w', self.k['HELPER_HOME']]
        # environment for tool:
        # can't use account.env because it provides quotes that become part of the actual value of the variables!
        #for e in account.env(session_id, timeout, params):
        #  args += ['-e', e]
        args += ['-e', 'DISPLAY=%s:0' % self.services_container_IP]
        args += ['-e', "SESSIONDIR=%s" % sessiondir_helper]
        args += ['-e', "RESULTSDIR=%s" % rpath_helper]
        args += ['-e', "SESSION=%s" % session_id]
        args += ['-e', "LANG=en_US.UTF-8"]
        args += ['-e', "LANGUAGE=en_US.UTF-8"]
        args += ['-e', "LC_ALL=en_US.UTF-8"]
        args += ['-e', "TIMEOUT=%s" % str(timeout)]
        args += ['-e', "USER=%s" % self.k['HELPER']]
    if '--user' not in args:
      # default way of invoking tools
      # run this command as user (no su needed on our part, so no issues with escaping characters)
      args += ['--user', user]
      # set current working directory.  Replaces the "cd" operation done by the OpenVZ middleware
      args += ['-w', account.homedir]
      # environment for tool:
      # can't use account.env because it provides quotes that become part of the actual value of the variables!
      #for e in account.env(session_id, timeout, params):
      #  args += ['-e', e]
      args += ['-e', 'DISPLAY=%s:0' % self.services_container_IP]
      args += ['-e', "SESSIONDIR=%s" % account.int_session_dir(session_id)]
      args += ['-e', "RESULTSDIR=%s" % account.int_results_dir(session_id)]
      args += ['-e', "SESSION=%s" % session_id]
      args += ['-e', "LANG=en_US.UTF-8"]
      args += ['-e', "LANGUAGE=en_US.UTF-8"]
      args += ['-e', "LC_ALL=en_US.UTF-8"]
      args += ['-e', "TIMEOUT=%s" % str(timeout)]
      args += ['-e', "USER=%s" % user]
    # invoke rewrites the path...  Is setting it ourselves useful?
    args += ['-e', "PATH=%s" % "/bin:/usr/bin:/usr/bin/X11:/sbin:/usr/sbin"]
    if params:
      args += ['-e', "TOOL_PARAMETERS=%s" % account.int_params_path(session_id)]
    try:
      if self.k["EXTRA_ENV_CMD"] != "":
        args += ['-e' + self.k["EXTRA_ENV_CMD"]]
    except KeyError:
      pass
    # container name comes after options
    args += ['%d.tool' % self.veid]

    # Setup "at" job to reconfigure cpu-shares after 1 hour. Note that CPU shares are not enabled
    # when GPU support is enabled. The reason being that GPU support changes permissions within
    # the container in a way that docker is not aware of (this is a function of the nvidia docker
    # support and not anything we have done). When docker then reconfigures the container
    # (say to reset cpu-shares) it also resets everything else to the way it thinks the container
    # is configured. This undoes the gpu permission changes which will then prevent
    # applications from accessing the gpus. This could be done without 'at' if we had
    # a wait loop rather than block on docker process run. *njk*

    atjob = None

    if ('DOCKER_CPU_SHARES' in self.k) and (self.k['DOCKER_CPU_SHARES'] != 1024) and (not 'DOCKER_GPUS' in self.k or not self.k['DOCKER_GPUS']):
      try:
        proc = subprocess.Popen( ['/bin/at', '-M', 'now', '+', '1', 'hour'],
                                stdin=subprocess.PIPE,
                                stdout=subprocess.PIPE,
                                stderr=subprocess.PIPE)

        stdout, stderr = proc.communicate("/bin/docker update --cpu-shares 1024 %d.tool" % self.veid)

        match = re.search(r'job\s+(\d+)\s+at', stderr)

        if match:
          atjob = match.group(1)

      except OSError:
        log("DEBUG:unable to execute 'at' command to reduce cpu shares")
        pass

    # the actual command we'll run inside the container.
    # Without "split", it tries to find an executable that matches the entire command including spaces and options
    args += command.split()
    ready_time = time.time()
    if DEBUG:
      log("DEBUG:time to get ready to start container was total %f seconds" % (ready_time - start_time))
    p = subprocess.Popen(args, stderr=err, stdout=out)
    p.communicate()

    end_time = time.time()
    tool_ec = p.returncode

    # Calculate time stats
    if VERBOSE:
      err.write("Processing stats\n")
    err.write("real\t%f\n" % (end_time - ready_time))
    self.__printDockerStats(err)
    err.write("Exit_Status: %d\n" % tool_ec)

    # Make sure we remove any pending 'at' jobs for this container or it will be applied
    # prematurely to a container reusing the same name. *njk*
    if atjob != None:
      try:
        proc = subprocess.Popen( ['/bin/atrm', atjob ], stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        stdout, stderr = proc.communicate()

        if proc.returncode != 0:
          log("DEBUG:atrm failed")

      except OSError:
        log("DEBUG:unable to execute 'atrm' command to cancel at job")
        pass

    # docker kill containers in this display
    self.stop()
    # Restore the firewall
    self.firewall_cleanup()
    # delete volume for services auth
    self.umount()
    # don't log anything else
    err.close()
    # delete lock directory (difference between "deleted" and "down" states)
    self.delete_confs()
    return 0

  @staticmethod
  def __check_or_create_network(name, CIDR, gateway, masquerade=False):
    """Check that the container network exists, or create it
    docker inspect self.servicenet_name returns 1 if does not exist
    create them without masquerade, so they don't have internet access by default
    bridge name is br_name, for ease of firewall rule creation or check
    usage: __check_or_create_network(self, self.servicenet_name, self.servicenet_CIDR, self.k["PRIVATE_NET"] % (0, 1))"""
    args = ['docker', 'inspect', name]
    p = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    p.communicate()
    if p.returncode != 0:
      # docker network create --driver bridge --subnet 192.168.0.0/16 --gateway 192.168.0.1 --internal --icc=false servicenet
      args = ['docker', 'network', 'create', '--driver', 'bridge', '--subnet', CIDR, '--gateway', gateway]
      # disable forwarding outside this network
      args += ['--internal']
      # isolate containers so tools can't attack each other
      args += ['-o', 'com.docker.network.bridge.enable_icc=false']
      args += ['-o', 'com.docker.network.bridge.name=br_' + name]
      if masquerade:
        args += ['-o', 'com.docker.network.bridge.enable_ip_masquerade=true']
      else:
        args += ['-o', 'com.docker.network.bridge.enable_ip_masquerade=false']
      args += [name]
      p = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
      (stdout, stderr) = p.communicate()
      if p.returncode != 0:
        log(" ".join(args))
        raise MaxwellError("Unable to create network " + name + " " + stderr)
      else:
        if VERBOSE:
          log("VERBOSE: Created network " + name)
    elif DEBUG:
      log("DEBUG: no action necessary for network " + name)

  @staticmethod
  def __check_or_add_iptables(rule, operation, chain='filter'):
    # iptables -C uses the logic for -D (delete) so it returns 0 if the rule exists, 1 if it doesn't exist
    # operation is -I for insert or -A for append
    args = ['/sbin/iptables', '--wait', '100', '-t', chain, '-C'] + rule.split()
    p = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    p.communicate()
    if p.returncode !=0:
      if VERBOSE:
        log("VERBOSE: Adding firewall rule %s" % rule)
      i=0
      while i < 3:
        i += 1
        # retry adding firewall rules.  Log final failure and keep going.
        try:
          # insert due to RETURN or DROP rules
          subprocess.check_call(['/sbin/iptables', '--wait', '100', '-t', chain, operation] + rule.split())
          i = 3
        except subprocess.CalledProcessError:
          log('Warning: iptables operation failed: %s' % rule)
          if i < 3:
            time.sleep(1)

  @staticmethod
  def __get_container_UUID(ctname):
    # get container ID from name
    # docker ps -aqf "name=24.tool" returns abbreviated name, we need the full name
    args = ['docker', 'inspect', '-f', '{{.Id}}', ctname]
    p = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    (stdout, stderr) = p.communicate()
    if DEBUG:
      log(args)
      log("container name %s" % stdout.rstrip('\n'))
    return stdout.rstrip('\n')

  @staticmethod
  def __parse_cpuacct(uuid):
    """
    argument is full length Docker container ID
    return (user, system) times parsed out of /cgroup/cpuacct/docker/%s/cpuacct.stat
    expecting to read something like:
      user 4000
      system 2394
    """
    # Path to cgroups varies by OS
    cgroup_path1 = '/sys/fs/cgroup/cpuacct/docker/%s/cpuacct.stat'
    cgroup_path2 = '/cgroup/cpuacct/docker/%s/cpuacct.stat'
    user_time = 0
    system_time = 0
    stats_path = cgroup_path1 % uuid
    if not os.path.isfile(stats_path):
      stats_path = cgroup_path2 % uuid
    if os.path.isfile(stats_path):
      try:
        f = open(stats_path)
        for line in f:
          if line.find('user') != -1:
            user_time = int(line.split()[1])
            continue
          if line.find('system') != -1:
            system_time = int(line.split()[1])
            continue
      except IOError, exc:
        # this can happen if container is already stopped
        if VERBOSE:
          log("VERBOSE: gathering container CPU stats failed due to exception:'%s'\n" % exc)
    return (user_time, system_time)

  @staticmethod
  def __get_container_IP(ctname):
    # get container ID from name
    # docker ps -aqf "name=24.tool" returns abbreviated name, we need the full name
    args = ['docker', 'inspect', '-f', '{{.NetworkSettings.Networks.toolnet.IPAddress}}', ctname]
    p = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    (stdout, stderr) = p.communicate()
    if p.returncode !=0:
      log("Unable to find IP address for %s" % ctname)
    if DEBUG:
      log(args)
      log("IP address is '%s'" % stdout.rstrip('\n'))
    return stdout.rstrip('\n')

class ContainerAWS(ContainerDocker):
  """For deployment on the onescienceplace.org gen 1 prototype (circa 2018), using a service to get user and group information, MySQL transactions, based on Docker.
  Allow for only 255 containers/execution host because hosts are smaller VMs"""


  def __init__(self, disp, machine_number, overrides={}):
    # machine_number: ignored (lint)
    self.k = CONTAINER_K
    self.k.update(overrides)
    self.disp = disp
    self.veid = disp
    self.vncpass = None
    # mounts:  array to record list of bind mounts to make when starting container
    self.mounts = []
    if disp < 1:
      raise MaxwellError("Container ID must be at least 1")
    #
    # we use /24 networks
    if disp > 253:
      # container count starts at 1;  we want container #1 to use the .1 address
      # 255 IP is broadcast address
      # 254th IP is the Docker bridge's IP
      raise MaxwellError("container ID %d too high, maximum is 253" % disp)

    # tools network doesn't offer any services, by default doesn't connect to internet
    self.toolnet_name = 'toolnet'

    # service network provides X server at known IP address for proxy to connect to it
    # without IP collision in .Xauthority file
    self.servicenet_name = 'servicenet'
    # service net CIDR provided by self.k["PRIVATE_NET"] and machine_number calculation

    self.toolnet_CIDR = self.k["DOCKERTOOL_NET"] + '/24'
    gwbytes = self.k["DOCKERTOOL_NET"].split('.')
    gateway = gwbytes[0:3] + ['254']
    self.toolnet_gateway = ".".join(gateway)
    IPa = gwbytes[0:3] + [str(disp)]
    self.tool_container_IP = ".".join(IPa)

    self.servicenet_CIDR = self.k["DOCKERSERVICE_NET"] + '/24'
    gwbytes = self.k["DOCKERSERVICE_NET"].split('.')
    gateway = gwbytes[0:3] + ['254']
    self.servicenet_gateway = ".".join(gateway)
    IPa = gwbytes[0:3] + [str(disp)]
    self.services_container_IP = ".".join(IPa)

    # for inherited stunnel function
    self.veaddr = self.services_container_IP

  def stunnel(self):
    """On AWS, hzvncproxy runs in a Docker container on each execution host and connects directly
    to the X server.  No need for stunnel.
    """
    pass
