# @package      hubzero-mw2-common
# @file         fileserver.py
# @author       Pascal Meunier <pmeunier@purdue.edu>
# @copyright    Copyright (c) 2016-2017 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Based on previous work by Richard L. Kennell and Nicholas Kisseberth
#
# Copyright (c) 2016-2017 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

"""
Fileserver encapsulates how to talk to a file server middleware service, and which operations the client can request.
"""

from errors import MaxwellError
from constants import HOST_K
from log import print_n_log
from host import Host
import time

class Fileserver:
  def __init__(self, c, user, sess, homedir_retries=10, overrides={}, host=None, zone=None):
    """Find the fileserver."""
    self.K = HOST_K
    self.K.update(overrides)
    if host is None:
      # find a file server
      if zone is None:
        arr = c.getall("""SELECT host.hostname FROM host
             JOIN hosttype ON host.provisions & hosttype.value != 0
             WHERE hosttype.name='fileserver' AND host.status='up'""", ())
        if len(arr) != 1:
          raise MaxwellError("Cannot find a single host with a 'fileserver' attribute set.")
        else:
          row = arr[0]
          self.fs_name = row[0]
          self.host = Host(row[0])
      else:
        arr = c.getall("""SELECT host.hostname FROM host
             JOIN hosttype ON host.provisions & hosttype.value != 0
             WHERE hosttype.name='fileserver' AND host.status='up'
             AND zone_id=%s""", (zone.zone_id))
        if len(arr) != 1:
          raise MaxwellError("Cannot find a single host with a 'fileserver' attribute set.")
        else:
          row = arr[0]
          self.fs_name = row[0]
          self.host = Host(row[0])

    else:
      self.fs_name = host.hostname  # FQDN of file server
      self.host = host
    self.user = user # on behalf of whom we operate
    self.sess = sess # session object
    self.homedir_retries = homedir_retries

  def remote(self, service_cmd, args, feed=None):
    """Call the maxwell_fs script on an execution host.  Preserve return value.

    Callers don't need to know details of how to call maxwell service on a host.  Also don't need to do
    preflight checks or remember them.
    args is an [].  Give an empty [] if there are none.

    feed is data to feed to remote service.  For example, Fileserver.update_resources does that."""

    self.host.check_ssh_keys(True)
    return self.host.ssh([self.K["FS_PATH"], service_cmd] + args, feed)

  def ask_remote(self, service_cmd, args, feed=None):
    """Call the maxwell_fs script on an execution host.  Preserve return value.

    Callers don't need to know details of how to call maxwell service on a host.  Also don't need to do
    preflight checks or remember them.
    args is an [].  Give an empty [] if there are none.

    feed is data to feed to remote service.  For example, Fileserver.update_resources does that."""

    self.host.check_ssh_keys(True)
    return self.host.ask_ssh([self.K["FS_PATH"], service_cmd] + args, feed)

  def update_resources(self, c, app, submit_app_list, version, visualization_params, hub_name, hub_url, hub_template, fileport, filecookie):
    """Append stuff to the session resource file."""
    #(appname,geometry,depth,command,hostreq,timeout,appfullname) = appinfo

    do_submit = HOST_K["DO_SUBMIT"]
    for sa in submit_app_list:
      if app.appname.startswith(sa):
        do_submit = True

    feed = "\n".join([
      "filexfer_port %d\n" % fileport,
      "filexfer_cookie %s\n" % filecookie,
      "filexfer_decoration",
      "\n",
      visualization_params,
      # visualization_params needed for the submit command, for example:
      # visualization_params="""nanovis_server render00:2000,render01:2000,render02:2000
      # molvis_server render00:2020,render01:2020,render02:2020
      # submit_target tls://devsubmit.nanohub.org:831
      # submit_target tcp://devsubmit.nanohub.org:830"""

      "\n",
      """application_name "%s"\n""" % app.appfullname,
      """version %s\n""" % version,
      """session_token %s\n""" % self.sess.sesstoken,
      """hub_name %s\n""" % hub_name,
      """hub_url %s\n""" % hub_url,
      """hub_template %s\n""" % hub_template,
      """job_protocol submit\n""" if do_submit else """job_protocol exec\n""",
      HOST_K["CACHE_HOSTS"]
      ])

    comm = [
      self.user,
      self.sess.sessname(),
      '%s' % self.sess.disp.dispnum
      ]
    attempt = 0
    exit_code = 1
    while True:
      attempt += 1
      try:
        exit_code = self.remote("update_resources", comm, feed)
      except IOError:
        pass
      if exit_code != 0:
        if attempt >= self.homedir_retries:
          raise MaxwellError("Unable to update resources of '%s'" % self.user)
        print_n_log("Unable to open pipe in update_resources.")
        time.sleep(3)
        continue
      break

  def setup_account(self, c, params):
    """Set up the account on the fileserver."""
    attempt = 0
    exit_code = 1
    while attempt < self.homedir_retries:
      attempt += 1
      try:
        # get a lock in case user makes several simultaneous requests
        c.lock("homedir_" + self.user)
      except MaxwellError:
        time.sleep(3)
      else:
        break
    attempt = 0
    while attempt < self.homedir_retries:
      attempt += 1
      try:
        # directory should not be available until it has been fully constructed and permissions set
        exit_code = self.remote("setup_dir", [self.user, self.sess.sessname(), params])
      except MaxwellError:
        time.sleep(3)
        continue
      if exit_code != 0:
        time.sleep(3)
        continue
      c.unlock("homedir_" + self.user)
      break

    if attempt >= self.homedir_retries:
      raise MaxwellError("Unable to set up the home directory of '%s'" % self.user)

