# @package      hubzero-mw2-common
# @file         user_account.py
# @author       Pascal Meunier <pmeunier@purdue.edu>
# @copyright    Copyright (c) 2016-2017 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Based on previous work by Richard L. Kennell and Nicholas Kisseberth
#
# Copyright (c) 2016-2017 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import os
import pwd
import time
from log import log
from errors import MaxwellError
from constants import FILE_CONFIG_FILE, LIB_PATH, VERBOSE, CONTAINER_K

DEBUG = True
AMOEBA_GROUPS_URL = 'http://groups.service.consul:3000'
 
def make_User_account(user, container_conf = {}):
  """Return a Container class or subclass instance depending on the OS and configuration"""
  if "account_class" in container_conf:
    if container_conf["account_class"] == "User_account":
      if DEBUG:
        print "creating classic account"
      return User_account(user, container_conf)
    elif container_conf["account_class"] == "User_account_anonymous":
      if DEBUG:
        print "creating anonymous account"
      return User_account_anonymous(user, container_conf)
    elif container_conf["account_class"] == "User_account_JSON":
      if DEBUG:
        print "creating JSON account"
      return User_account_JSON(user, container_conf)
    else:
      if DEBUG:
        print "creating classic account"
      return User_account(user, container_conf)
  else:
    print "creating default account"
    return User_account(user, container_conf)

class User_account():
  """Encapsulate methods to access user account information
  so that we can use LDAP, REST APIs or other calls transparently from caller, depending on class
  or subclass.
  """
  def __init__(self, user, container_conf = {}):
    self.user = user
    self.home_prefix = ''
    count = 0
    while count < 10:
      try:
        info = pwd.getpwnam(self.user)
        self.uid = info[2]
        self.gid = info[3]
        self.homedir = info[5]
        self.shell = info[6]
        count = 100
      except StandardError:
        log("No account information for '%s'." % self.user)
        count += 1
        time.sleep(count)
    if count == 10:
      raise MaxwellError("Giving up: account info for '%s'." % self.user)

    if DEBUG:
      log("DEBUG: got account information for '%s': uid: %d, gid %d" % (self.user, self.uid, self.gid))

  def home_quota(self, container_conf = {}):
    # create home directory, copy .environ template
    if not os.path.isfile(self.homedir + "/.environ"):
      if VERBOSE:
        log("missing .environ in home directory %s" % self.homedir)
      import subprocess
      # need to create home directory using root privileges
      self.create_home()
      if not os.path.isfile(os.path.dirname(FILE_CONFIG_FILE) + "/template.environ"):
        log("missing template.environ in service directory %s" % os.path.dirname(FILE_CONFIG_FILE))

      args = ['/bin/su', '-', self.user, '-s', '/bin/dash', '-c',
        '/bin/cp %s %s' % (os.path.dirname(FILE_CONFIG_FILE) + "/template.environ", self.homedir + "/.environ")]
      if DEBUG:
        log("Executing %s" % " ".join(args))
      retcode = subprocess.call(args)
      if retcode != 0:
        log("Unable to copy template.environ")

  def int_session_dir(self, session_id):
    """internal path inside container"""
    return self.homedir + "/data/sessions/%s" % session_id

  def session_dir(self, session_id):
    """Old account management: same as int_session_dir.  New account management: external path inside container"""
    return self.homedir + "/data/sessions/%s" % session_id

  def results_dir(self, session_id):
    return self.homedir + "/data/results/%s" % session_id

  def int_results_dir(self, session_id):
    """internal path from inside container"""
    return self.results_dir(session_id)

  def params_path(self, session_id):
    return self.int_session_dir(session_id) + "/parameters.hz"

  def env(self, session_id, timeout, params):
    e = [
      "HOME=\"%s\"" % self.homedir,
      "LANG=en_US.UTF-8",
      "LANGUAGE=en_US.UTF-8",
      "LC_ALL=en_US.UTF-8",
      "LOGNAME=\"%s\"" % self.user,
      "PATH=\"%s\"" % "/bin:/usr/bin:/usr/bin/X11:/sbin:/usr/sbin",
      "SESSION=\"%s\"" % session_id,
      "SESSIONDIR=\"%s\"" % self.session_dir(session_id),
      "RESULTSDIR=\"%s\"" % self.results_dir(session_id),
      "TIMEOUT=\"%s\"" % str(timeout),
      "USER=\"%s\"" % self.user
    ]
    if params:
      return e + ["TOOL_PARAMETERS=\"%s\"" % self.params_path(session_id)]
    else:
      return e

  def passwd_entry(self):
    return "%s:!:%d:%d::%s:%s" % (self.user, self.uid, self.gid, self.homedir, self.shell)

  def shadow_entry(self):
    # format:   login:password:last:min:maxdays:warning:max_inactive:exp:flag
    # 13281: is in 2006, 99999 gives hundreds of years before it needs to be changed
    return "%s:!:13281:0:99999:7:::" % (self.user)

  def relinquish(self):
    """Relinquish elevated privileges;  adopt user identity for file operations"""
    try:
      os.setregid(self.gid, self.gid)
      os.setreuid(self.uid, self.uid)
      if DEBUG:
        log("set uid to %d and gid to %d" % (self.uid, self.gid))
    except OSError:
      raise MaxwellError("unable to change uid and gid")

    still_root = True
    try:
      os.setreuid(0, 0)
    except OSError:
      still_root = False
    if still_root:
      raise MaxwellError("Was able to revert to root!")
      
  def su_uid(self, cmd, args):
    child_pid = os.fork()
    if child_pid == 0:
      self.relinquish()
      os.environ['UID'] = str(self.uid)
      os.execvp(cmd, [cmd] + args)
      os._exit(0)  
    pid, status = os.waitpid(child_pid, 0)
    return status
      
  def su_uid_multi(self, cmd_array):
    child_pid = os.fork()
    if child_pid == 0:
      import subprocess
      self.relinquish()
      os.environ['UID'] = str(self.uid)
      retcode = 0
      for args in cmd_array:
        # bitwise OR on each return code
        retcode = retcode | subprocess.call(args)
      sys.exit(retcode)  
    pid, status = os.waitpid(child_pid, 0)
    return status

  def groups(self):
    """Return list of group names the user belongs to, including primary groups.
    Some methods look like they should, but do not, return primary groups that are listed without members
    For example group 'public' is not returned when empty:
    #  >>> grp.getgrnam('public')
    #  ('public', '*', 3000, [])
    #import grp
    #return [g.gr_name for g in grp.getgrall() if self.user in g.gr_mem]
    However using "id" does it, after some processing with sed:
    """
    f = os.popen("id %s | sed -e 's/^.*groups=//' -e 's/[0-9]*[(]\([^)]*\)[)]/\\1/g'" % self.user)
    groups = f.read()
    groups = groups.strip()
    f.close()
    return groups.split(',')

  def initgroups(self):
    """Set the process supplementary group IDs, real group ID and effective group ID.
    Used by Virtual SSH.  As of Python 2.6.5 os.initgroups() didn't exist yet.
    Therefore, we have to do it ourselves.
    """
    # This should but doesn't include primary groups
    #import grp
    #grplist = [g.gr_gid for g in grp.getgrall() if self.user in g.gr_mem]
    pipe = os.popen("/usr/bin/id %s" % self.user, "r")
    line = pipe.readline()
    # line looks like uid=123(username) gid=123(group) groups=123(grp1),...
    arr = line.split(' ')
    if len(arr) != 3:
      raise KeyError("User not found")

    # Element 2 looks like groups=123(grp1),456(grp2),...
    arr = arr[2].split('=')

    # Now split it up by commas
    arr = arr[1].split(',')

    # Form the list of gid numbers.
    grplist = []
    for g in arr:
      gid = int(g.split('(')[0])
      grplist += [ gid ]

    os.setregid(self.gid, self.gid)
    return os.setgroups(grplist)
    
  def group_pairs(self):
    """return array of pairs of group names, gids
    something like this [['group10', 10], ['group11', 11]]
    encapsulate calls to getgrnam()."""
    import grp
    groups_a = []
    for g in self.groups():
      try:
        groupinfo = grp.getgrnam(g)
        gname = groupinfo[0]
        gid = groupinfo[2]
        groups_a.append([gname, gid])
      except EnvironmentError:
        pass
    return groups_a

  def ext_homedir(self):
    return self.home_prefix + self.homedir

  def create_home(self):
    if not os.path.isdir(self.ext_homedir()):
      if VERBOSE:
        log("creating home directory at %s" % self.ext_homedir())
      if self.home_prefix != '':
        if os.path.normpath(self.ext_homedir()).find(self.home_prefix) != 0:
          raise MaxwellError('home directory path does not start with ' + self.home_prefix)
      os.mkdir(self.ext_homedir(), 0700)
      os.chown(self.ext_homedir(), self.uid, self.gid)


class User_account_anonymous(User_account):
  """Anonymous isn't a member of any groups.  Do not make any OS or LDAP calls."""
  def __init__(self, user, container_conf = {}):
    self.user = "anonymous"
    self.uid = 1234
    self.homedir = container_conf["HOME_DIR"]+"/"+'anonymous'
  def groups(self):
    return []
  def group_pairs(self):
    return {}
 
class User_account_JSON(User_account):
  """Modified to get account information from REST API returning JSON data
  See https://gitlab.hubzero.org/onescienceplace/amoeba-groups
  1. curl <hostname>:<port>/user/<username> returns /etc/passwd information
  2. curl <hostname>:<port>/groups/<username> is like "groups <username>"
  3. curl <hostname>:<port>/group/<groupname> returns "getent group <groupname>" without the list of users
  4. curl <hostname>:<port>/users/<groupname> returns the full contents of "getent group <groupname>" (and more)
  """
  
  def __init__(self, user, container_conf = {}):
    self.container_conf = container_conf
    self.user = user
    self.groups_dict = None
    self.uid = None
    self.gid = None
    self.get_account()
    self.CONTAINER_MERGED = CONTAINER_K
    self.CONTAINER_MERGED.update(container_conf)
    if 'EXT_PREFIX' in self.CONTAINER_MERGED:
      # The filesystem path on the execution host doesn't match the path inside the container
      self.home_prefix = self.CONTAINER_MERGED['EXT_PREFIX']
    else:
      self.home_prefix = ''

  def get_account(self):
    import json
    import urllib2
    # to test:  info_json = json.loads("""{"id":182,"username":"pmeunier","home":"/Users/pmeunier"}""")
    if self.uid is not None:
      return
    count = 0
    while count < 10:
      try:
        info_json = json.load(urllib2.urlopen(AMOEBA_GROUPS_URL + '/user/' + self.user))
        self.uid = int(info_json['uid'])
        self.gid = int(info_json['primary-group']['gid'])
        self.gname = info_json['primary-group']['name']
        self.homedir = str(info_json['home'])
        self.shell = str(info_json['loginshell'])
        break
      except StandardError:
        if DEBUG:
          log("No account information for '%s' on try #%d" % (self.user, count))
        count += 1
        time.sleep(1)
    if count == 10:
      raise MaxwellError("Account info unavailable for '%s'." % self.user)

  def groups(self):
    """Return list of group names the user belongs to, including primary groups."""
    if self.groups_dict is None:
      self.__get_groups()
    return self.groups_dict.keys()
    
  def group_pairs(self):
    if self.groups_dict is None:
      self.__get_groups()
    return self.groups_dict.items()
    
  def __get_groups(self):
    import json
    import urllib2
    count = 0
    while count < 10:
      try:
        self.groups_dict = {}
        info_json = json.load(urllib2.urlopen(AMOEBA_GROUPS_URL + '/groups/' + self.user))
        self.groups_dict[str(info_json['primary']['name'])] = info_json['primary']['gid']
        if self.gid is None:
          self.gid = int(info_json['primary']['gid'])
        else:
          if self.gid != int(info_json['primary']['gid']):
            raise MaxwellError("inconsistent gid found: %d vs %s" % (self.gid, info_json['primary']['gid']))
        for group in info_json['groups']:
          self.groups_dict[str(group['name'])] = int(group['gid'])
        break
      except StandardError:
        log("No account information for '%s'." % self.user)
        count += 1
        time.sleep(count)
    if count == 10:
      raise MaxwellError("Giving up: account info for '%s'." % self.user)
      
  def initgroups(self):
    raise MaxwellError("initgroups not implemented")

  def results_dir(self, session_id):
    raise MaxwellError("results_dir not implemented")

  def params_path(self, session_id):
    return self.home_prefix + self.int_params_path(session_id)

  def session_dir(self, session_id):
    """External execution host path"""
    return self.home_prefix + self.int_session_dir(session_id)

  def int_results_dir(self, session_id):
    """internal path from inside container"""
    return self.homedir + "/data/results/%s" % session_id
    
  def ext_results_dir(self, session_id):
    """execution host path external to the container"""
    return self.home_prefix + self.int_results_dir(session_id)

  def int_params_path(self, session_id):
    """internal path inside container"""
    return self.int_session_dir(session_id) + "/parameters.hz"

  def home_quota(self, container_conf = {}):
    """
    setup_dir in maxwell_fs_aws doesn't need it
    """
    raise MaxwellError("home_quota not implemented")

