/**
 * @file       git.c
 * @copyright  Copyright (c) 2016-2020 The Regents of the University of California.
 * @license    http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
 *
 * Copyright (c) 2016-2020 The Regents of the University of California.
 *
 * This file is part of: The HUBzero(R) Platform for Scientific Collaboration
 *
 * The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
 * software: you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * HUBzero is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * HUBzero is a registered trademark of The Regents of the University of California.
 */

#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/stat.h>
#include <fuse.h>

#include "git.h"
#include "globals.h"



// Craptastic interface to git
int git(int op, path_t *spath, path_t *dpath, uid_t uid) {
  int ret;
  struct passwd *passwd = getpwuid(uid);

  debug("git(%d, %s, %s)", op, spath->full_path, passwd->pw_name);

  char cmd[2 * PATH_MAX + 100 * 2];
  switch (op) {
    case GIT_RM: {
      // Don't track . files
      if (spath->file_name[0] == '.') return 0;

      // Generate shell command string (ugh)
      snprintf(cmd, sizeof(cmd), "cd %s/%s/files; git rm './%s/%s'; git commit --author='%s <support@hubzero.org>' -m '[SFTP] removed file %s' './%s/%s' >> %s 2>&1",
                     source_dir, spath->repo_name, spath->file_path,
                     spath->file_name, 
                     passwd->pw_name, spath->file_name, spath->file_path,
                     spath->file_name, GIT_LOG);
      debug("%s", cmd);

      // Do it.
      ret = system(cmd);
      if (ret != 0) {
        error("git(): Failed executing '%s', return %d errno %d\n", cmd, ret,
              errno);
      }
      break;
    }

    case GIT_ADD: {
      // Don't track . files
      if (spath->file_name[0] == '.') return 0;

      // Using git on large files is a miserable failure
      struct stat sb;
      if (lstat(spath->full_path, &sb) != 0) {
        error("Error stat()ing %s for commit\n", spath->full_path);
        break;
      }

      // No empty files
      if (sb.st_size == 0) break;

      // No huge files
      if (sb.st_size > GIT_TRACK_MAX) break;

      // Command string
      snprintf(cmd, sizeof(cmd), "cd %s/%s/files; git add './%s/%s'; git commit --author='%s <support@hubzero.org>' -m '[SFTP] added or modified file %s' './%s/%s' >> %s 2>&1",
                     source_dir, spath->repo_name, spath->file_path,
                     spath->file_name,
                     passwd->pw_name, spath->file_name, spath->file_path,
                     spath->file_name, GIT_LOG);
      debug("%s", cmd);

      // Do it.
      ret = system(cmd);
      if (ret != 0) {
        error("git(): Failed executing '%s', return %d errno %d\n", cmd, ret,
              errno);
      }
      break;
    }

    case GIT_MV: {
      // Again, no . files
      if (spath->file_name[0] == '.' || dpath->file_name[0] == '.') return 0;

      // TEH COMMAND
      snprintf(cmd, sizeof(cmd), "cd %s/%s/files; git mv './%s/%s' './%s/%s'; git commit --author='%s <support@hubzero.org>' -m '[SFTP] moved %s to %s' './%s/%s' './%s/%s' >> %s 2>&1",
                     source_dir, dpath->repo_name, spath->file_path,
                     spath->file_name, dpath->file_path, dpath->file_name,
                     passwd->pw_name, spath->file_name,
                     dpath->file_name, spath->file_path, spath->file_name,
                     dpath->file_path, dpath->file_name, GIT_LOG);
      debug("%s", cmd);

      // Wheee
      ret = system(cmd);
      if (ret != 0) {
        error("git(): Failed executing '%s', return %d errno %d\n", cmd, ret,
              errno);
      }
      break;
    }

    default: {
      // Another superfluous comment
      error("git(): How'd we get here?");
      break;
    }
  }

  return 0;
}

