# @package      hubzero-python
# @file         webconfig.py
# @author       David Benham <dbenham@purdue.edu>
# @copyright    Copyright (c) 2012-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#


import ConfigParser
import hubzero.config.hubzerositeconfig
import hubzero.data.db
import re
import os
import json


def getDefaultSite():
	return hubzero.config.hubzerositeconfig.getHubzeroConfigOption("DEFAULT", "site")
	

def getWebConfigOption(optionName, hubName = ""):
	
	if not hubName: hubName = getDefaultSite()

	wwwDocRoot = hubzero.config.hubzerositeconfig.getHubzeroConfigOption(hubName, "DocumentRoot")	
	
	# read config values from file
	file = open(wwwDocRoot + "/configuration.php")
	configValues = file.read()
	file.close()
	
	# hubconfig might still be around, if so, dump this file into our search string... we're gonna assume no variable
	if os.path.exists(wwwDocRoot + "/hubconfiguration.php"):
		file = open(wwwDocRoot + "/hubconfiguration.php")
		hubconfigFileContents = file.read()
		file.close()
		configValues += hubconfigFileContents
	
	# grep for right line
	regEx = "(?:var|public)\s*\$" + optionName + "\s*=\s*'(.*)';"
	m = re.search(regEx, configValues, re.MULTILINE | re.IGNORECASE)

	# return whatever we find, if not found, this will be none type
	if m:
		return m.group(1)
	else:
		return ""
		    

def getWebConfigDBHost(hubName = ""):
	"""
	If hubName is not specified, getWebConfigOption returns value for the default hubName
	"""

	if not hubName:
		return getWebConfigOption("host")
	else:
		return getWebConfigOption("host", hubName)


def getWebConfigDBPassword(hubName = ""):
	"""
	If hubName is not specified, getWebConfigOption returns value for the default hubName
	"""

	if not hubName:
		return getWebConfigOption("password")
	else:
		return getWebConfigOption("password", hubName)


def getWebConfigDBUsername(hubName = ""):
	"""
	If hubName is not specified, getWebConfigOption returns value for the default hubName
	"""

	if not hubName: 
		return getWebConfigOption("user")
	else:
		return getWebConfigOption("user",  hubName)


def getWebConfigDBName(hubName = ""):
	"""
	If hubName is not specified, getWebConfigOption returns value for the default hubName
	"""

	if not hubName: 
		return getWebConfigOption("db")
	else:
		return getWebConfigOption("db", hubName)


def _writeParamField(componentOption, NewParamValueDictionary, hubName):
	"""
	Take a dictionary, transform it to properly formatted params string for Joomla, and 
	update the params field of the jos_components table for the specified component
	"""
	# database config
	dbHost = getWebConfigDBHost(hubName)
	dbPW = getWebConfigDBPassword(hubName)
	dbUserName = getWebConfigDBUsername(hubName)
	dbName = getWebConfigDBName(hubName)	
	db =  hubzero.data.db.MySQLConnection(dbHost, dbName, dbUserName, dbPW)

	try:
		parmString = json.dumps(NewParamValueDictionary)
		sql = "UPDATE jos_extensions set params = %s WHERE type='component' AND `element` = %s AND client_id=1"
		data = [parmString, componentOption]
		db.query_rowcount(sql, data)
	except:
		parmString = ""
		for parm in NewParamValueDictionary:
			parmString += parm + "=" + NewParamValueDictionary[parm] + "\n"
		sql = "UPDATE jos_components set params = %s WHERE `option` = %s"
		data = [parmString, componentOption]
		db.query_rowcount(sql, data)


def _writePluginParamField(folder, element, NewParamValueDictionary, hubName):
	"""
	Take a dictionary, transform it to properly formatted params string for Joomla, and 
	update the params field of the jos_extensions table for the specified plugin
	"""
	# database config
	dbHost = getWebConfigDBHost(hubName)
	dbPW = getWebConfigDBPassword(hubName)
	dbUserName = getWebConfigDBUsername(hubName)
	dbName = getWebConfigDBName(hubName)	
	db =  hubzero.data.db.MySQLConnection(dbHost, dbName, dbUserName, dbPW)

	try:
		parmString = json.dumps(NewParamValueDictionary)
		sql = "UPDATE `jos_extensions` SET `params`=%s WHERE `type`='plugin' AND `folder`=%s AND `element`=%s"
		data = [parmString, folder, element]
		db.query_rowcount(sql, data)
	except:
		parmString = ""
		for parm in NewParamValueDictionary:
			parmString += parm + "=" + NewParamValueDictionary[parm] + "\n"
		sql = "UPDATE `jos_plugins` SET `params`=%s WHERE `folder`=%s AND `element`=%s"
		data = [parmString, folder, element]
		db.query_rowcount(sql, data)


def _writeModuleParamField(module, NewParamValueDictionary, hubName):
        """
        Take a dictionary, transform it to properly formatted params string for Joomla, and 
        update the params field of the jos_extensions table for the specified plugin
        """
        # database config
	dbHost = getWebConfigDBHost(hubName)
        dbPW = getWebConfigDBPassword(hubName)
        dbUserName = getWebConfigDBUsername(hubName)
        dbName = getWebConfigDBName(hubName)
        db =  hubzero.data.db.MySQLConnection(dbHost, dbName, dbUserName, dbPW)

        try:
                parmString = json.dumps(NewParamValueDictionary)
                sql = "UPDATE `jos_modules` SET `params`=%s WHERE `title`=%s"
	        data = [parmString, module]
        	db.query_rowcount(sql, data)
        except:
                parmString = ""
                for parm in NewParamValueDictionary:
                        parmString += parm + "=" + NewParamValueDictionary[parm] + "\n"
                sql = "UPDATE `jos_modules` SET `params`=%s WHERE `title`=%s"
	        data = [parmString, module]
        	db.query_rowcount(sql, data)


def _writeTemplateParamField(template, NewParamValueDictionary, hubName):
        """
        Take a dictionary, transform it to properly formatted params string for Joomla, and 
        update the params field of the jos_template_styles table for the specified template
        """
        # database config
	dbHost = getWebConfigDBHost(hubName)
        dbPW = getWebConfigDBPassword(hubName)
        dbUserName = getWebConfigDBUsername(hubName)
        dbName = getWebConfigDBName(hubName)
        db =  hubzero.data.db.MySQLConnection(dbHost, dbName, dbUserName, dbPW)

        parmString = json.dumps(NewParamValueDictionary)
        sql = "UPDATE `jos_template_styles` SET `params`=%s WHERE `template`=%s"
        data = [parmString, template]
        db.query_rowcount(sql, data)


def _buildDictionary(componentOption, hubName):
	"""
	Put the parameter string of the database into a dictionary
	String follows the format:
	
	param1=value1
	param2=value2
	param3=value3
	
	"""
	# database config
	dbHost = getWebConfigDBHost(hubName)
	dbPW = getWebConfigDBPassword(hubName)
	dbUserName = getWebConfigDBUsername(hubName)
	dbName = getWebConfigDBName(hubName)	

	db =  hubzero.data.db.MySQLConnection(dbHost, dbName, dbUserName, dbPW)
	
	# look up parm string and parse it on newline, for each line, parse it further based on *first* '='
	try:
		sql = "SELECT `params` from jos_extensions WHERE type='component' AND `element`=%s AND client_id=1"
		data = [componentOption]
		paramsString = db.query_selectscalar(sql, data)
	except:
		sql = "SELECT `params` from jos_components WHERE `option`=%s"
		data = [componentOption]
		paramsString = db.query_selectscalar(sql, data)

	if paramsString is None:
		return {}
	
	if paramsString.startswith('{'):
		try:
			parmsDict = json.loads(paramsString)
			return parmsDict
		except:
			pass

	parmsDict = {}
	parms = paramsString.split("\n")
	for parmline in parms:
		if parmline: # in case of any blank lines in the params field
			n,v = parmline.split("=", 1)
			parmsDict[n] = v

	return parmsDict


def _buildPluginDictionary(folder, element, hubName):
	"""
	Put the parameter string of the database into a dictionary
	String follows the format:
	
	param1=value1
	param2=value2
	param3=value3
	
	"""
	# database config
	dbHost = getWebConfigDBHost(hubName)
	dbPW = getWebConfigDBPassword(hubName)
	dbUserName = getWebConfigDBUsername(hubName)
	dbName = getWebConfigDBName(hubName)	
	db =  hubzero.data.db.MySQLConnection(dbHost, dbName, dbUserName, dbPW)
	
	# look up parm string and parse it on newline, for each line, parse it further based on *first* '='
	try:
		sql = "SELECT `params` from jos_extensions WHERE type='plugin' AND folder=%s AND element=%s"
		data = [folder,element]
		paramsString = db.query_selectscalar(sql, data)
	except:
		sql = "SELECT `params` from jos_plugins WHERE `folder`=%s AND `element`=%s"
		data = [folder,element]
		paramsString = db.query_selectscalar(sql, data)


	if paramsString.startswith('{'):
		try:
			parmsDict = json.loads(paramsString)
			return parmsDict
		except:
			pass

	parmsDict = {}

        if paramsString:
		parms = paramsString.split("\n")
		for parmline in parms:
			if parmline: # in case of any blank lines in the params field
				n,v = parmline.split("=", 1)
				parmsDict[n] = v

	return parmsDict
	

def _buildModuleDictionary(module, hubName):
        """
        Put the parameter string of the database into a dictionary
        String follows the format:
        
        param1=value1
        param2=value2
        param3=value3
        
        """
        # database config
	dbHost = getWebConfigDBHost(hubName)
        dbPW = getWebConfigDBPassword(hubName)
        dbUserName = getWebConfigDBUsername(hubName)
        dbName = getWebConfigDBName(hubName)
        db =  hubzero.data.db.MySQLConnection(dbHost, dbName, dbUserName, dbPW)

        # look up parm string and parse it on newline, for each line, parse it further based on *first* '='
        sql = "SELECT `params` from jos_modules WHERE title=%s"

        data = [module]
        paramsString = db.query_selectscalar(sql, data)

        if paramsString.startswith('{'):
                try:
                        parmsDict = json.loads(paramsString)
                        return parmsDict
                except:
                        pass

        parmsDict = {}

        if paramsString:
                parms = paramsString.split("\n")
                for parmline in parms:
                        if parmline: # in case of any blank lines in the params field
                                n,v = parmline.split("=", 1)
                                parmsDict[n] = v

        return parmsDict


def _buildTemplateDictionary(template, hubName):
        """
        Put the parameter string of the database into a dictionary
        String follows the format:
        
        param1=value1
        param2=value2
        param3=value3
        
        """
        # database config
	dbHost = getWebConfigDBHost(hubName)
        dbPW = getWebConfigDBPassword(hubName)
        dbUserName = getWebConfigDBUsername(hubName)
        dbName = getWebConfigDBName(hubName)
        db =  hubzero.data.db.MySQLConnection(dbHost, dbName, dbUserName, dbPW)

        # look up parm string and parse it on newline, for each line, parse it further based on *first* '='
        sql = "SELECT `params` from jos_template_styles WHERE template=%s"

        data = [template]
        paramsString = db.query_selectscalar(sql,data)

        if paramsString.startswith('{'):
                try:
                        parmsDict = json.loads(paramsString)
                        return parmsDict
                except:
                        pass

        parmsDict = {}

        if paramsString:
                parms = paramsString.split("\n")
                for parmline in parms:
                        if parmline: # in case of any blank lines in the params field
                                n,v = parmline.split("=", 1)
                                parmsDict[n] = v

        return parmsDict


def getComponentParam(componentOption, parameterName, hubName = ""):
	"""
	Get individual param value for a component and return to caller
	"""
	paramDict = _buildDictionary(componentOption, hubName)
	
	if paramDict.has_key(parameterName):
		return paramDict[parameterName]
	else:
		return None	


def getPluginParam(folder,element, parameterName, hubName = ""):
	"""
	Get individual param value for a component and return to caller
	"""
	paramDict = _buildPluginDictionary(folder,element, hubName)
	
	if paramDict.has_key(parameterName):
		return paramDict[parameterName]
	else:
		return None	


def addComponentParam(componentOption, parameterName, parameterValue, allowOverwrite=True, hubName = ""):
	"""
	Add value to the param string in the jos_components table for the specified component.
	"""
	paramDict = _buildDictionary(componentOption, hubName)

	if allowOverwrite:
		paramDict[parameterName] = parameterValue
	else:
		if paramDict.has_key(parameterName):
			raise Exception("ERROR: addCompoentParam called for existing parameter with allowOverwrite = false")
		else:
			paramDict[parameterName] = parameterValue

	_writeParamField(componentOption, paramDict, hubName)	


def addPluginParam(folder, element, parameterName, parameterValue, allowOverwrite=True, hubName = ""):
	"""
	Add value to the param string in the jos_components table for the specified component.
	"""
	paramDict = _buildPluginDictionary(folder,element,hubName)

	if allowOverwrite:
		paramDict[parameterName] = parameterValue
	else:
		if paramDict.has_key(parameterName):
			raise Exception("ERROR: addPluginParam called for existing parameter with allowOverwrite = false")
		else:
			paramDict[parameterName] = parameterValue

	_writePluginParamField(folder,element, paramDict, hubName)	

def addModuleParam(module, parameterName, parameterValue, allowOverwrite=True, hubName = ""):
	"""
	Add value to the param string in the jos_components table for the specified component.
	"""
	paramDict = _buildModuleDictionary(module,hubName)

	if allowOverwrite:
		paramDict[parameterName] = parameterValue
	else:
		if paramDict.has_key(parameterName):
			raise Exception("ERROR: addModuleParam called for existing parameter with allowOverwrite = false")
		else:
			paramDict[parameterName] = parameterValue

	_writeModuleParamField(module, paramDict, hubName)	


def addTemplateParam(template, parameterName, parameterValue, allowOverwrite=True, hubName = ""):
        """
        Add value to the param string in the jos_components table for the specified component.
        """
        paramDict = _buildTemplateDictionary(template,hubName)

        if allowOverwrite:
                paramDict[parameterName] = parameterValue
        else:
                if paramDict.has_key(parameterName):
                        raise Exception("ERROR: addModuleParam called for existing parameter with allowOverwrite = false")
                else:
                        paramDict[parameterName] = parameterValue

        _writeTemplateParamField(template, paramDict, hubName)

	
def delComponentParam(componentOption, parameterName, hubName = ""):
	"""
	Delete value from the param string in the in the jos_extensions table for the specified component
	"""
	paramDict = _buildDictionary(componentOption, hubName)
	del paramDict[parameterName]
	_writeParamField(componentOption, paramDict, hubName)


def delPluginParam(folder, element, parameterName, hubName = ""):
	"""
	Delete value from the param string in the in the jos_extensions table for the specified plugin
	"""
	paramDict = _buildPluginDictionary(folder, element, hubName)
	del paramDict[parameterName]
	_writePluginParamField(folder, element, paramDict, hubName)
	
	
def setPluginEnabled(pluginFolder, pluginElement, enable, hubName = ""):
	"""
	Enable or disable a plugin by setting the published field for the plugin in the jos_plugins table
	"""
	
	# database config
	dbHost = getWebConfigDBHost(hubName)
	dbPW = getWebConfigDBPassword(hubName)
	dbUserName = getWebConfigDBUsername(hubName)
	dbName = getWebConfigDBName(hubName)	
	db =  hubzero.data.db.MySQLConnection(dbHost, dbName, dbUserName, dbPW)
		
	try:
		sql = "UPDATE jos_extensions set enabled = %s WHERE type='plugin' AND `folder` = %s AND `element` = %s"
		data = [enable, pluginFolder, pluginElement]
		db.query_rowcount(sql, data)	
	except:
		sql = "UPDATE jos_plugins set published = %s WHERE `folder` = %s AND `element` = %s"
		data = [enable, pluginFolder, pluginElement]
		db.query_rowcount(sql, data)	


def getPluginEnabled(pluginFolder, pluginElement, hubName = ""):
	"""
	Get published field for the given plugin in the jos_plugins table
	"""
	
	# database config
	dbHost = getWebConfigDBHost(hubName)
	dbPW = getWebConfigDBPassword(hubName)
	dbUserName = getWebConfigDBUsername(hubName)
	dbName = getWebConfigDBName(hubName)	
	db =  hubzero.data.db.MySQLConnection(dbHost, dbName, dbUserName, dbPW)
		
	try:
		sql = "SELECT enabled FROM jos_extensions  WHERE type='plugins' AND `folder` = %s AND `element` = %s"
		data = [pluginFolder, pluginElement]
		published = db.query_selectscalar(sql, data)
	except:
		sql = "SELECT published FROM jos_plugins WHERE `folder` = %s AND `element` = %s"
		data = [pluginFolder, pluginElement]
		published = db.query_selectscalar(sql, data)

	return published


def getHubSiteConfig(hubname, sectionName, valueName):
	"""
	get a value from the /etc/hubzero-cms/HUBNAME.conf file
	"""
	filename = "/etc/hubzero-cms/" + hubname + ".conf"
	
	if not os.path.exists(filename):
		print "Cannot find " + filename
	
	hubzeroCMSConfig = ConfigParser.ConfigParser()
	hubzeroCMSConfig.readfp(open(filename))
	
	try:
		rv = hubzeroCMSConfig.get(sectionName, valueName)
	except ConfigParser.NoOptionError:
		rv = ''
        except ConfigParser.NoSectionError:
                rv = ''
	
	return rv

	
def getCMSversion(document_root = None):

	if (document_root == None):
		default_site = hubzero.config.hubzerositeconfig.getHubDefaultSite()
		document_root = hubzero.config.hubzerositeconfig.getHubzeroConfigOption(default_site, 'documentroot')

        # 2.1.1 
	if os.path.isfile(document_root + "/core/bootstrap/version.php"):
		f = open(document_root + "/core/bootstrap/version.php", 'r')
		file_text =  f.read()
		p = re.compile(r"HVERSION['\"]+\s*,\s*['\"]+\s*(.*?)\s*['\"]+", re.MULTILINE)
		m = p.search(file_text)
		if not m:
			return None
		else:
			return m.group(1)

	# 2.1.0 look in the vendor hubzero framework (this will change in 2.1.1)
	if os.path.isfile(document_root + "/vendor/hubzero/framework/src/Version/Version.php"):
		f = open(document_root + "/vendor/hubzero/framework/src/Version/Version.php", 'r')
		file_text =  f.read()
		p = re.compile(r"VERSION\s*=\s*['\"]+\s*(.*?)\s*['\"]+", re.MULTILINE)
		m = p.search(file_text)
		if not m:
			return None
		else:
			return m.group(1)

	# 2.0.0 should actually look in core/libraries/Hubzero/Version/Version.php but the regex would be different
	if os.path.isfile(document_root + "/core/libraries/cms.php"):
		f = open(document_root + "/core/libraries/cms.php", 'r')
		file_text =  f.read()
		p = re.compile(r"HVERSION['\"]+\s*,\s*['\"]+\s*(.*?)\s*['\"]+", re.MULTILINE)
		m = p.search(file_text)
		if not m:
			return None
		else:
			return m.group(1)

	# 1.2.0,1.2.1,1.2.2,1.2.3,1.3.0,1.3.1
	if os.path.isfile(document_root + "/libraries/cms.php"):
		f = open(document_root + "/libraries/cms.php", 'r')
		file_text =  f.read()
		p = re.compile(r"HVERSION['\"]+\s*,\s*['\"]+\s*(.*?)\s*['\"]+", re.MULTILINE)
		m = p.search(file_text)
		if not m:
			return None
		else:
			return m.group(1)

	elif os.path.isdir(document_root + "/libraries/Hubzero/Oauth"):
		return "1.1.0"
	elif os.path.isdir(document_root + "/libraries/Hubzero/Session"):
		return "1.0.0"
	elif not os.path.isdir(document_root + "/libraries/Hubzero"):
		return "0.8.0"


