/**
 *
 * @file     parser.h
 * @author   Aleix Conchillo Flaque <aleix@member.fsf.org>
 * @date     Mon Nov 25, 2002 00:57
 * @brief    SCEW parser type declaration
 *
 * $Id: parser.h,v 1.2 2004/05/25 20:23:05 aleix Exp $
 *
 * @if copyright
 *
 * Copyright (C) 2002, 2003, 2004 Aleix Conchillo Flaque
 *
 * SCEW is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * SCEW is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * @endif
 *
 * These are the parser functions that allow to read an XML tree from a
 * file or a memory buffer.
 */


#ifndef PARSER_H_ALEIX0211250057
#define PARSER_H_ALEIX0211250057

#include "types.h"

#include <expat.h>

#include <stdio.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/**
 * Creates a new parser. The parser is needed to load XML documents.
 */
extern scew_parser*
scew_parser_create();

/**
 * Frees a parser memory structure. This function will <b>not</b> free
 * the <code>scew_tree</code> generated by the parser, so it is
 * important that you keep a pointer to it and remember to free it.
 *
 * @see scew_tree_free
 */
extern void
scew_parser_free(scew_parser* parser);

/**
 * Loads an XML tree from the specified file name using the given
 * parser.
 *
 * @param parser the SCEW parser.
 * @param file_name the file to load the XML from.
 *
 * @see scew_parser_create
 *
 * @return 1 if file was successfully loaded, 0 otherwise.
 */
extern unsigned int
scew_parser_load_file(scew_parser* parser, char const* file_name);

/**
 * Loads an XML tree from the specified file pointer using the
 * given parser.
 *
 * @param parser the SCEW parser.
 * @param in the file pointer to load the XML from.
 *
 * @see scew_parser_create
 *
 * @return 1 if file was successfully loaded, 0 otherwise.
 */
extern unsigned int
scew_parser_load_file_fp(scew_parser* parser, FILE* in);

/**
 * Loads an XML tree from the specified memory buffer of the specified
 * size using the given parser.
 *
 * @param parser the SCEW parser.
 * @param buffer memory buffer to load XML from.
 * @param size size in bytes of the memory buffer.
 *
 * @see scew_parser_create
 *
 * @return 1 if buffer was successfully loaded, 0 otherwise.
 */
extern unsigned int
scew_parser_load_buffer(scew_parser* parser, char const* buffer,
                        unsigned int size);

/**
 * Loads an XML tree from the specified stream buffer. Will call the
 * callback (set using scew_parser_set_stream_callback) at the end of
 * each message.
 *
 * @param parser the SCEW parser.
 * @param buffer memory buffer to load XML from.
 * @param size size in bytes of the memory buffer.
 *
 * @see scew_parser_create
 * @see scew_parser_set_stream_callback
 *
 * @return 1 if buffer was successfully loaded, 0 otherwise.
 */
extern unsigned int
scew_parser_load_stream(scew_parser* parser, char const* buffer,
                        unsigned int size);

/**
 * Sets the callback for use when reading streams.
 *
 * @param parser the SCEW parser
 * @param cb the callback function
 */
void
scew_parser_set_stream_callback(scew_parser* parser, SCEW_CALLBACK* cb);

/**
 * Returns the XML tree read by the parser. Remember that
 * <code>scew_parser_free</code> does not free the
 * <code>scew_tree</code> read.
 *
 * @see tree.h
 */
extern scew_tree*
scew_parser_tree(scew_parser const* parser);

/**
 * Returns the internal Expat parser. Probably some low-level Expat
 * functions need to be called. This function gives you access to the
 * Expat parser so you will be able to call those functions. If you
 * modify the Expat parser event handling routines, SCEW will not be
 * able to load the XML tree.
 */
extern XML_Parser
scew_parser_expat(scew_parser* parser);

/**
 * Tells the parser how to treat white spaces. The default is to ignore
 * heading and trailing white spaces.
 *
 * There is a new section in XML specification which talks about how to
 * handle white spaces in XML. One can set an optional attribtue to an
 * element, this attribute is called 'xml:space', and it can be set to
 * 'default' or 'preserve', and it inherits its value from parent
 * elements. 'preserve' means to leave white spaces as their are found,
 * and 'default' means that white spaces are handled by the XML
 * processor (Expat in our case) the way it wants to.
 *
 * This function gives the possibility to change the XML processor
 * behaviour.
 *
 * @param parser the parser to set the option to.
 * @param ignore 0 if you do <b>not</b> want to ignore white spaces, any
 * other value otherwise.
 */
extern void
scew_parser_ignore_whitespaces(scew_parser* parser, int ignore);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* PARSER_H_ALEIX0211250057 */
