# @package      hubzero-submit-client
# @file         LocalWorkflowPEGASUS.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2014 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012-2014 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import os
import re
import logging

from Pegasus.DAX3 import *

class LocalWorkflowPEGASUS:
   def __init__(self,
                hubUserName,
                hubUserId,
                runName,
                jobDirectory,
                scratchDirectory,
                useSetup,
                pegasusVersion,
                pegasusHome,
                localJobId,
                instanceId,
                pegasusTemplates,
                timeHistoryLogs):
      self.logger           = logging.getLogger(__name__)
      self.hubUserName      = hubUserName
      self.hubUserId        = hubUserId
      self.runName          = runName
      self.jobDirectory     = jobDirectory
      self.scratchDirectory = scratchDirectory
      self.useSetup         = useSetup
      self.pegasusVersion   = pegasusVersion
      self.pegasusHome      = pegasusHome
      self.localJobId       = localJobId
      if int(instanceId) > 0:
         self.instanceId    = instanceId
      else:
         self.instanceId    = None

      self.daxPath          = os.path.join(self.jobDirectory,"%s.dax" % (self.localJobId))
      self.executable       = 'pegasus-plan'
      self.arguments        = "--dax %s" % (self.daxPath)

      self.pegasusTemplates = pegasusTemplates
      self.timestampStart   = os.path.join(self.jobDirectory,timeHistoryLogs['timestampStart'])
      self.timestampFinish  = os.path.join(self.jobDirectory,timeHistoryLogs['timestampFinish'])
      self.timeResults      = os.path.join(self.jobDirectory,timeHistoryLogs['timeResults'])


   def __makeWorkflowTemplate(self):
      return """#!/bin/sh
# LocalWorkflowPEGASUS:makeWorkflowTemplate
#
exitStatus=0

. USESETUP
use -e -r pegasus-PEGASUSVERSION

mkdir -p INSTANCEDIRECTORY/work
mkdir -p SCRATCHDIRECTORY/work

TIMEPATH=
for timePath in ${HOME}/bin/time /usr/bin/time /usr/local/bin/time /apps/submit/bin/time ; do
   if [ -x ${timePath} ] ; then
      TIMEPATH=${timePath}
      break
   fi
done

#Discarded Arguments      = DISCARDED
#Not Recognized Arguments = NOTRECOGNIZED

date +"%s" > TS_START

${TIMEPATH} --format "Command exited with status %x\\nreal %e\\nuser %U\\nsys %S" -o TIME_RESULTS \\
             EXECUTABLE --conf INSTANCEDIRECTORY/JOBID.pegasusrc \\
             --sites local \\
             --dir SCRATCHDIRECTORY \\
             --relative-dir scratch \\
             --relative-submit-dir work \\
             --output local \\
             ARGUMENTS \\
             --quiet \\
             --submit < /dev/null 2> INSTANCEDIRECTORY/RUNNAME.stderr
exitStatus=$?

date +"%s" > TS_FINISH

exit ${exitStatus}
"""


   def __buildWorkflowScript(self):
      # setup regex's for the template
      re_useSetup               = re.compile("USESETUP")
      re_pegasusVersion         = re.compile("PEGASUSVERSION")
      re_jobId                  = re.compile("JOBID")
      re_runName                = re.compile("RUNNAME")
      re_instanceDirectory      = re.compile("INSTANCEDIRECTORY")
      re_scratchDirectory       = re.compile("SCRATCHDIRECTORY")
      re_executable             = re.compile("EXECUTABLE")
      re_arguments              = re.compile("ARGUMENTS")
      re_discardedArguments     = re.compile("DISCARDED")
      re_notRecognizedArguments = re.compile("NOTRECOGNIZED")
      re_tsStart                = re.compile("TS_START")
      re_tsFinish               = re.compile("TS_FINISH")
      re_timeResults            = re.compile("TIME_RESULTS")
      re_hubUserName            = re.compile("HUBUSERNAME")
      re_hubUserId              = re.compile("HUBUSERID")

      template = self.__makeWorkflowTemplate()

      template = re_useSetup.sub(self.useSetup,template)
      template = re_pegasusVersion.sub(self.pegasusVersion,template)
      template = re_jobId.sub(self.localJobId,template)
      template = re_runName.sub(self.runName,template)
      template = re_instanceDirectory.sub(self.jobDirectory,template)
      template = re_scratchDirectory.sub(self.scratchDirectory,template)
      template = re_executable.sub(self.executable,template)
      template = re_tsStart.sub(self.timestampStart,template)
      template = re_tsFinish.sub(self.timestampFinish,template)
      template = re_timeResults.sub(self.timeResults,template)

      notRecognizedArguments = []
      discardedArguments     = []
      auxiliaryArguments     = []
      userArguments = self.arguments.split()
      for userArgument in userArguments:
         auxiliaryArguments.append(userArgument)

      if 'rc' in self.pegasusTemplates:
         searchString = os.path.basename(self.executable) + '.arguments'
         settings = self.pegasusTemplates['rc'].split('\n')
         for setting in settings:
#pegasus-plan.arguments = --nocleanup
            if setting.count(searchString) > 0:
               try:
                  parameter,value = setting.split('=')
                  auxiliaryArguments.append(value.strip())
               except:
                  pass

      arguments = ' '.join(auxiliaryArguments)
      template  = re_arguments.sub(arguments.strip(),template)
      arguments = ' '.join(discardedArguments)
      template  = re_discardedArguments.sub(arguments.strip(),template)
      arguments = ' '.join(notRecognizedArguments)
      template  = re_notRecognizedArguments.sub(arguments.strip(),template)
      template  = re_hubUserName.sub(self.hubUserName,template)
      template  = re_hubUserId.sub(str(self.hubUserId),template)

      return(template)


   def __buildCatalogs(self):
      # setup regex's for the template
      re_jobId             = re.compile("JOBID")
      re_instanceId        = re.compile("INSTANCEID")
      re_instanceIdEmpty   = re.compile("_INSTANCEID")
      re_instanceDirectory = re.compile("INSTANCEDIRECTORY")
      re_scratchDirectory  = re.compile("SCRATCHDIRECTORY")
      re_pegasusHome       = re.compile("PEGASUSHOME")

      for templateType in self.pegasusTemplates:
         template = self.pegasusTemplates[templateType]
         template = re_jobId.sub(self.localJobId,template)
         if self.instanceId:
            template = re_instanceId.sub(self.instanceId,template)
         else:
            template = re_instanceIdEmpty.sub("",template)
         template = re_instanceDirectory.sub(self.jobDirectory,template)
         template = re_scratchDirectory.sub(self.scratchDirectory,template)
         template = re_pegasusHome.sub(self.pegasusHome,template)
         if   templateType == 'rc':
            pegasusFile = "%s.pegasusrc" % (self.localJobId)
         elif templateType == 'sites':
            pegasusFile = "%s_sites.xml" % (self.localJobId)
         pegasusPath = os.path.join(self.jobDirectory,pegasusFile)
         fpPegasusFile = open(pegasusPath,'w')
         if fpPegasusFile:
            fpPegasusFile.write(template)
            fpPegasusFile.close()


   def __buildDAX(self):
      instanceIds = []
      dirFiles = os.listdir(self.jobDirectory)
      for dirFile in dirFiles:
         dirPath = os.path.join(self.jobDirectory,dirFile)
         if os.path.isdir(dirPath):
            instanceIds.append(dirFile)
      instanceIds.sort()

      nInstances = len(instanceIds)
      dax = ADAG(self.localJobId)

      reAppScriptPath  = re.compile("[0-9_]+.sh")
      for instanceId in instanceIds:
         instanceDirectory = os.path.join(self.jobDirectory,instanceId)
         dirFiles = os.listdir(instanceDirectory)
         appScriptFiles  = filter(reAppScriptPath.search,dirFiles)

         if len(appScriptFiles) == 1:
            appScriptName = appScriptFiles[0]
            appScriptPath = os.path.join(instanceDirectory,appScriptName)

            appScriptPath = "file://" + appScriptPath
            appScript = Executable(name=appScriptName,arch=Arch.X86_64,os=OS.LINUX,installed=True)
            appScript.addPFN(PFN(appScriptPath,"local"))
            dax.addExecutable(appScript)

            jobId = "%s_%s" % (self.runName,instanceId)
            job = Job(name=appScriptName,id=jobId)
            dax.addJob(job)

      fpDAX = open(self.daxPath,'w')
      if fpDAX:
         dax.writeXML(fpDAX)
         fpDAX.close()


   def buildWorkflowScript(self):
      workflowScriptName = "%s.pegasus" % (self.localJobId)
      workflowScript = self.__buildWorkflowScript()

      self.__buildCatalogs()
      self.__buildDAX()

      return(workflowScriptName,workflowScript)


