# @package      hubzero-submit-common
# @file         RemoteProbeMonitor.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2013 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012-2013 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

from hubzero.submit.LogMessage  import logID as log
from hubzero.submit.MessageCore import MessageCore

class RemoteProbeMonitor(MessageCore):
   def __init__(self,
                host,
                port,
                repeatDelay=5,
                fixedBufferSize=128,
                ignoreProbeSites=[]):
      MessageCore.__init__(self,listenerHost=host,listenerPort=port,repeatDelay=repeatDelay)
      self.fixedBufferSize = fixedBufferSize
      self.results = {}
      self.ignoreProbeSites = []
      for site in ignoreProbeSites:
         self.ignoreProbeSites.append(site)


   def queryProbeSiteStatus(self,
                            siteName):
      timeEvaluated = ""
      status        = ""
      waitRank      = ""

      queryMessage = "Q:" + siteName
      nTry,response = self.requestMessageResponse(queryMessage,
                                                  self.fixedBufferSize,
                                                  self.fixedBufferSize,
                                                  True)

      if nTry > 1:
         log("confirmation: Q(%d):%s" % (nTry,response))

      if response != "":
         timeEvaluated,status,waitRank = response.split()

      return(timeEvaluated,status,waitRank)


   def isSiteAvailable(self,
                       queueAtSite):
      siteName = queueAtSite.split('@')[-1]
      if   siteName in self.results:
         siteAvailable = self.results[siteName]
      elif siteName in self.ignoreProbeSites:
         siteAvailable = True
         self.results[siteName] = siteAvailable
      else:
         timeEvaluated,status,waitRank = self.queryProbeSiteStatus(siteName)
         siteAvailable = (status == "PASSED")
         self.results[siteName] = siteAvailable

      return(siteAvailable)


   def postProbeSiteUpdate(self,
                           site,
                           timeEvaluated,
                           status,
                           waitTime):
      queryMessage = "U:" + ",".join([site,timeEvaluated,status,waitTime])
      nTry,response = self.requestMessageResponse(queryMessage,
                                                  self.fixedBufferSize,
                                                  self.fixedBufferSize,
                                                  True)

      if nTry > 1:
         log("confirmation: U(%d):%s" % (nTry,response))


   def reportProbeSiteStatus(self):
      report = ""

      queryMessage = "R:"
      nTry,report = self.requestMessageVariableResponse(queryMessage,
                                                        self.fixedBufferSize,
                                                        self.fixedBufferSize,
                                                        True)

      if nTry > 1:
         log("confirmation: R(%d): %d" % (nTry,len(report)))

      return(report)


   def purgeOfflineSites(self,
                         siteNames):
      markedForDeletion = []
      for siteName in siteNames:
         if not self.isSiteAvailable(siteName):
            markedForDeletion.append(siteName)
      for siteName in markedForDeletion:
         siteNames.remove(siteName)
      del markedForDeletion


