#
# Copyright (c) 2004-2010 Purdue University All rights reserved.
# 
# Developed by: HUBzero Technology Group, Purdue University
#               http://hubzero.org
# 
# HUBzero is free software: you can redistribute it and/or modify it under the terms of the
# GNU Lesser General Public License as published by the Free Software Foundation, either
# version 3 of the License, or (at your option) any later version.
# 
# HUBzero is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
# without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU Lesser General Public License for more details.  You should have received a
# copy of the GNU Lesser General Public License along with HUBzero.
# If not, see <http://www.gnu.org/licenses/>.
# 
# GNU LESSER GENERAL PUBLIC LICENSE
# Version 3, 29 June 2007
# Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>
#
import re

from LogMessage  import logID as log
from MessageCore import MessageCore

class JobMonitor(MessageCore):
   def __init__(self,
                host,
                port,
                repeatDelay=5,
                fixedBufferSize=64,
                activeJobDumpPath="monitorJob.dump"):
      MessageCore.__init__(self,bindHost=host,bindPort=port,repeatDelay=repeatDelay)
      self.fixedBufferSize   = fixedBufferSize
      self.activeJobDumpPath = activeJobDumpPath
      self.activeJobs        = {}
      self.activeJobSites    = {}


   def loadActiveJobs(self):
      dumpedJobs = []
      try:
         dumpFile = open(self.activeJobDumpPath,"r")
         dumpedJobs = dumpFile.readlines()
         dumpFile.close()
      except:
         pass

      for dumpedJob in dumpedJobs:
         try:
            globalJobId,jobStatus,jobStage = dumpedJob.split()
         except:
            globalJobId,jobStatus = dumpedJob.split()
            jobStage = "?"

         self.activeJobs[globalJobId] = (jobStatus,jobStage)


   def dumpActiveJobs(self):
      dumpFile = open(self.activeJobDumpPath,"w")
      for activeJob in self.activeJobs.items():
         dumpFile.write("%s %s %s\n" % (activeJob[0],activeJob[1][0],activeJob[1][1]))
      dumpFile.close()


   def addActiveJobSite(self,
                        jobSite,
                        value):
      self.activeJobSites[jobSite] = value


   def deleteActiveJobSite(self,
                           jobSite):
      del self.activeJobSites[jobSite]


   def addActiveJob(self,
                    globalJobId,
                    jobStatus,
                    jobStage):
      self.activeJobs[globalJobId] = (jobStatus,jobStage)


   def markNewActiveJobsAsDone(self,
                               jobSite):
      siteMatch = re.compile(jobSite+":.*")
      newJobs = filter(siteMatch.match,self.activeJobs.keys())
      for newJob in newJobs:
#        log("Potential new job key %s" % (newJob))
         if self.activeJobs[newJob][0] == "N":
            self.activeJobs[newJob] = ("D",self.activeJobs[newJob][1])


   def isJobSiteActive(self,
                       jobSite):
      return(jobSite in self.activeJobSites)


   def processRequest(self,
                      channel):
      channelClosed = False
      newJobSite    = ""
      newJobId      = ""

      message = self.receiveMessage(channel,self.fixedBufferSize)
      if message != "":
         if re.match("[QSRT]:",message):
            try:
               messageType,siteId     = message.split(':')
               messageSite,localJobId = siteId.split()
            except:
               log("Failed QSRT message request: " + message)
               messageType = ""
         else:
            log("Failed message request: " + message)
            messageType = ""

         if   messageType == "Q":                        # job Query
            globalJobId = messageSite + ":" + localJobId
            if globalJobId in self.activeJobs:
               jobStatus,jobStage = self.activeJobs[globalJobId]
            else:
               jobStatus,jobStage = ("?","?")
            if self.sendMessage(channel,jobStatus + " " + jobStage,self.fixedBufferSize) > 0:
               newJobSite = messageSite
         elif messageType == "S":                        # new job Submission
            globalJobId = messageSite + ":" + localJobId
            if globalJobId in self.activeJobs:
               jobStatus,jobStage = self.activeJobs[globalJobId]
            else:
               jobStatus,jobStage = ("N","Job")
               self.activeJobs[globalJobId] = (jobStatus,jobStage)
            self.dumpActiveJobs()
            if self.sendMessage(channel,jobStatus + " " + jobStage,self.fixedBufferSize) > 0:
               newJobSite = messageSite
               newJobId   = localJobId
         elif messageType == "T":                        # Terminate job
            globalJobId = messageSite + ":" + localJobId
            if globalJobId in self.activeJobs:
               jobStatus,jobStage = self.activeJobs[globalJobId]
            else:
               jobStatus,jobStage = ("?","?")
            if self.sendMessage(channel,jobStatus + " " + jobStage,self.fixedBufferSize) > 0:
               if (jobStatus != "?") or (jobStage != "?"):
                  jobStatus,jobStage = ("D","Job")
                  self.activeJobs[globalJobId] = (jobStatus,jobStage)
         elif messageType == "R":                        # Report active jobs
            if localJobId == "0":
               siteMatch = re.compile(messageSite+".*")
               report = ""
               reportJobs = filter(siteMatch.match,self.activeJobs.keys())
               for reportJob in reportJobs:
                  jobStatus,jobStage = self.activeJobs[reportJob]
                  if report == "":
                     report += reportJob.split(":")[1] + " " + jobStatus + " " + jobStage
                  else:
                     report += " : " + reportJob.split(":")[1] + " " + jobStatus + " " + jobStage
            else:
               globalJobId = messageSite + ":" + localJobId
               if globalJobId in self.activeJobs:
                  jobStatus,jobStage = self.activeJobs[globalJobId]
               else:
                  jobStatus,jobStage = ("?","?")
               report = localJobId + " " + jobStatus + " " + jobStage

            if messageSite in self.activeJobSites:
               lastReportTime = self.activeJobSites[messageSite]
            else:
               lastReportTime = 0
            reportLength = len(report)
            if self.sendMessage(channel,str(reportLength) + " " + str(lastReportTime),self.fixedBufferSize) > 0:
               self.sendMessage(channel,report)
      else:
         try:
            channel.close()
            channelClosed = True
         except:
            log("close channel failed")

      return(channelClosed,newJobSite,newJobId)


