#
# Copyright (c) 2004-2010 Purdue University All rights reserved.
# 
# Developed by: HUBzero Technology Group, Purdue University
#               http://hubzero.org
# 
# HUBzero is free software: you can redistribute it and/or modify it under the terms of the
# GNU Lesser General Public License as published by the Free Software Foundation, either
# version 3 of the License, or (at your option) any later version.
# 
# HUBzero is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
# without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU Lesser General Public License for more details.  You should have received a
# copy of the GNU Lesser General Public License along with HUBzero.
# If not, see <http://www.gnu.org/licenses/>.
# 
# GNU LESSER GENERAL PUBLIC LICENSE
# Version 3, 29 June 2007
# Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>
#
import os
import os.path
import re
import random

from LogMessage      import log as log
from GroupMembership import isGroupMember

class SitesInfo:
   def __init__(self,
                infoDirectory,
                sitesFile):
      self.sites           = {}

      sitePattern     = re.compile('(.*\[)(.*)(])')
      keyValuePattern = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile('\s*#.*')
      siteName        = ""

      fp = open(os.path.join(infoDirectory,sitesFile), "r")
      if fp:
         eof = False
         while not eof:
            record = fp.readline()
            if record != "":
               record = commentPattern.sub("",record)
               if   sitePattern.match(record):
                  siteName = sitePattern.match(record).group(2)
                  self.sites[siteName] = {'venues':[], \
                                          'tunnelDesignator':'', \
                                          'siteMonitorDesignator':'', \
                                          'venueMechanism':'', \
                                          'venueGatekeeper':',,'.split(','), \
                                          'remoteUser':'', \
                                          'remoteBatchSystem':'', \
                                          'remoteBatchQueue':'', \
                                          'remoteBatchPartition':'', \
                                          'remoteBatchPartitionSize':'', \
                                          'remoteBatchConstraints':'', \
                                          'remoteBinDirectory':os.path.join('${HOME}','bin'), \
                                          'remoteScratchDirectory':'', \
                                          'remotePpn':'', \
                                          'remoteManager':'', \
                                          'remoteHostAttribute':'', \
                                          'stageFiles':True, \
                                          'passUseEnvironment':False, \
                                          'arbitraryExecutableAllowed':True, \
                                          'members':[], \
                                          'state':'enabled', \
                                          'failoverSite':'None', \
                                          'checkProbeResult':True, \
                                          'restrictedToUsers':[], \
                                          'restrictedToGroups':[], \
                                          'logUserRemotely':False \
                                         }
               elif keyValuePattern.match(record):
                  key,value = keyValuePattern.match(record).group(2,4)
                  if key in self.sites[siteName]:
                     if   isinstance(self.sites[siteName][key],list):
                        self.sites[siteName][key] = [e.strip() for e in value.split(',')]
                     elif isinstance(self.sites[siteName][key],bool):
                        self.sites[siteName][key] = bool(value.lower() == 'true')
                     else:
                        self.sites[siteName][key] = value
                  else:
                     log("Undefined key = value pair %s = %s for site %s" % (key,value,siteName))
            else:
               eof = True
         fp.close()

         userName = os.getenv("USER")
         for site in self.sites:
            restrictedToUsers = self.sites[site]['restrictedToUsers']
            if len(restrictedToUsers) > 0:
               if not userName in restrictedToUsers:
                  self.sites[site]['state'] = 'restrictedByUser'

         for site in self.sites:
            if self.sites[site]['state'] == 'enabled':
               restrictedToGroups = self.sites[site]['restrictedToGroups']
               if len(restrictedToGroups) > 0:
                  groupOK = False
                  for restrictedToGroup in restrictedToGroups:
                     if isGroupMember(restrictedToGroup):
                        groupOK = True
                        break
                  if not groupOK:
                     self.sites[site]['state'] = 'restrictedByGroup'

         for site in self.sites:
            failoverSite = self.sites[site]['failoverSite']
            if failoverSite != 'None' and failoverSite != 'grid':
               if self.sites[failoverSite]['state'] != 'enabled':
                  self.sites[site]['failoverSite'] = 'None'


   def getEnabledSites(self):
      enabledSites = []
      for site in self.sites:
         if self.sites[site]['state'] == 'enabled':
            enabledSites.append(site)

      return(enabledSites)


   def purgeDisabledSites(self,
                          siteNames):
      markedForDeletion = []
      for siteName in siteNames:
         try:
            site  = self.sites[siteName]
            state = site['state']
            if state != 'enabled':
               markedForDeletion.append(siteName)
         except:
            pass
      for siteName in markedForDeletion:
         siteNames.remove(siteName)
      del markedForDeletion


   def __checkProbeResult(self,
                          siteName):
      try:
         site = self.sites[siteName]
         checkProbeResult = site['checkProbeResult']
      except:
         checkProbeResult = True

      return(checkProbeResult)


   def getIgnoreProbeSites(self):
      ignoreProbeSites = []
      for site in self.sites:
         if self.sites[site]['state'] == 'enabled':
            if not self.__checkProbeResult(site):
               ignoreProbeSites.append(site)

      return(ignoreProbeSites)


   def purgeOfflineSites(self,
                         siteNames,
                         remoteProbeMonitor):
      goToGrid = False
      if remoteProbeMonitor:
         markedForDeletion = []
         substituteSites   = []
         for siteName in siteNames:
            if self.__checkProbeResult(siteName) and not remoteProbeMonitor.isSiteAvailable(siteName):
               markedForDeletion.append(siteName)
               try:
                  site         = self.sites[siteName]
                  failoverSite = site['failoverSite']
                  if not failoverSite == 'None':
                     if failoverSite == 'grid':
                        goToGrid = True
                     else:
                        if remoteProbeMonitor.isSiteAvailable(failoverSite):
                           substituteSites.append(failoverSite)
               except:
                  pass
         for siteName in markedForDeletion:
            siteNames.remove(siteName)
         if len(siteNames) > 0:
            goToGrid = False
         else:
            if len(substituteSites) > 0:
               for siteName in substituteSites:
                  siteNames.append(siteName)
               goToGrid = False

         del substituteSites
         del markedForDeletion

      return(goToGrid)


   def selectSite(self,
                  siteNames,
                  arbitraryExecutableAllowedRequired):
      selectedSite = ""

      if len(siteNames) > 0:
         possibleSites = []
         for siteName in siteNames:
            try:
               site    = self.sites[siteName]
               members = site['members']
               if len(members) > 0:
                  for member in members:
                     if not member in possibleSites:
                        possibleSites.append(member)
               else:
                  if not siteName in possibleSites:
                     possibleSites.append(siteName)
            except:
               pass

         if len(possibleSites) > 0:
            if arbitraryExecutableAllowedRequired:
               markedForDeletion = []
               for possibleSite in possibleSites:
                  try:
                     site = self.sites[possibleSite]
                     if site['arbitraryExecutableAllowed'] != "True":
                        markedForDeletion.append(possibleSite)
                  except:
                     markedForDeletion.append(possibleSite)
               for possibleSite in markedForDeletion:
                  possibleSites.remove(possibleSite)
               del markedForDeletion

            if len(possibleSites) > 0:
               possibleIndex = random.randint(0,len(possibleSites)-1)
               selectedSite = possibleSites[possibleIndex]

      return(selectedSite)


   def selectSites(self,
                   siteNames):
      selectedSites = []

      if len(siteNames) > 0:
         siteNameIndex = random.randint(0,len(siteNames)-1)
         siteName = siteNames[siteNameIndex]
         try:
            site    = self.sites[siteName]
            members = site['members']
            if len(members) > 0:
               memberIndex = random.randint(0,len(members)-1)
               selectedSites.append(members[memberIndex])
            else:
               selectedSites.append(siteName)
         except:
            markedForDeletion = []
            for siteName in siteNames:
               try:
                  site = self.sites[siteName]
                  markedForDeletion.append(siteName)
               except:
                  selectedSites.append(siteName)
            for siteName in markedForDeletion:
               siteNames.remove(siteName)
            del markedForDeletion

      return(selectedSites)


   def getSitesWithKeyValue(self,
                            key,
                            value,
                            siteNames):
      sitesWithKeyValue = []

      if len(siteNames) > 0:
         for siteName in siteNames:
            if siteName in self.sites:
               site    = self.sites[siteName]
               members = site['members']
               if len(members) > 0:
                  for member in members:
                     if member in self.sites:
                        site = self.sites[member]
                        if key in site:
                           if site[key] == value:
                              sitesWithKeyValue.append(member)
               else:
                  if key in site:
                     if site[key] == value:
                        sitesWithKeyValue.append(siteName)

      return(sitesWithKeyValue)


   def getSitesWithoutKeyValue(self,
                               key,
                               value,
                               siteNames):
      sitesWithoutKeyValue = []

      if len(siteNames) > 0:
         for siteName in siteNames:
            if siteName in self.sites:
               site    = self.sites[siteName]
               members = site['members']
               if len(members) > 0:
                  for member in members:
                     if member in self.sites:
                        site = self.sites[member]
                        if key in site:
                           if site[key] != value:
                              sitesWithoutKeyValue.append(member)
                        else:
                           sitesWithoutKeyValue.append(siteName)
               else:
                  if key in site:
                     if site[key] != value:
                        sitesWithoutKeyValue.append(siteName)
                  else:
                     sitesWithoutKeyValue.append(siteName)

      return(sitesWithoutKeyValue)


   def siteExists(self,
                  siteName):
      return(siteName in self.sites)


   def getSiteInfo(self,
                   siteName):
      site = self.sites[siteName]

      venues                     = site['venues']
      venueIndex                 = random.randint(0,len(site['venues'])-1)
      venue                      = venues[venueIndex]

      tunnelDesignator           = site['tunnelDesignator']
      venueMechanism             = site['venueMechanism']
      siteMonitorDesignator      = site['siteMonitorDesignator']
      venueGatekeeper            = site['venueGatekeeper']
      remoteUser                 = site['remoteUser']
      stageFiles                 = site['stageFiles']
      passUseEnvironment         = site['passUseEnvironment']
      remoteBatchSystem          = site['remoteBatchSystem']
      remoteBatchQueue           = site['remoteBatchQueue']
      remoteBatchPartition       = site['remoteBatchPartition']
      remoteBatchPartitionSize   = site['remoteBatchPartitionSize']
      remoteBatchConstraints     = site['remoteBatchConstraints']
      binDirectory               = site['remoteBinDirectory']
      scratchDirectory           = site['remoteScratchDirectory']
      if 'remoteManager' in site:
         remoteManager           = site['remoteManager']
      else:
         remoteManager           = ""

      hostAttributes             = site['remoteHostAttribute']
      arbitraryExecutableAllowed = site['arbitraryExecutableAllowed']
      logUserRemotely            = site['logUserRemotely']
      remotePpn                  = site['remotePpn']

      return(venues,venueIndex,venue,venueMechanism,tunnelDesignator,siteMonitorDesignator,venueGatekeeper,remoteUser,
             stageFiles,passUseEnvironment,logUserRemotely,
             remoteBatchSystem,remoteBatchQueue,remoteBatchPartition,remoteBatchPartitionSize,remoteBatchConstraints,
             binDirectory,scratchDirectory,
             remoteManager,hostAttributes,arbitraryExecutableAllowed,remotePpn)


