#
# Copyright (c) 2004-2010 Purdue University All rights reserved.
# 
# Developed by: HUBzero Technology Group, Purdue University
#               http://hubzero.org
# 
# HUBzero is free software: you can redistribute it and/or modify it under the terms of the
# GNU Lesser General Public License as published by the Free Software Foundation, either
# version 3 of the License, or (at your option) any later version.
# 
# HUBzero is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
# without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU Lesser General Public License for more details.  You should have received a
# copy of the GNU Lesser General Public License along with HUBzero.
# If not, see <http://www.gnu.org/licenses/>.
# 
# GNU LESSER GENERAL PUBLIC LICENSE
# Version 3, 29 June 2007
# Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>
#
import os.path
import re
import random

from LogMessage      import log as log
from GroupMembership import isGroupMember

class ToolsInfo:
   def __init__(self,
                infoDirectory,
                toolsFile):
      self.tools           = {}

      toolPattern     = re.compile('(.*\[)(.*)(])')
      keyValuePattern = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile('\s*#.*')
      toolName        = ""

      fp = open(os.path.join(infoDirectory,toolsFile), "r")
      if fp:
         eof = False
         while not eof:
            record = fp.readline()
            if record != "":
               record = commentPattern.sub("",record)
               if   toolPattern.match(record):
                  toolName = toolPattern.match(record).group(2)
                  if not toolName in self.tools:
                     self.tools[toolName] = []
                  toolGroupIndex = len(self.tools[toolName])
                  self.tools[toolName].append({'destinations':[], \
                                               'executablePath':'', \
                                               'isMPI':'', \
                                               'restrictedToUsers':[], \
                                               'restrictedToGroups':[], \
                                               'softenvExtensions':{}, \
                                               'remoteManager':'', \
                                               'state':'enabled' \
                                              })
               elif keyValuePattern.match(record):
                  key,value = keyValuePattern.match(record).group(2,4)
                  if key in self.tools[toolName][toolGroupIndex]:
                     if   isinstance(self.tools[toolName][toolGroupIndex][key],list):
                        self.tools[toolName][toolGroupIndex][key] = [e.strip() for e in value.split(',')]
                     elif isinstance(self.tools[toolName][toolGroupIndex][key],dict):
                        dictKey,dictValue = value.split(':')
                        self.tools[toolName][toolGroupIndex][key][dictKey.strip()] = dictValue.strip()
                     elif isinstance(self.tools[toolName][toolGroupIndex][key],bool):
                        self.tools[toolName][toolGroupIndex][key] = bool(value.lower() == 'true')
                     else:
                        self.tools[toolName][toolGroupIndex][key] = value
                  else:
                     log("Undefined tool key %s %s in tool %s" % (key,value,toolName))
            else:
               eof = True
         fp.close()

         for toolName in self.tools:
            markedForDeletion = []
            for toolGroupIndex in xrange(len(self.tools[toolName])):
               state = self.tools[toolName][toolGroupIndex]['state']
               if state == 'disabled':
                  markedForDeletion.append(toolGroupIndex)
            markedForDeletion.reverse()
            for toolGroupIndex in markedForDeletion:
               del self.tools[toolName][toolGroupIndex]
            del markedForDeletion


   def applyUserRestriction(self,
                            userName):
      for toolName in self.tools:
         markedForDeletion = []
         for toolGroupIndex in xrange(len(self.tools[toolName])):
            restrictedToUsers = self.tools[toolName][toolGroupIndex]['restrictedToUsers']
            if len(restrictedToUsers) > 0:
               if userName in restrictedToUsers:
                  userOK = True
               else:
                  userOK = False
               if not userOK:
                  markedForDeletion.append(toolGroupIndex)
         markedForDeletion.reverse()
         for toolGroupIndex in markedForDeletion:
            del self.tools[toolName][toolGroupIndex]
         del markedForDeletion


   def applyGroupRestriction(self):
      for toolName in self.tools:
         markedForDeletion = []
         for toolGroupIndex in xrange(len(self.tools[toolName])):
            restrictedToGroups = self.tools[toolName][toolGroupIndex]['restrictedToGroups']
            if len(restrictedToGroups) > 0:
               groupOK = False
               for restrictedToGroup in restrictedToGroups:
                  if isGroupMember(restrictedToGroup):
                     groupOK = True
                     break
               if not groupOK:
                  markedForDeletion.append(toolGroupIndex)
         markedForDeletion.reverse()
         for toolGroupIndex in markedForDeletion:
            del self.tools[toolName][toolGroupIndex]
         del markedForDeletion


   def isExecutableTool(self,
                        executable):
      return(executable in self.tools)


   def isPermissionGranted(self,
                           executable):
      permissionGranted = False
      if len(self.tools[executable]) > 0:
         permissionGranted = True

      return(permissionGranted)


   def purgeOfflineSites(self,
                         siteNames,
                         remoteProbeMonitor):
      if remoteProbeMonitor:
         markedForDeletion = []
         for siteName in siteNames:
            if not remoteProbeMonitor.isSiteAvailable(siteName):
               markedForDeletion.append(siteName)
         for siteName in markedForDeletion:
            siteNames.remove(siteName)
         del markedForDeletion


   def selectTool(self,
                  executable,
                  userDestinations,
                  remoteProbeMonitor):
      toolDestinations  = []
      executablePath    = ""
      softenvExtensions = {}
      remoteManager     = ""

      toolInfo = self.tools[executable]
      if len(userDestinations) == 0:
         toolGroupIndexes = range(len(toolInfo))
         while len(toolGroupIndexes) > 0 and len(toolDestinations) == 0:
            toolGroupIndex = random.choice(toolGroupIndexes)
            for toolDestination in toolInfo[toolGroupIndex]['destinations']:
               toolDestinations.append(toolDestination)
            self.purgeOfflineSites(toolDestinations,remoteProbeMonitor)
            if len(toolDestinations) == 0:
               toolGroupIndexes.remove(toolGroupIndex)
      else:
         toolGroupIndex = -1
         for userDestination in userDestinations:
            for i1 in xrange(len(toolInfo)):
               if userDestination in toolInfo[i1]['destinations']:
                  toolGroupIndex = i1
                  break
            if toolGroupIndex != -1:
               break

         if toolGroupIndex != -1:
            for toolDestination in toolInfo[toolGroupIndex]['destinations']:
               if toolDestination in userDestinations:
                  toolDestinations.append(userDestination)
         else:
            log("Invalid destination specified: %s" % (userDestination))

      if len(toolDestinations) > 0:
         executablePath       = toolInfo[toolGroupIndex]['executablePath']
         if 'softenvExtensions' in toolInfo[toolGroupIndex]:
            softenvExtensions = toolInfo[toolGroupIndex]['softenvExtensions']
         if 'remoteManager' in toolInfo[toolGroupIndex]:
            remoteManager     = toolInfo[toolGroupIndex]['remoteManager']

      return(toolDestinations,executablePath,remoteManager,softenvExtensions)


