#!/usr/bin/env python
#
# Copyright (c) 2004-2010 Purdue University All rights reserved.
# 
# Developed by: HUBzero Technology Group, Purdue University
#               http://hubzero.org
# 
# HUBzero is free software: you can redistribute it and/or modify it under the terms of the
# GNU Lesser General Public License as published by the Free Software Foundation, either
# version 3 of the License, or (at your option) any later version.
# 
# HUBzero is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
# without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU Lesser General Public License for more details.  You should have received a
# copy of the GNU Lesser General Public License along with HUBzero.
# If not, see <http://www.gnu.org/licenses/>.
# 
# GNU LESSER GENERAL PUBLIC LICENSE
# Version 3, 29 June 2007
# Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>
#
# ----------------------------------------------------------------------
#  monitorTunnel.py
#
#  script which monitors ssh tunnels for use with distributor and related processes.
#
import sys
import os
import os.path
import select
import signal
import socket
import time
from errno import EINTR

from LogMessage    import getLogMessageFileNo, openLog, logPID as log
from TunnelsInfo   import TunnelsInfo
from TunnelMonitor import TunnelMonitor

monitorHost        = ""
monitorPort        = 5729
monitorRoot        = os.path.join(os.sep,'opt','submit')
monitorLogLocation = os.path.join(os.sep,'var','log','submit')
logFileName        = "monitorTunnel.log"
activeTunnels      = {}

tunnelsInfo        = None

IDENTITY           = os.path.join(monitorRoot,".ssh","tunnel_rsa")


def terminate():
   global activeTunnels

   for activeTunnel in activeTunnels:
      tunnelPid,useCount = activeTunnels[activeTunnel]
      log("Send TERM to tunnel %s process" % (activeTunnel))
      os.kill(int(tunnelPid),signal.SIGTERM)

   log("*********************************")
   log("* tunnel server monitor stopped *")
   log("*********************************")


def sigGEN_handler(signalType, frame):
   terminate()
   sys.exit(1)


def sigINT_handler(signal, frame):
   global monitorRoot
   global tunnelsInfo

   log("Received SIGINT!")
   del tunnelsInfo
   tunnelsInfo = TunnelsInfo(monitorRoot,"tunnels.dat")
   log("Tunnel Info Reloaded!")
#  sigGEN_handler(signal, frame)

def sigHUP_handler(signal, frame):
   log("Received SIGHUP!")
   sigGEN_handler(signal, frame)

def sigQUIT_handler(signal, frame):
   log("Received SIGQUIT!")
   sigGEN_handler(signal, frame)

def sigABRT_handler(signal, frame):
   log("Received SIGABRT!")
   sigGEN_handler(signal, frame)

def sigTERM_handler(signal, frame):
   log("Received SIGTERM!")
   sigGEN_handler(signal, frame)

signal.signal(signal.SIGINT, sigINT_handler)
signal.signal(signal.SIGHUP, sigHUP_handler)
signal.signal(signal.SIGQUIT, sigQUIT_handler)
signal.signal(signal.SIGABRT, sigABRT_handler)
signal.signal(signal.SIGTERM, sigTERM_handler)


if __name__ == '__main__':

   openLog(os.path.join(monitorRoot,monitorLogLocation,logFileName))

   logMessageFileNo = getLogMessageFileNo()
   if logMessageFileNo != sys.stdout.fileno():
      os.close(sys.stdin.fileno())
      os.close(sys.stdout.fileno())
      os.close(sys.stderr.fileno())
      os.dup2(logMessageFileNo,1)
      os.dup2(logMessageFileNo,2)
      devnull = open("/dev/null","rw")
      os.dup2(sys.stdin.fileno(),devnull.fileno())

   if os.fork() != 0:
      os.wait()
      sys.exit(0)
   else:
      os.setsid()
      pid = os.fork()
      if pid != 0:
         sys.exit(0)

   log("*********************************")
   log("* tunnel server monitor started *")
   log("*********************************")

   time.sleep(2)

   tunnelsInfo = TunnelsInfo(monitorRoot,"tunnels.dat")

   del activeTunnels
   activeTunnels = {}

   inputDescriptors  = []
   clientConnections = []

   tunnelMonitor = TunnelMonitor(monitorHost,monitorPort)
   if not tunnelMonitor.isBound():
      sys.exit(1)

   monitorSocketFd = tunnelMonitor.boundFileDescriptor()
   inputDescriptors.append(monitorSocketFd)

   while 1:
      try:
         readyInputFds = select.select(inputDescriptors,[],[])[0]
      except select.error,err:
         if err[0] == EINTR:
            readyInputFds = []
         else:
            for inputDescriptor in inputDescriptors:
               if isinstance(inputDescriptor,socket.socket):
                  try:
                     os.fstat(inputDescriptor.fileno())
                  except:
                     log(inputDescriptor)
               else:
                  try:
                     os.fstat(inputDescriptor)
                  except:
                     log(inputDescriptor)
            terminate()
            raise

      if monitorSocketFd in readyInputFds:
         channel = tunnelMonitor.acceptConnection()
         clientConnections.append(channel)
         inputDescriptors.append(channel)

      for channel in readyInputFds:
         if channel in clientConnections:
            channelClosed = tunnelMonitor.processRequest(channel,tunnelsInfo,IDENTITY,activeTunnels)
            if channelClosed:
               clientConnections.remove(channel)
               inputDescriptors.remove(channel)

