#
# @package      hubzero-submit-distributor
# @file         MonitorsInfo.py
# @author       Steve Clark <clarks@purdue.edu>
# @copyright    Copyright 2004-2011 Purdue University. All rights reserved.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2004-2011 Purdue University
# All rights reserved.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of Purdue University.
#
import os.path
import re

from LogMessage import logID as log

class MonitorsInfo:
   def __init__(self,
                infoDirectory,
                monitorsFile):
      self.monitors = {}

      monitorPattern  = re.compile('(.*\[)(.*)(])')
      keyValuePattern = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile('\s*#.*')
      monitorName     = ""

      fpInfo = open(os.path.join(infoDirectory,monitorsFile), "r")
      if fpInfo:
         eof = False
         while not eof:
            record = fpInfo.readline()
            if record != "":
               record = commentPattern.sub("",record)
               if   monitorPattern.match(record):
                  monitorName = monitorPattern.match(record).group(2)
                  self.monitors[monitorName] = {'venue':'', \
                                                'venueMechanism':'ssh', \
                                                'tunnelDesignator':'', \
                                                'remoteUser':'', \
                                                'x509ProxyPath':'', \
                                                'remoteMonitorCommand':'', \
                                                'state':'enabled', \
                                               }
               elif keyValuePattern.match(record):
                  key,value = keyValuePattern.match(record).group(2,4)
                  if key in self.monitors[monitorName]:
                     if   isinstance(self.monitors[monitorName][key],list):
                        self.monitors[monitorName][key] = [e.strip() for e in value.split(',')]
                     elif isinstance(self.monitors[monitorName][key],bool):
                        self.monitors[monitorName][key] = bool(value.lower() == 'true')
                     else:
                        self.monitors[monitorName][key] = value
                  else:
                     log("Undefined key = value pair %s = %s for site %s" % (key,value,monitorName))
            else:
               eof = True
         fpInfo.close()

         markedForDeletion = []
         for monitorName in self.monitors:
            if self.monitors[monitorName]['state'] != 'enabled':
               markedForDeletion.append(monitorName)
         for monitorName in markedForDeletion:
            del self.monitors[monitorName]
         del markedForDeletion


   def getSSHCommand(self,
                     monitorName,
                     sshIdentity,
                     remoteTunnelMonitor):
      sshCommand       = ""
      tunnelDesignator = ""
      try:
         if self.monitors[monitorName]['venueMechanism'] == 'ssh':
            venue                = self.monitors[monitorName]['venue']
            tunnelDesignator     = self.monitors[monitorName]['tunnelDesignator']
            user                 = self.monitors[monitorName]['remoteUser']
            remoteMonitorCommand = self.monitors[monitorName]['remoteMonitorCommand']

            sshBaseCommand = "ssh -T -x -a -i " + sshIdentity
            if tunnelDesignator == "":
               sshCommand = sshBaseCommand + " " + user + "@" + venue + " \"" + remoteMonitorCommand + "\""
            else:
               tunnelAddress,tunnelPort = remoteTunnelMonitor.getTunnelAddressPort(tunnelDesignator)
               sshCommand = sshBaseCommand + " -p " + tunnelPort + " " + user + "@" + tunnelAddress + \
                                                                   " \"" + remoteMonitorCommand + "\""
         else:
            venue                = self.monitors[monitorName]['venue']
            x509ProxyPath        = self.monitors[monitorName]['x509ProxyPath']
            remoteMonitorCommand = self.monitors[monitorName]['remoteMonitorCommand']

            sshBaseCommand = "X509_USER_PROXY=" + x509ProxyPath + " gsissh -T -x -a "
            sshCommand = sshBaseCommand + " " + venue + " \"" + remoteMonitorCommand + "\""
      except:
         log("Build sshCommand failed for %s" % (monitorName))

      return(sshCommand,tunnelDesignator)

