# @package      hubzero-submit-distributor
# @file         RemoteBatchAppScript.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2014 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012-2014 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import os.path
import re
import logging

class RemoteBatchAppScript:
   def __init__(self,
                hubUserName,
                hubUserId,
                runName,
                localJobId,
                instanceId,
                isPegasus,
                transferExecutable,
                executable,
                stdinput,
                arguments,
                useEnvironment,
                environment,
                isMultiCoreRequest,
                siteInfo,
                managerInfo,
                timeHistoryLogs):
      self.logger               = logging.getLogger(__name__)
      self.hubUserName          = hubUserName
      self.hubUserId            = hubUserId
      self.runName              = runName
      self.localJobId           = localJobId
      self.instanceId           = instanceId
      self.isPegasus            = isPegasus
      self.sharedUserSpace      = siteInfo['sharedUserSpace']
      self.stageFiles           = siteInfo['stageFiles']
      self.transferExecutable   = transferExecutable
      if transferExecutable:
         if self.stageFiles:
            self.executable     = os.path.join(".",os.path.basename(executable))
         else:
            self.executable     = executable
      else:
         if not executable.startswith('/') and not executable.startswith('$') and siteInfo['remoteApplicationRootDirectory'] != "":
            self.executable     = os.path.join(siteInfo['remoteApplicationRootDirectory'],executable)
         else:
            self.executable     = executable
      self.stdinput             = stdinput
      self.arguments            = arguments
      self.useEnvironment       = useEnvironment
      self.environment          = environment
      self.isMultiCoreRequest   = isMultiCoreRequest
      self.computationMode      = managerInfo['computationMode']
      self.mpiRankVariable      = managerInfo['mpiRankVariable']
      self.preManagerCommands   = managerInfo['preManagerCommands']
      self.managerCommand       = managerInfo['managerCommand']
      self.postManagerCommands  = managerInfo['postManagerCommands']
      self.timestampTransferred = timeHistoryLogs['timestampTransferred']
      self.timestampStart       = timeHistoryLogs['timestampStart']
      self.timestampFinish      = timeHistoryLogs['timestampFinish']
      self.timeResults          = timeHistoryLogs['timeResults']


   def __makeSerialTemplate(self):
      return """#!/bin/sh
# RemoteBatchAppScript:makeSerialTemplate
#
trap cleanup HUP INT QUIT ABRT TERM

cleanup()
{
   echo "Abnormal termination by signal"
   if [ ! -s TS_FINISH ] ; then
      date +"%s" > TS_FINISH
   fi
   exit 1
}

CURDIR=`pwd`
export PATH=${CURDIR}:${PATH}

USEENVIRONMENT
ENVIRONMENT

TIMEPATH=
for timePath in ${HOME}/bin/time /usr/bin/time /usr/local/bin/time /apps/submit/bin/time ; do
   if [ -x ${timePath} ] ; then
      TIMEPATH=${timePath}
      break
   fi
done

date +"%s" > TS_START

PREMANAGERCOMMANDS
${TIMEPATH} --format "Command exited with status %x\\nreal %e\\nuser %U\\nsys %S" -o TIME_RESULTS MANAGERCOMMAND EXECUTABLE ARGUMENTS STDIN > RUNNAME_INSTANCEID.stdout 2> RUNNAME_INSTANCEID.stderr
POSTMANAGERCOMMANDS

date +"%s" > TS_FINISH
"""


   def __makeSerialPackingTemplate(self):
      return """#!/bin/sh
# RemoteBatchAppScript:makeSerialPackingTemplate
#
trap cleanup HUP INT QUIT ABRT TERM

cleanup()
{
   echo "Abnormal termination by signal"
   if [ ! -s TS_FINISH ] ; then
      date +"%s" > TS_FINISH
   fi
   exit 1
}

touch JOBID_INSTANCEID_output.tar.gz

JOB_DIRECTORY=`date "+%s"`_JOBID_INSTANCEID
mkdir ${JOB_DIRECTORY}
cd ${JOB_DIRECTORY}

CURDIR=`pwd`
export PATH=${CURDIR}:${PATH}

touch .__fileTimeMarker
sleep 1

tar xfz ../JOBID_INSTANCEID_input.tar.gz

date +"%s" > TS_TRANSFERRED

USEENVIRONMENT
ENVIRONMENT

TIMEPATH=
for timePath in ${HOME}/bin/time /usr/bin/time /usr/local/bin/time /apps/submit/bin/time ; do
   if [ -x ${timePath} ] ; then
      TIMEPATH=${timePath}
      break
   fi
done

date +"%s" > TS_START

PREMANAGERCOMMANDS
${TIMEPATH} --format "Command exited with status %x\\nreal %e\\nuser %U\\nsys %S" -o TIME_RESULTS MANAGERCOMMAND EXECUTABLE ARGUMENTS STDIN > RUNNAME_INSTANCEID.stdout 2> RUNNAME_INSTANCEID.stderr
POSTMANAGERCOMMANDS

date +"%s" > TS_FINISH

# Allow for file/directory names with embedded spaces.
find . -newer .__fileTimeMarker -not -name . -not -iname '*hub-proxy.*' -print0 | xargs --null tar zcf ../JOBID_INSTANCEID_output.tar.gz --no-recursion

cd ..
rm -rf ${JOB_DIRECTORY}
"""


   def __makePegasusTemplate(self):
      return """#!/bin/sh
# RemoteBatchAppScript:makePegasusTemplate
#
trap cleanup HUP INT QUIT ABRT TERM

cleanup()
{
   echo "Abnormal termination by signal"
   if [ ! -s TS_FINISH ] ; then
      date +"%s" > TS_FINISH
   fi
   exit 1
}

exitStatus=0
CURDIR=`pwd`
export PATH=${CURDIR}:${PATH}

USEENVIRONMENT
ENVIRONMENT

date +"%s" > TS_START

PREMANAGERCOMMANDS
MANAGERCOMMAND EXECUTABLE ARGUMENTS STDIN
exitStatus=$?
POSTMANAGERCOMMANDS

date +"%s" > TS_FINISH

exit ${exitStatus}
"""


   def __makePegasusPackingTemplate(self):
      return """#!/bin/sh
# RemoteBatchAppScript:makePegasusPackingTemplate
#
trap cleanup HUP INT QUIT ABRT TERM

cleanup()
{
   echo "Abnormal termination by signal"
   if [ ! -s TS_FINISH ] ; then
      date +"%s" > TS_FINISH
   fi
   exit 1
}

exitStatus=0
touch JOBID_INSTANCEID_output.tar.gz

JOB_DIRECTORY=`date "+%s"`_JOBID_INSTANCEID
mkdir ${JOB_DIRECTORY}
cd ${JOB_DIRECTORY}

CURDIR=`pwd`
export PATH=${CURDIR}:${PATH}

touch .__fileTimeMarker
sleep 1

tar xfz ../JOBID_INSTANCEID_input.tar.gz

date +"%s" > TS_TRANSFERRED

USEENVIRONMENT
ENVIRONMENT

date +"%s" > TS_START

PREMANAGERCOMMANDS
MANAGERCOMMAND EXECUTABLE ARGUMENTS STDIN
exitStatus=$?
POSTMANAGERCOMMANDS

date +"%s" > TS_FINISH

# Allow for file/directory names with embedded spaces.
find . -newer .__fileTimeMarker -not -name . -not -iname '*hub-proxy.*' -print0 | xargs --null tar zcf ../JOBID_INSTANCEID_output.tar.gz --no-recursion

cd ..
rm -rf ${JOB_DIRECTORY}

exit ${exitStatus}
"""


   def __buildAppSerialScript(self,
                              doInputOutputPacking):
      # setup regex's for the template
      re_runName             = re.compile("RUNNAME")
      re_jobId               = re.compile("JOBID")
      re_instanceId          = re.compile("INSTANCEID")
      re_exe                 = re.compile("EXECUTABLE")
      re_stdinput            = re.compile("STDIN")
      re_arguments           = re.compile("ARGUMENTS")
      re_useEnvironment      = re.compile("USEENVIRONMENT")
      re_environment         = re.compile("ENVIRONMENT")
      re_tsTransferred       = re.compile("TS_TRANSFERRED")
      re_tsStart             = re.compile("TS_START")
      re_tsFinish            = re.compile("TS_FINISH")
      re_timeResults         = re.compile("TIME_RESULTS")
      re_preManagerCommands  = re.compile("PREMANAGERCOMMANDS")
      re_managerCommand      = re.compile("MANAGERCOMMAND")
      re_postManagerCommands = re.compile("POSTMANAGERCOMMANDS")
      re_hubUserName         = re.compile("HUBUSERNAME")
      re_hubUserId           = re.compile("HUBUSERID")

      if self.isPegasus:
         if doInputOutputPacking:
            template = self.__makePegasusPackingTemplate()
         else:
            template = self.__makePegasusTemplate()
      else:
         if doInputOutputPacking:
            template = self.__makeSerialPackingTemplate()
         else:
            template = self.__makeSerialTemplate()

      environmentExport = ""
      environmentVars = self.environment.split()
      for environmentVar in environmentVars:
         environmentExport += "export " + environmentVar + "\n"
      useEnvironmentExport = self.useEnvironment

      if self.stdinput == "":
         template = re_stdinput.sub(self.stdinput,template)
      else:
         template = re_stdinput.sub(" < "+self.stdinput,template)
      template = re_runName.sub(self.runName,template)
      template = re_jobId.sub(self.localJobId,template)
      template = re_instanceId.sub(self.instanceId,template)
      template = re_exe.sub(self.executable,template)
      template = re_arguments.sub(self.arguments,template)
      template = re_useEnvironment.sub(useEnvironmentExport,template)
      template = re_environment.sub(environmentExport,template)
      template = re_tsTransferred.sub(self.timestampTransferred,template)
      template = re_tsStart.sub(self.timestampStart,template)
      template = re_tsFinish.sub(self.timestampFinish,template)
      template = re_timeResults.sub(self.timeResults,template)
      template = re_preManagerCommands.sub("\n".join(self.preManagerCommands),template)
      template = re_postManagerCommands.sub("\n".join(self.postManagerCommands),template)
      template = re_managerCommand.sub(self.managerCommand,template)
      template = re_hubUserName.sub(self.hubUserName,template)
      template = re_hubUserId.sub(str(self.hubUserId),template)

      return(template)


   def __makeMPITemplate(self):
      return """#!/bin/sh
# RemoteBatchAppScript:makeMPITemplate
#
SCRIPT=`basename $0`
WORKINGDIR=`dirname $0`

cd ${WORKINGDIR}

STD_INPUT=STDIN

USEENVIRONMENT
ENVIRONMENT

TIMEPATH=
for timePath in ${HOME}/bin/time /usr/bin/time /usr/local/bin/time /apps/submit/bin/time ; do
   if [ -x ${timePath} ] ; then
      TIMEPATH=${timePath}
      break
   fi
done

rankExtension=`printf "%06d" ${MPIRANKVARIABLE}`
${TIMEPATH} --format "Command exited with status %x\\nreal %e\\nuser %U\\nsys %S" -o TIME_RESULTS-${rankExtension} EXECUTABLE ARGUMENTS "$@" < ${STD_INPUT}
"""


   def __makeMPINoRankTemplate(self):
      return """#!/bin/sh
# RemoteBatchAppScript:makeMPINoRankTemplate
#
SCRIPT=`basename $0`
WORKINGDIR=`dirname $0`

args=( "$@" )
for (( i1=0 ; i1 < ${#args[@]} ; i1++ )) ; do
   if   [ "${args[i1]}" = "-p4pg" ] ; then
      p4rank=0
   elif [ "${args[i1]}" = "-p4wd" ] ; then
      (( i1++ ))
      args[${i1}]=${WORKINGDIR}
   elif [ "${args[i1]}" = "-p4rmrank" ] ; then
      (( i1++ ))
      p4rank=${args[i1]}
   fi
done
if [ "${p4rank}" = "0" ] ; then
   set -- ${args[@]}
fi

cd ${WORKINGDIR}

STD_INPUT=STDIN

USEENVIRONMENT
ENVIRONMENT

TIMEPATH=
for timePath in ${HOME}/bin/time /usr/bin/time /usr/local/bin/time /apps/submit/bin/time ; do
   if [ -x ${timePath} ] ; then
      TIMEPATH=${timePath}
      break
   fi
done

rankExtension=`printf "%06d" ${p4rank}`
if [ "${p4rank}" = "0" ] ; then
   ${TIMEPATH} --format "Command exited with status %x\\nreal %e\\nuser %U\\nsys %S" -o TIME_RESULTS-${rankExtension} EXECUTABLE ARGUMENTS "$@" < ${STD_INPUT}
else
   ${TIMEPATH} --format "Command exited with status %x\\nreal %e\\nuser %U\\nsys %S" -o TIME_RESULTS-${rankExtension} EXECUTABLE "$@" < ${STD_INPUT}
fi
"""


   def __makeMPIPackingTemplate(self):
      return """#!/bin/sh
# RemoteBatchAppScript:makeMPIPackingTemplate
#
SCRIPT=`basename $0`
WORKINGDIR=`dirname $0`

cd ${WORKINGDIR}

STD_INPUT=STDIN

USEENVIRONMENT
ENVIRONMENT

tar xfz JOBID_INSTANCEID_input.tar.gz
touch JOBID_INSTANCEID_output.tar.gz
touch .__fileTimeMarker
sleep 1

TIMEPATH=
for timePath in ${HOME}/bin/time /usr/bin/time /usr/local/bin/time /apps/submit/bin/time ; do
   if [ -x ${timePath} ] ; then
      TIMEPATH=${timePath}
      break
   fi
done

rankExtension=`printf "%06d" ${MPIRANKVARIABLE}`
${TIMEPATH} --format "Command exited with status %x\\nreal %e\\nuser %U\\nsys %S" -o TIME_RESULTS-${rankExtension} EXECUTABLE ARGUMENTS "$@" < ${STD_INPUT}

# Allow for file/directory names with embedded spaces.
find . -newer .__fileTimeMarker -not -name . -not -iname '*hub-proxy.*' -print0 | xargs --null tar zcf JOBID_output.tar.gz --no-recursion
"""


   def __makeMPINoRankPackingTemplate(self):
      return """#!/bin/sh
# RemoteBatchAppScript:makeMPINoRankPackingTemplate
#
SCRIPT=`basename $0`
WORKINGDIR=`dirname $0`

args=( "$@" )
for (( i1=0 ; i1 < ${#args[@]} ; i1++ )) ; do
   if   [ "${args[i1]}" = "-p4pg" ] ; then
      p4rank=0
   elif [ "${args[i1]}" = "-p4wd" ] ; then
      (( i1++ ))
      args[${i1}]=${WORKINGDIR}
   elif [ "${args[i1]}" = "-p4rmrank" ] ; then
      (( i1++ ))
      p4rank=${args[i1]}
   fi
done
if [ "${p4rank}" = "0" ] ; then
   set -- ${args[@]}
fi

cd ${WORKINGDIR}

STD_INPUT=STDIN

tar xfz JOBID_input.tar.gz
touch JOBID_output.tar.gz
touch .__fileTimeMarker
sleep 1

TIMEPATH=
for timePath in ${HOME}/bin/time /usr/bin/time /usr/local/bin/time /apps/submit/bin/time ; do
   if [ -x ${timePath} ] ; then
      TIMEPATH=${timePath}
      break
   fi
done

USEENVIRONMENT
ENVIRONMENT

rankExtension=`printf "%06d" ${p4rank}`
if [ "${p4rank}" = "0" ] ; then
   ${TIMEPATH} --format "Command exited with status %x\\nreal %e\\nuser %U\\nsys %S" -o TIME_RESULTS-${rankExtension} EXECUTABLE ARGUMENTS "$@" < ${STD_INPUT}
else
   ${TIMEPATH} --format "Command exited with status %x\\nreal %e\\nuser %U\\nsys %S" -o TIME_RESULTS-${rankExtension} EXECUTABLE "$@" < ${STD_INPUT}
fi

# Allow for file/directory names with embedded spaces.
find . -newer .__fileTimeMarker -not -name . -not -iname '*hub-proxy.*' -print0 | xargs --null tar zcf JOBID_output.tar.gz --no-recursion
"""


   def __buildAppMPIScript(self,
                           doInputOutputPacking):
      # setup regex's for the template
      re_runName         = re.compile("RUNNAME")
      re_jobId           = re.compile("JOBID")
      re_instanceId      = re.compile("INSTANCEID")
      re_executable      = re.compile("EXECUTABLE")
      re_stdinput        = re.compile("STDIN")
      re_cmdstdinput     = re.compile(" < \${STD_INPUT}")
      re_arguments       = re.compile("ARGUMENTS")
      re_useEnvironment  = re.compile("USEENVIRONMENT")
      re_environment     = re.compile("ENVIRONMENT")
      re_mpiRankVariable = re.compile("MPIRANKVARIABLE")
      re_timeResults     = re.compile("TIME_RESULTS")
      re_hubUserName     = re.compile("HUBUSERNAME")
      re_hubUserId       = re.compile("HUBUSERID")

      if doInputOutputPacking:
         if self.mpiRankVariable == "":
            template = self.__makeMPINoRankPackingTemplate()
         else:
            template = self.__makeMPIPackingTemplate()
      else:
         if self.mpiRankVariable == "":
            template = self.__makeMPINoRankTemplate()
         else:
            template = self.__makeMPITemplate()

      environmentExport = ""
      environmentVars = self.environment.split()
      for environmentVar in environmentVars:
         environmentExport += "export " + environmentVar + "\n"
      useEnvironmentExport = self.useEnvironment

      template = re_executable.sub(self.executable,template)
      template = re_stdinput.sub(self.stdinput,template)
      if self.stdinput == "":
         template = re_cmdstdinput.sub("",template)
      template = re_arguments.sub(self.arguments,template)
      template = re_useEnvironment.sub(useEnvironmentExport,template)
      template = re_environment.sub(environmentExport,template)
      template = re_mpiRankVariable.sub(self.mpiRankVariable,template)
      template = re_timeResults.sub(self.timeResults,template)
      template = re_runName.sub(self.runName,template)
      template = re_jobId.sub(self.localJobId,template)
      template = re_instanceId.sub(self.instanceId,template)
      template = re_hubUserName.sub(self.hubUserName,template)
      template = re_hubUserId.sub(str(self.hubUserId),template)

      return(template)


   def buildAppScript(self):
      appScriptName = "%s_%s.sh" % (self.localJobId,self.instanceId)
      doInputOutputPacking = self.sharedUserSpace and self.stageFiles
      if self.isMultiCoreRequest:
         if   self.computationMode == 'mpi':
            appScript = self.__buildAppMPIScript(doInputOutputPacking)
         elif self.computationMode == 'mpiNoFork':
            appScript = ""
         elif self.computationMode == 'parallel':
            appScript = ""
         elif self.computationMode == 'matlabmpi':
            appScript = ""
      else:
         appScript = self.__buildAppSerialScript(doInputOutputPacking)

      return(appScriptName,appScript)


