# @package      hubzero-submit-distributor
# @file         FileMoversInfo.py
# @copyright    Copyright (c) 2020-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2020-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import os
import re
import glob
import json
import logging

from hubzero.submit.LogMessage      import getLogJobIdMessage as getLogMessage
from hubzero.submit.GroupMembership import GroupMembership

class FileMoversInfo:
   def __init__(self,
                fileMoversPath,
                restrictionUser=None):
      self.logger = logging.getLogger(__name__)

      self.fileMovers = {}

      if os.path.isdir(fileMoversPath):
         for fileMoversInfoPath in glob.iglob(os.path.join(fileMoversPath,'*')):
            self.readFileMoversInfoFile(fileMoversInfoPath)
      else:
         for fileMoversInfoPath in glob.iglob(fileMoversPath):
            self.readFileMoversInfoFile(fileMoversInfoPath)

      if restrictionUser:
         for fileMover in self.fileMovers:
            restrictedToUsers = self.fileMovers[fileMover]['restrictedToUsers']
            if len(restrictedToUsers) > 0:
               if not restrictionUser in restrictedToUsers:
                  self.fileMovers[fileMover]['state'] = 'restrictedByUser'

         groupMembership = GroupMembership(restrictionUser=restrictionUser)
         for fileMover in self.fileMovers:
            if self.fileMovers[fileMover]['state'] == 'enabled':
               restrictedToGroups = self.fileMovers[fileMover]['restrictedToGroups']
               if len(restrictedToGroups) > 0:
                  groupOK = False
                  for restrictedToGroup in restrictedToGroups:
                     if groupMembership.isGroupMember(restrictedToGroup):
                        groupOK = True
                        break
                  if not groupOK:
                     self.fileMovers[fileMover]['state'] = 'restrictedByGroup'


   def readFileMoversInfoFile(self,
                              fileMoversInfoPath):
      fileMoverPattern = re.compile(r'(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern  = re.compile(r'( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern   = re.compile(r'\s*#.*')
      fileMoverName    = ""

      if os.path.exists(fileMoversInfoPath):
         try:
            fpInfo = open(fileMoversInfoPath,'r')
            try:
               eof = False
               while not eof:
                  record = fpInfo.readline()
                  if record != "":
                     record = commentPattern.sub("",record)
                     if   fileMoverPattern.match(record):
                        fileMoverName = fileMoverPattern.match(record).group(2)
                        self.fileMovers[fileMoverName] = {'venue':'',
                                                          'fileMoverType':'tapis3',
                                                          'identityManager':'user',
                                                          'remoteUser':'USER',
                                                          'remoteBinDirectory':os.path.join('${HOME}','bin'),
                                                          'preStageDirectory':'',
                                                          'tapisStorageSystem':'',
                                                          'restrictedToUsers':[],
                                                          'restrictedToGroups':[],
                                                          'state':'enabled'
                                                         }
                     elif keyValuePattern.match(record):
                        key,value = keyValuePattern.match(record).group(2,4)
                        if key in self.fileMovers[fileMoverName]:
                           if   isinstance(self.fileMovers[fileMoverName][key],list):
                              self.fileMovers[fileMoverName][key] = [e.strip() for e in value.split(',')]
                           elif isinstance(self.fileMovers[fileMoverName][key],bool):
                              self.fileMovers[fileMoverName][key] = bool(value.lower() == 'true')
                           elif isinstance(self.fileMovers[fileMoverName][key],float):
                              self.fileMovers[fileMoverName][key] = float(value)
                           elif isinstance(self.fileMovers[fileMoverName][key],int):
                              self.fileMovers[fileMoverName][key] = int(value)
                           elif isinstance(self.fileMovers[fileMoverName][key],dict):
                              try:
                                 sampleKey   = list(self.fileMovers[fileMoverName][key].keys())[0]
                                 sampleValue = self.fileMovers[fileMoverName][key][sampleKey]
                              except:
                                 try:
                                    self.fileMovers[fileMoverName][key] = json.loads(value)
                                 except:
                                    self.fileMovers[fileMoverName][key] = {}
                                    sampleKey   = "key"
                                    sampleValue = "value"
                              else:
                                 self.fileMovers[fileMoverName][key] = {}
               
                              if not self.fileMovers[fileMoverName][key]:
                                 for e in value.split(','):
                                    dictKey,dictValue = e.split(':')
                                    if isinstance(sampleKey,int):
                                       dictKey = int(dictKey)
                                    if   isinstance(sampleValue,int):
                                       dictValue = int(dictValue)
                                    elif isinstance(sampleValue,float):
                                       dictValue = float(dictValue)
                                    elif isinstance(sampleValue,bool):
                                       dictValue = bool(dictValue.lower() == 'true')
                                    self.fileMovers[fileMoverName][key][dictKey] = dictValue
                           else:
                              self.fileMovers[fileMoverName][key] = value
                        else:
                           message = "Undefined key = value pair %s = %s for fileMover %s" % (key,value,fileMoverName)
                           self.logger.log(logging.WARNING,getLogMessage(message))
                  else:
                     eof = True
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("FileMovers configuration file %s could not be read" % \
                                                                                             (fileMoversInfoPath)))
            finally:
               fpInfo.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("FileMovers configuration file %s could not be opened" % \
                                                                                            (fileMoversInfoPath)))
      else:
         self.logger.log(logging.ERROR,getLogMessage("FileMovers configuration file %s is missing" % \
                                                                                (fileMoversInfoPath)))


   def getEnabledFileMovers(self):
      enabledFileMovers = []
      for fileMover in self.fileMovers:
         if self.fileMovers[fileMover]['state'] == 'enabled':
            enabledFileMovers.append(fileMover)

      return(enabledFileMovers)


   def purgeDisabledFileMovers(self,
                               fileMoverNames):
      reasonsDenied = {}
      markedForDeletion = []
      for fileMoverName in fileMoverNames:
         try:
            fileMover = self.fileMovers[fileMoverName]
            state     = fileMover['state']
            if state != 'enabled':
               markedForDeletion.append(fileMoverName)
               reasonsDenied[fileMoverName] = state
         except:
            pass
      for fileMoverName in markedForDeletion:
         fileMoverNames.remove(fileMoverName)
      del markedForDeletion

      return(reasonsDenied)


   def getFileMoverKeyValue(self,
                            fileMoverName,
                            key):
      value = ""

      if fileMoverName in self.fileMovers:
         fileMover = self.fileMovers[fileMoverName]
         if key in fileMover:
            value = fileMover[key]

      return(value)


   def getFileMoverVenue(self,
                         fileMoverName):
      venue = []
      if fileMoverName in self.fileMovers:
         fileMover = self.fileMovers[fileMoverName]
         venue = fileMover['venue']

      return(venue)


   def fileMoverExists(self,
                       fileMoverName):
      return(fileMoverName in self.fileMovers)


   def getDefaultFileMoverInfo(self):
      fileMoverInfo = {}
      fileMoverInfo['fileMoverName']      = "default"
      fileMoverInfo['fileMoverType']      = "tapis3"
      fileMoverInfo['venue']              = "Unknown"
      fileMoverInfo['identityManager']    = ""
      fileMoverInfo['remoteUser']         = ""
      fileMoverInfo['remoteBinDirectory'] = os.path.join('${HOME}','bin')
      fileMoverInfo['preStageDirectory']  = ""
      fileMoverInfo['tapisStorageSystem'] = ""

      return(fileMoverInfo)


   def getFileMoverInfo(self,
                        fileMoverName):
      fileMoverInfo = {}

      fileMover = self.fileMovers[fileMoverName]
      fileMoverInfo['fileMoverName']      = fileMoverName
      fileMoverInfo['fileMoverType']      = fileMover['fileMoverType']
      fileMoverInfo['venue']              = fileMover['venue']
      fileMoverInfo['identityManager']    = fileMover['identityManager']
      fileMoverInfo['remoteUser']         = fileMover['remoteUser']
      fileMoverInfo['remoteBinDirectory'] = fileMover['remoteBinDirectory']
      fileMoverInfo['preStageDirectory']  = fileMover['preStageDirectory']
      fileMoverInfo['tapisStorageSystem'] = fileMover['tapisStorageSystem']

      return(fileMoverInfo)


