# @package      hubzero-submit-distributor
# @file         RemoteBatchCONDOR.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
import re
import logging

from hubzero.submit.LogMessage        import getLogJobIdMessage as getLogMessage 
from hubzero.submit.ParameterTemplate import ParameterTemplate

class RemoteBatchCONDOR:
   SUBMISSIONSCRIPTCOMMANDPREFIX = ''

   def __init__(self,
                hubUserName,
                hubUserId,
                submitterClass,
                runName,
                localJobId,
                instanceId,
                instanceDirectory,
                appScriptName,
                environment,
                isMultiCoreRequest,
                siteInfo,
                toolFilesInfo,
                dockerImageInfo,
                submissionScriptsInfo,
                managerInfo,
                wallTime):
      self.logger                   = logging.getLogger(__name__)
      self.hubUserName              = hubUserName
      self.hubUserId                = hubUserId
      self.submitterClass           = submitterClass
      self.runName                  = runName
      self.localJobId               = localJobId
      self.instanceId               = instanceId
      self.instanceDirectory        = instanceDirectory
      self.stageFiles               = siteInfo['stageFiles'] and not siteInfo['sharedUserSpace']
      self.submissionScriptCommands = siteInfo['submissionScriptCommands']
      self.toolFilesInfo            = toolFilesInfo
      self.dockerImageInfo          = dockerImageInfo
      self.appScriptName            = appScriptName
      self.environment              = environment
      self.isMultiCoreRequest       = isMultiCoreRequest
      self.submissionScriptsInfo    = submissionScriptsInfo
      self.computationMode          = managerInfo['computationMode']
      self.wallTime                 = str(int(wallTime))

      self.nodeFileName = ""
      self.nodeList     = []

      self.toolInputTemplateFileName  = ""
      self.toolInputTemplate          = ""
      self.toolOutputTemplateFileName = ""
      self.toolOutputTemplate         = ""


   def __buildSerialFile(self):
      if self.stageFiles:
         rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','CONDOR','serialstaged')
      else:
         rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','CONDOR','serialnotstaged')

      commandsCONDOR = ""
      if self.submissionScriptCommands:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            commandSeparator = "\n%s " % self.SUBMISSIONSCRIPTCOMMANDPREFIX
            commandsCONDOR = self.SUBMISSIONSCRIPTCOMMANDPREFIX + " " + commandSeparator.join(self.submissionScriptCommands)
         else:
            commandSeparator = "\n"
            commandsCONDOR = commandSeparator.join(self.submissionScriptCommands)

      substitutions = {}
      substitutions["RUNNAME"]                  = self.runName
      substitutions["JOBID"]                    = self.localJobId
      substitutions["INSTANCEID"]               = self.instanceId
      substitutions["APPSCRIPTNAME"]            = self.appScriptName
      substitutions["WALLTIME"]                 = self.wallTime
      substitutions["ENVIRONMENT"]              = self.environment
      substitutions["INSTANCEDIRECTORY"]        = self.instanceDirectory
      substitutions["HUBUSERNAME"]              = self.hubUserName
      substitutions["HUBUSERID"]                = str(self.hubUserId)
      substitutions["SUBMISSIONSCRIPTCOMMANDS"] = commandsCONDOR

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (err[0])))
      except TypeError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))
      else:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            submissionScript = re.sub("(\n)*\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),
                                           "\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),submissionScript)

      return(submissionScript)


   def buildBatchScript(self):
      batchLogName = ""
      batchScriptName = ""
      if self.isMultiCoreRequest:
         if   self.computationMode == 'mpi':
            batchScript = ""
         elif self.computationMode == 'parallel':
            batchScript = ""
         elif self.computationMode == 'matlabmpi':
            batchScript = ""
      else:
         batchLogName = "condor_%s_%s.log" % (self.runName,self.instanceId)
         batchScript = self.__buildSerialFile()
         batchScriptName = "%s_%s.condor" % (self.localJobId,self.instanceId)
      batchScriptExecutable = False

      return(batchLogName,batchScriptName,batchScript,batchScriptExecutable)


   def getBatchNodeList(self):
      return(self.nodeFileName,self.nodeList)


   def getBatchToolInputTemplate(self):
      return(self.toolInputTemplateFileName,self.toolInputTemplate)


   def getBatchToolOutputTemplate(self):
      return(self.toolOutputTemplateFileName,self.toolOutputTemplate)


   def getRemoteJobIdNumber(self,
                            remoteJobId):
      try:
# Submitting job(s). Logging submit event(s). 1 job(s) submitted to cluster 469609.
         remoteJobIdNumber = re.search('cluster [0-9]+',remoteJobId).group().split()[1] + ".0"
      except:
         remoteJobIdNumber = "-1"

      return(remoteJobIdNumber)


