#!/usr/bin/env python
#
# @package      hubzero-submit-monitors
# @file         Scripts/boinc/fetchBatchOutput.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2017-2018 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2017-2018 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
# ----------------------------------------------------------------------
#  fetchBatchOutput.py
#
#  script which monitors the BOINC queue and reports changes in job status
#
import sys
import os
import pycurl
import subprocess
from errno import EDQUOT

from submit_api import *

BOINCPROJECTURL    = ""
BOINCAUTHENTICATOR = ""


class FetchBoincOutput:
   def __init__(self,
                boincProjectURL,
                boincAuthenticator,
                batchId,
                resultBundle):
      self.boincProjectURL    = boincProjectURL
      self.boincAuthenticator = boincAuthenticator
      self.batchId            = int(batchId)
      self.resultBundle       = resultBundle


   def run(self):
      exitStatus = 0
      batchRequest = CREATE_BATCH_REQ()
      batchRequest.project       = self.boincProjectURL
      batchRequest.authenticator = self.boincAuthenticator
      batchRequest.batch_id      = self.batchId

      fetchURL = get_output_files(batchRequest)

      try:
         with open(self.resultBundle,'wb') as fpResultBundle:
            curl = pycurl.Curl()
            curl.setopt(curl.URL,fetchURL)
            curl.setopt(curl.WRITEDATA,fpResultBundle)
            curl.perform()
            curl.close()
      except IOError as e:
         sys.write.stderr("Error fetching result bundle: %s\n" % (os.strerror(e.args[0])))
         exitStatus = 1
      except pycurl.error:
         sys.write.stderr("curl error fetching result bundle.\n")
         exitStatus = 1
      else:
         if not os.path.exists(self.resultBundle):
            exitStatus = 1
         elif os.path.getsize(self.resultBundle) == 0:
            try:
               os.remove(self.resultBundle)
            except:
               pass
            exitStatus = 1

      return(exitStatus)


if __name__ == '__main__':

   if len(sys.argv) == 3:
      batchId      = sys.argv[1]
      resultBundle = sys.argv[2]
   else:
      sys.exit(1)

   __fetchBoincOutput__ = FetchBoincOutput(BOINCPROJECTURL,BOINCAUTHENTICATOR,
                                           batchId,resultBundle)

   __exitStatus__ = __fetchBoincOutput__.run()

   sys.exit(__exitStatus__)


