# @package      hubzero-submit-distributor
# @file         RemoteJobMonitor.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
import sys
import time
import csv
import os
import copy
import traceback
import logging
from errno import EPIPE

from hubzero.submit.LogMessage        import getLogIDMessage as getLogMessage
from hubzero.submit.MessageConnection import MessageConnection
from hubzero.submit.JobOutput         import JobOutput

class RemoteJobMonitor(MessageConnection):
   def __init__(self,
                listenURI,
                writeStdoutMessages=True,
                writeStderrMessages=True):
      MessageConnection.__init__(self,listenURI)

      self.logger = logging.getLogger(__name__)

      self.writeStdoutMessages = writeStdoutMessages
      self.writeStderrMessages = writeStderrMessages

      self.jobOutput      = JobOutput()
      self.enteredCommand = None
      self.startDate      = None
      self.finishDate     = None

      self.jobStatus = {}
      self.jobStatus['WF'] = {'state': 'waiting', 'message': 'Pending Submission'}
      self.jobStatus['RM'] = {'state': 'waiting', 'message': 'Removed'}
      self.jobStatus['DF'] = {'state': 'waiting', 'message': 'Deferred'}
      self.jobStatus['HS'] = {'state': 'waiting', 'message': 'User & System Hold'}
      self.jobStatus['RP'] = {'state': 'waiting', 'message': 'Remove Pending'}
      self.jobStatus['TX'] = {'state': 'waiting', 'message': 'Terminated'}
      self.jobStatus['NF'] = {'state': 'waiting', 'message': 'Node_Fail'}
      self.jobStatus['TO'] = {'state': 'waiting', 'message': 'Timeout'}
      self.jobStatus['PD'] = {'state': 'waiting', 'message': 'Pending'}
      self.jobStatus['NQ'] = {'state': 'waiting', 'message': 'Not Queued'}
      self.jobStatus['NR'] = {'state': 'waiting', 'message': 'Not Run'}
      self.jobStatus['RJ'] = {'state': 'waiting', 'message': 'Rejected'}
      self.jobStatus['CK'] = {'state': 'executing', 'message': 'Checkpointing'}
      self.jobStatus['C']  = {'state': 'executing', 'message': 'Complete'}
      self.jobStatus['E']  = {'state': 'executing', 'message': 'Exiting'}
      self.jobStatus['D']  = {'state': 'finished', 'message': 'Done'}
      self.jobStatus['F']  = {'state': 'waiting', 'message': 'Failed'}
      self.jobStatus['I']  = {'state': 'waiting', 'message': 'Idle'}
      self.jobStatus['H']  = {'state': 'waiting', 'message': 'Held'}
      self.jobStatus['CA'] = {'state': 'waiting', 'message': 'Cancelled'}
      self.jobStatus['CG'] = {'state': 'executing', 'message': 'Completing'}
      self.jobStatus['CF'] = {'state': 'waiting', 'message': 'Configuring'}
      self.jobStatus['N']  = {'state': 'waiting', 'message': 'Submitted'}
      self.jobStatus['Q']  = {'state': 'waiting', 'message': 'Queued'}
      self.jobStatus['P']  = {'state': 'waiting', 'message': 'Pending'}
      self.jobStatus['S']  = {'state': 'waiting', 'message': 'Suspended'}
      self.jobStatus['R']  = {'state': 'executing', 'message': 'Running'}
      self.jobStatus['T']  = {'state': 'waiting', 'message': 'Moving'}
      self.jobStatus['W']  = {'state': 'waiting', 'message': 'Waiting'}
      self.jobStatus['V']  = {'state': 'waiting', 'message': 'Vacated'}
      self.jobStatus['X']  = {'state': 'aborted', 'message': 'Marked For Deletion'}
      self.jobStatus['XP'] = {'state': 'waiting', 'message': 'Reject Pending'}
      self.jobStatus['CP'] = {'state': 'executing', 'message': 'Complete Pending'}
      self.jobStatus['EP'] = {'state': 'waiting', 'message': 'Preempt Pending'}
      self.jobStatus['VP'] = {'state': 'waiting', 'message': 'Vacate Pending'}
      self.jobStatus['PT'] = {'state': 'waiting', 'message': 'Preempted'}
      self.jobStatus['ST'] = {'state': 'waiting', 'message': 'Starting'}
      self.jobStatus['CD'] = {'state': 'executing', 'message': 'Completed'}
      self.jobStatus['SH'] = {'state': 'waiting', 'message': 'System Hold'}
      self.jobStatus['MP'] = {'state': 'waiting', 'message': 'Resume Pending'}
      self.jobStatus['SE'] = {'state': 'failed', 'message': 'Submission Error'}
      self.jobStatus['EF'] = {'state': 'failed', 'message': 'Execution Failed'}
      self.jobStatus['d']  = {'state': 'aborted', 'message': 'Marked For Deletion'}
      self.jobStatus['e']  = {'state': 'aborted', 'message': 'Submission Error'}
      self.jobStatus['r']  = {'state': 'executing', 'message': 'Running'}
      self.jobStatus['t']  = {'state': 'waiting', 'message': 'Transferring'}
      self.jobStatus['w']  = {'state': 'waiting', 'message': 'Waiting'}
      self.jobStatus['A']  = {'state': 'aborted', 'message': 'Aborted'}
      self.jobStatus['RT'] = {'state': 'finished', 'message': 'Retired'}
      self.jobStatus['K']  = {'state': 'aborted', 'message': 'Aborted'}
      self.jobStatus['KD'] = {'state': 'finished', 'message': 'Done'}

      self.jobStatusReportOrders = {}
      self.jobStatusReportOrders['waiting']    = 5
      self.jobStatusReportOrders['aborted']    = 1
      self.jobStatusReportOrders['setup']      = 6
      self.jobStatusReportOrders['setting up'] = 6
      self.jobStatusReportOrders['failed']     = 3
      self.jobStatusReportOrders['executing']  = 4
      self.jobStatusReportOrders['finished']   = 2


   def __writeToStdout(self,
                       message):
      if self.writeStdoutMessages:
         try:
            sys.stdout.write(message)
            sys.stdout.flush()
         except IOError as e:
            if not e.args[0] in [EPIPE]:
               self.logger.log(logging.ERROR,getLogMessage("Can't write to stdout: %s" % (message)))


   def __writeToStderr(self,
                       message):
      if self.writeStderrMessages:
         try:
            sys.stderr.write(message)
            sys.stderr.flush()
         except IOError as e:
            if not e.args[0] in [EPIPE]:
               self.logger.log(logging.ERROR,getLogMessage("Can't write to stderr: %s" % (message)))


   def postJobRegistration(self,
                           siteName,
                           runName,
                           localJobId,
                           instanceId,
                           hubUserId,
                           submitterClass,
                           distributorPid):
      queryMessage = {'messageType':'registerJob',
                      'siteName':siteName,
                      'runName':runName,
                      'localJobId':localJobId,
                      'instanceId':instanceId,
                      'hubUserId':hubUserId,
                      'submitterClass':submitterClass,
                      'distributorPid':distributorPid}
      nTry,response = self.requestJsonExchange(queryMessage)

      if nTry > 1:
         self.logger.log(logging.WARNING,getLogMessage("confirmation: registerJob(%d):%s" % (nTry,response['messageType'])))


   def deleteJobRegistration(self,
                             localJobId,
                             instanceId):
      queryMessage = {'messageType':'deleteRegisteredJob',
                      'localJobId':localJobId,
                      'instanceId':instanceId}
      nTry,response = self.requestJsonExchange(queryMessage)

      if nTry > 1:
         self.logger.log(logging.WARNING,getLogMessage("confirmation: deleteRegisteredJob(%d):%s" % (nTry,response['messageType'])))


   def isJobReleased(self,
                     localJobId,
                     instanceId):
      jobReleased = False
      queryMessage = {'messageType':'isRegisteredJobReleased',
                      'localJobId':localJobId,
                      'instanceId':instanceId}
      nTry,response = self.requestJsonExchange(queryMessage)

      if nTry > 1:
         self.logger.log(logging.WARNING,getLogMessage("confirmation: isRegisteredJobReleased(%d):%s" % \
                                                                       (nTry,response['messageType'])))

      jobReleased = response['jobReleased']

      return(jobReleased)


   def postJobSubmission(self,
                         siteName,
                         identityNames,
                         siteMonitorDesignator,
                         remoteJobId,
                         hubUserId,
                         tailFiles,
                         enteredCommand,
                         jobWorkDirectory,
                         localJobId,
                         instanceId,
                         destination,
                         runName,
                         nCores,
                         distributorPid):
      self.enteredCommand = enteredCommand
      self.startDate      = time.strftime("%a %b %e %X %Z %Y")

      queryMessage = {'messageType':'postJob',
                      'siteName':siteName,
                      'identityNames':identityNames,
                      'siteDesignator':siteMonitorDesignator,
                      'remoteJobId':remoteJobId,
                      'hubUserId':hubUserId,
                      'jobWorkDirectory':jobWorkDirectory,
                      'localJobId':localJobId,
                      'instanceId':instanceId,
                      'destination':destination,
                      'runName':runName,
                      'nCores':nCores,
                      'distributorPid':distributorPid,
                      'tailFiles':{}}
      for tailFile in tailFiles:
         fileName,nLines = tailFile.split(':')
         queryMessage['tailFiles'][fileName] = {'nLines':int(nLines)}
      nTry,response = self.requestJsonExchange(queryMessage)

      self.logger.log(logging.WARNING,getLogMessage("confirmation: postJob(%d):%s" % (nTry,response['messageType'])))


   def postPegasusWorkflowSubmission(self,
                                     siteName,
                                     identityNames,
                                     siteMonitorDesignator,
                                     remoteJobId,
                                     hubUserId,
                                     enteredCommand,
                                     localJobId,
                                     nInstances,
                                     destination,
                                     runName,
                                     nCores,
                                     distributorPid):
      if nInstances <= 0:
         raise ValueError

      self.enteredCommand = enteredCommand
      self.startDate      = time.strftime("%a %b %e %X %Z %Y")

      instancesInitialJobStatus = []
      queryMessage = {'messageType':'postWorkflow',
                      'siteName':siteName,
                      'identityNames':identityNames,
                      'siteDesignator':siteMonitorDesignator,
                      'remoteJobId':remoteJobId,
                      'hubUserId':hubUserId,
                      'localJobId':localJobId,
                      'nInstances':nInstances,
                      'instancesInitialJobStatus':instancesInitialJobStatus,
                      'destination':destination,
                      'runName':runName,
                      'nCores':nCores,
                      'distributorPid':distributorPid}
      nTry,response = self.requestJsonExchange(queryMessage)

      if nTry > 1:
         self.logger.log(logging.WARNING,getLogMessage("confirmation: postWorkflow(%d):%s" % (nTry,response['messageType'])))


   def postBoincWorkflowSubmission(self,
                                   siteName,
                                   identityNames,
                                   siteMonitorDesignator,
                                   remoteJobId,
                                   hubUserId,
                                   enteredCommand,
                                   localJobId,
                                   nInstances,
                                   instancesInitialJobStatus,
                                   destination,
                                   runName,
                                   nCores,
                                   distributorPid):
      if nInstances <= 0:
         raise ValueError

      self.enteredCommand = enteredCommand
      self.startDate      = time.strftime("%a %b %e %X %Z %Y")

      queryMessage = {'messageType':'postWorkflow',
                      'siteName':siteName,
                      'identityNames':identityNames,
                      'siteDesignator':siteMonitorDesignator,
                      'remoteJobId':remoteJobId,
                      'hubUserId':hubUserId,
                      'localJobId':localJobId,
                      'nInstances':nInstances,
                      'instancesInitialJobStatus':instancesInitialJobStatus,
                      'destination':destination,
                      'runName':runName,
                      'nCores':nCores,
                      'distributorPid':distributorPid}
      nTry,response = self.requestJsonExchange(queryMessage)

      if nTry > 1:
         self.logger.log(logging.WARNING,getLogMessage("confirmation: postWorkflow(%d):%s" % (nTry,response['messageType'])))


   def queryRemoteJobStatus(self,
                            siteMonitorDesignator,
                            remoteJobId):
      queryMessage = {'messageType':'queryJob',
                      'siteDesignator':siteMonitorDesignator,
                      'remoteJobId':remoteJobId}
      nTry,response = self.requestJsonExchange(queryMessage)

      if nTry > 1:
         self.logger.log(logging.WARNING,getLogMessage("confirmation: queryJob(%d):%s" % (nTry,response['messageType'])))

      return(response)


   def queryRemoteJobsStatus(self,
                             jobs):
      queryMessage = {'messageType':'queryJobs',
                      'jobs':jobs}
      nTry,response = self.requestJsonExchange(queryMessage)

      if nTry > 1:
         self.logger.log(logging.WARNING,getLogMessage("confirmation: queryJobs(%d):%s" % (nTry,response['messageType'])))

      return(response['jobsStatus'])


   def queryPegasusWorkflowStatus(self,
                                  siteMonitorDesignator,
                                  remoteJobId,
                                  nInstances):
      queryMessage = {'messageType':'queryWorkflow',
                      'siteDesignator':siteMonitorDesignator,
                      'remoteJobId':remoteJobId,
                      'nInstances':nInstances}
      nTry,response = self.requestJsonExchange(queryMessage)

      if nTry > 1:
         self.logger.log(logging.WARNING,getLogMessage("confirmation: queryWorkflow(%d):%s" % (nTry,response['messageType'])))

      dagStatus = response['status']
      dagStage  = response['stage']
      dagSite   = response['site']

      if dagStage == '?':
         dagStage = "DAG"

      wfInstances = {}
      for instance in response['instances']:
         wfInstances[int(instance)] = {}
         wfInstances[int(instance)]['jobStatus'] = response['instances'][instance]['status']
         wfInstances[int(instance)]['jobStage']  = response['instances'][instance]['stage']
         wfInstances[int(instance)]['jobSite']   = response['instances'][instance]['site']

      return(dagStatus,dagStage,dagSite,wfInstances)


   def queryBoincWorkflowStatus(self,
                                siteMonitorDesignator,
                                remoteJobId,
                                nInstances):
      queryMessage = {'messageType':'queryWorkflow',
                      'siteDesignator':siteMonitorDesignator,
                      'remoteJobId':remoteJobId,
                      'nInstances':nInstances}
      nTry,response = self.requestJsonExchange(queryMessage)

      if nTry > 1:
         self.logger.log(logging.WARNING,getLogMessage("confirmation: queryWorkflow(%d):%s" % (nTry,response['messageType'])))

      dagStatus = response['status']
      dagStage  = response['stage']
      dagSite   = response['site']

      if dagStage == '?':
         dagStage = "DAG"

      wfInstances = {}
      for instance in response['instances']:
         wfInstances[int(instance)] = {}
         wfInstances[int(instance)]['jobStatus'] = response['instances'][instance]['status']
         wfInstances[int(instance)]['jobStage']  = response['instances'][instance]['stage']
         wfInstances[int(instance)]['jobSite']   = response['instances'][instance]['site']

      return(dagStatus,dagStage,dagSite,wfInstances)


   def terminateRemoteJob(self,
                          siteMonitorDesignator,
                          remoteJobId):
      queryMessage = {'messageType':'terminateJob',
                      'siteDesignator':siteMonitorDesignator,
                      'remoteJobId':remoteJobId}
      nTry,response = self.requestJsonExchange(queryMessage)

      self.logger.log(logging.WARNING,getLogMessage("confirmation: terminateJob(%d):%s" % (nTry,response['messageType'])))


   def queryRemoteActiveJobStatus(self,
                                  siteMonitorDesignator,
                                  remoteJobId):
      queryMessage = {'messageType':'reportSites',
                      'siteDesignator':siteMonitorDesignator,
                      'remoteJobId':remoteJobId}
      nTry,response = self.requestJsonExchange(queryMessage)

      if nTry > 1:
         self.logger.log(logging.WARNING,getLogMessage("confirmation: reportSites(%d):%s" % (nTry,response['messageType'])))

      return(response)


   def queryUsersActivity(self,
                          hubUserIds):
      queryMessage = {'messageType':'listUserActivity',
                      'hubUserIds':hubUserIds}
      nTry,response = self.requestJsonExchange(queryMessage)

      if nTry > 1:
         self.logger.log(logging.WARNING,getLogMessage("confirmation: listUserActivity(%d):%s" % (nTry,response['messageType'])))

      usersActivity = {}
      for siteName in response['usersActivity']:
         usersActivity[siteName] = {}
         for user in response['usersActivity'][siteName]:
            usersActivity[siteName][int(user)] = response['usersActivity'][siteName][user]
      response['usersActivity'] = usersActivity

      return(response)


   def queryUsersPriority(self,
                          hubUserIds):
      queryMessage = {'messageType':'listUserPriority',
                      'hubUserIds':hubUserIds}
      nTry,response = self.requestJsonExchange(queryMessage)

      if nTry > 1:
         self.logger.log(logging.WARNING,getLogMessage("confirmation: listUserPriority(%d):%s" % (nTry,response['messageType'])))

      usersPriority = {}
      for siteName in response['usersPriority']:
         usersPriority[siteName] = {}
         for user in response['usersPriority'][siteName]:
            usersPriority[siteName][int(user)] = response['usersPriority'][siteName][user]
      response['usersPriority'] = usersPriority

      return(response)


   def queryUserActiveJobStatus(self,
                                hubUserId):
      queryMessage = {'messageType':'listUserJobs',
                      'hubUserId':hubUserId}
      nTry,response = self.requestJsonExchange(queryMessage)

      if nTry > 1:
         self.logger.log(logging.WARNING,getLogMessage("confirmation: listUserJobs(%d):%s" % (nTry,response['messageType'])))

      reportedJobs = {}
      jobs = response['userActiveJobs']
      if len(jobs) > 0:
         for job in jobs:
            localJobId    = job['localJobId']
            instanceId    = job['instanceId']
            runName       = job['runName']
            jobQueue      = job['jobQueue']
            site          = job['destination']
            jobStatus     = job['jobStatus']
            jobStage      = job['jobStage']
            executionHost = job['executionHost']
            if executionHost != '?':
               jobSite = executionHost
            else:
               jobSite = site
            jobStatusMessage = self.__getJobStatusMessage(jobStatus)
            if jobStage == '?':
               jobStage = 'Job'
            if not localJobId in reportedJobs:
               reportedJobs[localJobId] = {}
            reportedJobs[localJobId][instanceId] = (runName,jobQueue,jobSite,jobStatusMessage,jobStage)

      jobs = response['userRegisteredJobs']
      if len(jobs) > 0:
         for job in jobs:
            localJobId       = job['localJobId']
            instanceId       = job['instanceId']
            runName          = job['runName']
            jobQueue         = '?'
            jobSite          = job['siteName']
            jobStatusMessage = 'Registered'
            jobStage         = 'Registered'
            if not localJobId in reportedJobs:
               reportedJobs[localJobId] = {}
            reportedJobs[localJobId][instanceId] = (runName,jobQueue,jobSite,jobStatusMessage,jobStage)

      return(response['reportTime'],reportedJobs)


   def queryUserActiveJobPid(self,
                             hubUserId,
                             localJobId):
      queryMessage = {'messageType':'getJobPID',
                      'hubUserId':hubUserId,
                      'localJobId':localJobId}
      nTry,response = self.requestJsonExchange(queryMessage)

      if nTry > 1:
         self.logger.log(logging.WARNING,getLogMessage("confirmation: getJobPID(%d):%s" % (nTry,response['messageType'])))

      activeJobPid = response['pid']

      return(activeJobPid)


   def queryIdentityActiveJobUsers(self,
                                   identityName):
      identityActiveJobUsers = []
      queryMessage = {'messageType':'queryIdentityUsers',
                      'identityName':identityName}
      nTry,response = self.requestJsonExchange(queryMessage)

      if nTry > 1:
         self.logger.log(logging.WARNING,getLogMessage("confirmation: queryIdentityUsers(%d):%s" % (nTry,response['messageType'])))

      identityActiveJobUsers = response['identityActiveJobUsers']

      return(identityActiveJobUsers)


   def __getJobStatusMessage(self,
                             jobStatus):
      try:
         jobStatusMessage = self.jobStatus[jobStatus]['message']
      except:
         self.logger.log(logging.ERROR,getLogMessage(traceback.format_exc()))
         jobStatusMessage = 'Unknown Status(%s)' % (jobStatus)

      return(jobStatusMessage)


   def __getJobStatusState(self,
                           jobStatus):
# 'waiting'
# 'aborted'
# 'setting up'
# 'failed'
# 'executing'
# 'finished'

      try:
         jobStatusState = self.jobStatus[jobStatus]['state']
      except:
         self.logger.log(logging.ERROR,getLogMessage(traceback.format_exc()))
         jobStatusState = 'Unknown State(%s)' % (jobStatus)

      return(jobStatusState)


   def __getJobStatusReportOrder(self,
                                 jobStatusState):
      try:
         jobStatusReportOrder = self.jobStatusReportOrders[jobStatusState]
      except:
         jobStatusReportOrder = 99

      return(jobStatusReportOrder)


   @staticmethod
   def __getNewTailLines(newTail,
                         prefix=""):
      newTailLines = ""
      if newTail:
         if newTail[-1] == '\n':
            newTailList = newTail[:-1].split('\n')
         else:
            newTailList = newTail.split('\n')
      else:
         newTailList = []
      if prefix:
         newTailList = [ prefix + ': ' + s for s in newTailList ]

      if newTailList:
         newTailLines = "%s\n" % ('\n'.join(newTailList))

      return(newTailLines)


   @staticmethod
   def __getPegasusRunStatusMessage(runStatusPath):
      runStatusMessage = ""
      if os.path.exists(runStatusPath):
         try:
            fpRunStatus = open(runStatusPath,'r')
            try:
               runStatusText = fpRunStatus.readlines()
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (runStatusPath)))
            else:
               nUnreadyJobs    = 0
               nReadyJobs      = 0
               nPreJobs        = 0
               nQueuedJobs     = 0
               nPostJobs       = 0
               nFailureJobs    = 0
               nSuccessJobs    = 0
               percentJobsDone = "0."
               nTotalJobs      = 0
               for textLine in runStatusText:
                  textLine = textLine.strip()
                  if   textLine.startswith('UNREADY'):
                     nUnreadyJobs = int(textLine.split(':')[1])
                     nTotalJobs += nUnreadyJobs
                  elif textLine.startswith('READY'):
                     nReadyJobs = int(textLine.split(':')[1])
                     nTotalJobs += nReadyJobs
                  elif textLine.startswith('PRE'):
                     nPreJobs = int(textLine.split(':')[1])
                     nTotalJobs += nPreJobs
                  elif textLine.startswith('QUEUED'):
                     nQueuedJobs = int(textLine.split(':')[1])
                     nTotalJobs += nQueuedJobs
                  elif textLine.startswith('POST'):
                     nPostJobs = int(textLine.split(':')[1])
                     nTotalJobs += nPostJobs
                  elif textLine.startswith('SUCCESS'):
                     nSuccessJobs = int(textLine.split(':')[1])
                     nTotalJobs += nSuccessJobs
                  elif textLine.startswith('FAILURE'):
                     nFailureJobs = int(textLine.split(':')[1])
                     nTotalJobs += nFailureJobs
                  elif textLine.startswith('%DONE'):
                     percentJobsDone = textLine.split(':')[1]

               if nTotalJobs > 0:
                  nPendingJobs = nUnreadyJobs+nReadyJobs
                  nSubmittedJobs = nPreJobs+nQueuedJobs+nPostJobs
                  runStatusMessage = "=PEGASUS-PROGRESS=> pending=%d submitted=%d failed=%d success=%d done=%s%%." % \
                                         (nPendingJobs,nSubmittedJobs,nFailureJobs,nSuccessJobs,percentJobsDone)
            finally:
               fpRunStatus.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (runStatusPath)))

      return(runStatusMessage)


   def __updateRedundantStatus(self,
                               waitForJobsInfo):
      runProgressMessage = ""
      jobStatusStates = {}
      for jobStatusState in self.jobStatusReportOrders:
         jobStatusStates[jobStatusState] = 0
      nCompleted = 0
      nInstances = 0
      for instance in waitForJobsInfo:
         if waitForJobsInfo[instance]['state'] == 'released':
            jobStatusState       = self.__getJobStatusState(waitForJobsInfo[instance]['recentJobStatus'])
            jobStatusReportOrder = self.__getJobStatusReportOrder(jobStatusState)
         else:
            jobStatusState       = 'setup'
            jobStatusReportOrder = self.__getJobStatusReportOrder(jobStatusState)

         if not jobStatusState in jobStatusStates:
            jobStatusStates[jobStatusState] = 0
         jobStatusStates[jobStatusState] += 1
         if jobStatusState in ['finished','failed']:
            nCompleted += 1
         nInstances += 1
      percentDone = 100.*float(nCompleted)/float(nInstances)

      jobStatusReports = {}
      for jobStatusState in jobStatusStates:
         jobStatusReportOrder = self.__getJobStatusReportOrder(jobStatusState)
         if not jobStatusReportOrder in jobStatusReports:
            jobStatusReports[jobStatusReportOrder] = {}
         jobStatusReports[jobStatusReportOrder][jobStatusState] = jobStatusStates[jobStatusState]

      activeJobStatusReportOrders = jobStatusReports.keys()
      activeJobStatusReportOrders.sort()

      runProgressMessage = "=SUBMIT-PROGRESS=>"
      for jobStatusReportOrder in activeJobStatusReportOrders:
         for jobStatusState in jobStatusReports[jobStatusReportOrder]:
            runProgressMessage +=" %s=%d" % (jobStatusState.replace(' ','_'), \
                                             jobStatusReports[jobStatusReportOrder][jobStatusState])
      runProgressMessage += " %%done=%.2f" % (percentDone)

      return(runProgressMessage)


   def __updateSweepStatus(self,
                           waitForJobsInfo,
                           parameterCombinationsPath):
      runProgressMessage = ""
      parameterCombinationsDir  = os.path.dirname(parameterCombinationsPath)
      parameterCombinationsBase = os.path.basename(parameterCombinationsPath)
      if '.' in parameterCombinationsBase:
         parameterCombinationsBase = parameterCombinationsBase.split('.')[0]
      tmpParameterCombinationsFile = parameterCombinationsBase + '.tmp'
      tmpParameterCombinationsPath = os.path.join(parameterCombinationsDir,tmpParameterCombinationsFile)
      copyTmpFile = False

      if os.path.exists(parameterCombinationsPath):
         try:
            fpCSVOut = open(tmpParameterCombinationsPath,'wb')
            try:
               fpCSVIn = open(parameterCombinationsPath,'rb')
               try:
                  csvReader = csv.reader(fpCSVIn)
                  parameterNames = csvReader.next()
                  while len(parameterNames) > 0 and parameterNames[0][0] == '#':
                     parameterNames = csvReader.next()

                  jobStatusStates = {}
                  for jobStatusState in self.jobStatusReportOrders:
                     jobStatusStates[jobStatusState] = 0
                  parameterCombinations = {}
                  nCompleted = 0
                  nInstances = 0
                  for parameterCombination in csvReader:
                     instance = int(parameterCombination[0])
                     if instance in waitForJobsInfo:
                        if waitForJobsInfo[instance]['state'] == 'released':
                           jobStatusState       = self.__getJobStatusState(waitForJobsInfo[instance]['recentJobStatus'])
                           jobStatusReportOrder = self.__getJobStatusReportOrder(jobStatusState)
                           parameterCombination[1] = jobStatusState
                        else:
                           jobStatusState       = parameterCombination[1]
                           jobStatusReportOrder = self.__getJobStatusReportOrder(jobStatusState)
                     else:
                        jobStatusState       = parameterCombination[1]
                        jobStatusReportOrder = self.__getJobStatusReportOrder(jobStatusState)
                     if not jobStatusReportOrder in parameterCombinations:
                        parameterCombinations[jobStatusReportOrder] = []
                     parameterCombinations[jobStatusReportOrder].append(parameterCombination)
                     if not jobStatusState in jobStatusStates:
                        jobStatusStates[jobStatusState] = 0
                     jobStatusStates[jobStatusState] += 1
                     if jobStatusState in ['finished','failed','aborted']:
                        nCompleted += 1
                     nInstances += 1
                  percentDone = 100.*float(nCompleted)/float(nInstances)
                  activeJobStatusReportOrders = parameterCombinations.keys()
                  activeJobStatusReportOrders.sort()
               except StopIteration:
                  self.logger.log(logging.ERROR,getLogMessage("%s header row is missing" % (parameterCombinationsPath)))
               except csv.Error:
                  self.logger.log(logging.ERROR,getLogMessage("csv reader failed on %s" % (parameterCombinationsPath)))
               else:
                  try:
                     csvWriter = csv.writer(fpCSVOut)
                     if self.enteredCommand:
                        csvWriter.writerow(('# command: ' + self.enteredCommand,))
                     if self.startDate:
                        csvWriter.writerow(('# started: ' + self.startDate,))
                     if self.finishDate:
                        csvWriter.writerow(('# finished: ' + self.finishDate,))
                     csvWriter.writerow(('# completed: %d/%d jobs' % (nCompleted,nInstances),))
                     csvWriter.writerow(parameterNames)
                     for jobStatusReportOrder in activeJobStatusReportOrders:
                        for parameterCombination in parameterCombinations[jobStatusReportOrder]:
                           csvWriter.writerow(parameterCombination)
                  except csv.Error:
                     self.logger.log(logging.ERROR,getLogMessage("csv writer failed on %s" % (tmpParameterCombinationsPath)))
                  else:
                     copyTmpFile = True

                  jobStatusReports = {}
                  for jobStatusState in jobStatusStates:
                     jobStatusReportOrder = self.__getJobStatusReportOrder(jobStatusState)
                     if not jobStatusReportOrder in jobStatusReports:
                        jobStatusReports[jobStatusReportOrder] = {}
                     jobStatusReports[jobStatusReportOrder][jobStatusState] = jobStatusStates[jobStatusState]

                  activeJobStatusReportOrders = jobStatusReports.keys()
                  activeJobStatusReportOrders.sort()

                  runProgressMessage = "=SUBMIT-PROGRESS=>"
                  for jobStatusReportOrder in activeJobStatusReportOrders:
                     for jobStatusState in jobStatusReports[jobStatusReportOrder]:
                        runProgressMessage +=" %s=%d" % (jobStatusState.replace(' ','_'), \
                                                         jobStatusReports[jobStatusReportOrder][jobStatusState])
                  runProgressMessage += " %%done=%.2f" % (percentDone)
               finally:
                  fpCSVIn.close()
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (parameterCombinationsPath)))
            finally:
               fpCSVOut.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (tmpParameterCombinationsPath)))

      if copyTmpFile:
         os.rename(tmpParameterCombinationsPath,parameterCombinationsPath)

      return(runProgressMessage)


   def __updateWorkflowStatus(self,
                              wfInstances,
                              parameterCombinationsPath):
      runProgressMessage = ""
      parameterCombinationsDir  = os.path.dirname(parameterCombinationsPath)
      parameterCombinationsBase = os.path.basename(parameterCombinationsPath)
      if '.' in parameterCombinationsBase:
         parameterCombinationsBase = parameterCombinationsBase.split('.')[0]
      tmpParameterCombinationsFile = parameterCombinationsBase + '.tmp'
      tmpParameterCombinationsPath = os.path.join(parameterCombinationsDir,tmpParameterCombinationsFile)
      copyTmpFile = False

      if os.path.exists(parameterCombinationsPath):
         try:
            fpCSVOut = open(tmpParameterCombinationsPath,'wb')
            try:
               fpCSVIn = open(parameterCombinationsPath,'rb')
               try:
                  enteredCommand = None
                  startDate      = None
                  finishDate     = None

                  csvReader = csv.reader(fpCSVIn)
                  parameterNames = csvReader.next()
                  while len(parameterNames) > 0 and parameterNames[0][0] == '#':
                     header = ','.join(parameterNames)
                     if   header.startswith('# command: '):
                        enteredCommand = header[len('# command: '):]
                     elif header.startswith('# started: '):
                        startDate = header[len('# started: '):]
                     elif header.startswith('# finished: '):
                        finishDate = header[len('# finished: '):]
                     parameterNames = csvReader.next()

                  jobStatusStates = {}
                  for jobStatusState in self.jobStatusReportOrders:
                     jobStatusStates[jobStatusState] = 0
                  parameterCombinations = {}
                  nCompleted = 0
                  nInstances = 0
                  for parameterCombination in csvReader:
                     instance = int(parameterCombination[0])
                     try:
                        jobStatusState = self.__getJobStatusState(wfInstances[instance]['jobStatus'])
                     except KeyError:
                        jobStatusState       = parameterCombination[1]
                        jobStatusReportOrder = self.__getJobStatusReportOrder(jobStatusState)
                     else:
                        jobStatusState       = self.__getJobStatusState(wfInstances[instance]['jobStatus'])
                        jobStatusReportOrder = self.__getJobStatusReportOrder(jobStatusState)
                        parameterCombination[1] = jobStatusState
                     if not jobStatusReportOrder in parameterCombinations:
                        parameterCombinations[jobStatusReportOrder] = []
                     parameterCombinations[jobStatusReportOrder].append(parameterCombination)
                     if not jobStatusState in jobStatusStates:
                        jobStatusStates[jobStatusState] = 0
                     jobStatusStates[jobStatusState] += 1
                     if jobStatusState in ['finished','failed','aborted']:
                        nCompleted += 1
                     nInstances += 1
                  percentDone = 100.*float(nCompleted)/float(nInstances)
                  activeJobStatusReportOrders = parameterCombinations.keys()
                  activeJobStatusReportOrders.sort()
               except StopIteration:
                  self.logger.log(logging.ERROR,getLogMessage("%s header row is missing" % (parameterCombinationsPath)))
               except csv.Error:
                  self.logger.log(logging.ERROR,getLogMessage("csv reader failed on %s" % (parameterCombinationsPath)))
               else:
                  try:
                     csvWriter = csv.writer(fpCSVOut)
                     if   self.enteredCommand:
                        csvWriter.writerow(('# command: ' + self.enteredCommand,))
                     elif enteredCommand:
                        csvWriter.writerow(('# command: ' + enteredCommand,))
                     if   self.startDate:
                        csvWriter.writerow(('# started: ' + self.startDate,))
                     elif startDate:
                        csvWriter.writerow(('# started: ' + startDate,))
                     if   self.finishDate:
                        csvWriter.writerow(('# finished: ' + self.finishDate,))
                     elif finishDate:
                        csvWriter.writerow(('# finished: ' + finishDate,))
                     csvWriter.writerow(('# completed: %d/%d jobs' % (nCompleted,nInstances),))
                     csvWriter.writerow(parameterNames)
                     for jobStatusReportOrder in activeJobStatusReportOrders:
                        for parameterCombination in parameterCombinations[jobStatusReportOrder]:
                           csvWriter.writerow(parameterCombination)
                  except csv.Error:
                     self.logger.log(logging.ERROR,getLogMessage("csv writer failed on %s" % (tmpParameterCombinationsPath)))
                  else:
                     copyTmpFile = True

                  jobStatusReports = {}
                  for jobStatusState in jobStatusStates:
                     jobStatusReportOrder = self.__getJobStatusReportOrder(jobStatusState)
                     if not jobStatusReportOrder in jobStatusReports:
                        jobStatusReports[jobStatusReportOrder] = {}
                     jobStatusReports[jobStatusReportOrder][jobStatusState] = jobStatusStates[jobStatusState]

                  activeJobStatusReportOrders = jobStatusReports.keys()
                  activeJobStatusReportOrders.sort()

                  runProgressMessage = "=SUBMIT-PROGRESS=>"
                  for jobStatusReportOrder in activeJobStatusReportOrders:
                     for jobStatusState in jobStatusReports[jobStatusReportOrder]:
                        runProgressMessage +=" %s=%d" % (jobStatusState.replace(' ','_'), \
                                                         jobStatusReports[jobStatusReportOrder][jobStatusState])
                  runProgressMessage += " %%done=%.2f" % (percentDone)
               finally:
                  fpCSVIn.close()
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (parameterCombinationsPath)))
            finally:
               fpCSVOut.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (tmpParameterCombinationsPath)))

      if copyTmpFile:
         os.rename(tmpParameterCombinationsPath,parameterCombinationsPath)

      return(runProgressMessage)


   def updateRunStatus(self,
                       runType,
                       waitForJobsInfo,
                       parameterCombinationsPath=None):
      if   runType == 'redundant':
         runProgressMessage = self.__updateRedundantStatus(waitForJobsInfo)
      elif runType == 'sweep':
         runProgressMessage = self.__updateSweepStatus(waitForJobsInfo,
                                                       parameterCombinationsPath)
      elif runType == 'pegasus':
         runProgressMessage = self.__updateWorkflowStatus(waitForJobsInfo,
                                                          parameterCombinationsPath)
      elif runType == 'boinc':
         runProgressMessage = self.__updateWorkflowStatus(waitForJobsInfo,
                                                          parameterCombinationsPath)

      return(runProgressMessage)


   def checkRedundantJobs(self,
                          waitForJobsInfo,
                          progressReport,
                          completeRemoteJobIndexes,
                          minimumDelay,
                          sleepTime,
                          delayTime,
                          nDelays,
                          maximumReportDelay,
                          timeLastReported,
                          nIncompleteJobs,
                          showCurrentRunStatus,
                          previousRunProgressMessage,
                          previous):
      current = copy.copy(previous)

      for instance in waitForJobsInfo:
         if waitForJobsInfo[instance]['state'] == 'released':
            if waitForJobsInfo[instance]['recentJobStatus'] != 'D':
               if waitForJobsInfo[instance]['isBatchJob']:
                  siteMonitorDesignator = waitForJobsInfo[instance]['siteMonitorDesignator']
                  remoteJobId           = waitForJobsInfo[instance]['remoteJobId']
                  knownSite             = waitForJobsInfo[instance]['knownSite']

                  previousRunStatusMessage = previous[instance]['runStatusMessage']
                  if previousRunStatusMessage != None:
                     runStatusPath = os.path.join(waitForJobsInfo[instance]['instanceDirectory'],'work','pegasusstatus.txt')
                     currentRunStatusMessage = self.__getPegasusRunStatusMessage(runStatusPath)
                     if currentRunStatusMessage and currentRunStatusMessage != previousRunStatusMessage:
                        if progressReport == 'pegasus':
                           self.__writeToStdout("(%s) %s\n" % (remoteJobId,currentRunStatusMessage))
                  else:
                     currentRunStatusMessage = None

                  previousJobStatus = previous[instance]['status']
                  previousJobStage  = previous[instance]['stage']
                  previousJobSite   = previous[instance]['site']
                  currentInstance  = self.queryRemoteJobStatus(siteMonitorDesignator,remoteJobId)
                  currentJobStatus = currentInstance['status']
                  currentJobStage  = currentInstance['stage']
                  currentJobSite   = currentInstance['site']
                  currentTailFiles = currentInstance['tailFiles']
                  if currentJobSite == "" or currentJobSite == '?':
                     if knownSite != "":
                        currentJobSite = knownSite
                  if currentJobSite != "" and currentJobSite != '?':
                     waitForJobsInfo[instance]['recentJobSite'] = currentJobSite
                  if currentJobStatus != previousJobStatus or \
                     currentJobStage != previousJobStage or \
                     currentJobSite != previousJobSite:
                     jobStatusMessage = self.__getJobStatusMessage(currentJobStatus)
                     if currentJobSite == "" or currentJobSite == '?':
                        message = "status:%s %s" % (currentJobStage,currentJobStatus)
                        self.logger.log(logging.INFO,getLogMessage(message))
                        if progressReport == 'text':
                           self.__writeToStdout("(%s) %s %s %s\n" % (remoteJobId,
                                                                     currentJobStage,jobStatusMessage,
                                                                     time.ctime()))
                     else:
                        message = "status:%s %s %s" % (currentJobStage,currentJobStatus,currentJobSite)
                        self.logger.log(logging.INFO,getLogMessage(message))
                        if progressReport == 'text':
                           self.__writeToStdout("(%s) %s %s at %s %s\n" % (remoteJobId,
                                                                           currentJobStage,jobStatusMessage,currentJobSite,
                                                                           time.ctime()))
                     waitForJobsInfo[instance]['recentJobStatus'] = currentJobStatus
                     if currentJobStatus == 'D':
                        completeRemoteJobIndexes.append(instance)
                        nIncompleteJobs -= 1
                     timeLastReported = delayTime
                     sleepTime = minimumDelay
                     nDelays = 0
                  else:
                     if delayTime >= (timeLastReported + maximumReportDelay):
                        jobStatusMessage = self.__getJobStatusMessage(currentJobStatus)
                        if progressReport == 'text':
                           if currentJobSite == "" or currentJobSite == '?':
                              self.__writeToStdout("(%s) %s %s %s\n" % (remoteJobId,
                                                                        currentJobStage,jobStatusMessage,
                                                                        time.ctime()))
                           else:
                              self.__writeToStdout("(%s) %s %s at %s %s\n" % (remoteJobId,
                                                                              currentJobStage,jobStatusMessage,currentJobSite,
                                                                              time.ctime()))
                        timeLastReported = delayTime

                  if "#STDOUT#" in currentInstance['tailFiles']:
                     currentTail = currentInstance['tailFiles']["#STDOUT#"]['tailText']
                     try:
                        previousTail = previous[instance]['tailFiles']["#STDOUT#"]['tailText']
                     except:
                        previousTail = ""
                     if currentTail != previousTail:
                        newTailLines = self.__getNewTailLines(currentTail)
                        if newTailLines:
                           if progressReport == 'text' or progressReport == 'submit':
                              self.__writeToStdout(newTailLines)
                  if "#STDERR#" in currentInstance['tailFiles']:
                     currentTail = currentInstance['tailFiles']["#STDERR#"]['tailText']
                     previousTail = previous[instance]['tailFiles']["#STDERR#"]['tailText']
                     if currentTail != previousTail:
                        newTailLines = self.__getNewTailLines(currentTail)
                        if newTailLines:
                           if progressReport == 'text' or progressReport == 'submit':
                              self.__writeToStderr(newTailLines)
                  for tailFile in currentInstance['tailFiles']:
                     if tailFile != "#STDOUT#" and tailFile != "#STDERR#":
                        currentTail = currentInstance['tailFiles'][tailFile]['tailText']
                        previousTail = previous[instance]['tailFiles'][tailFile]['tailText']
                        if currentTail != previousTail:
                           newTailLines = self.__getNewTailLines(currentTail,tailFile)
                           if newTailLines:
                              if progressReport == 'text' or progressReport == 'submit':
                                 self.__writeToStdout(newTailLines)
               else:
                  currentJobStatus        = 'D'
                  currentJobStage         = 'Job'
                  currentJobSite          = ''
                  currentRunStatusMessage = None
                  currentTailFiles        = {}
                  waitForJobsInfo[instance]['recentJobStatus'] = currentJobStatus
                  completeRemoteJobIndexes.append(instance)
                  nIncompleteJobs -= 1

               current[instance]['status']           = currentJobStatus 
               current[instance]['stage']            = currentJobStage  
               current[instance]['site']             = currentJobSite   
               current[instance]['runStatusMessage'] = currentRunStatusMessage
               current[instance]['tailFiles']        = currentTailFiles

      currentRunProgressMessage = self.updateRunStatus('redundant',waitForJobsInfo)
      if showCurrentRunStatus:
         if progressReport == 'submit':
            if currentRunProgressMessage and currentRunProgressMessage != previousRunProgressMessage:
               self.__writeToStdout("%s timestamp=%.1f\n" % (currentRunProgressMessage,time.time()))

      return(current,currentRunProgressMessage,
             nIncompleteJobs,timeLastReported,sleepTime,nDelays)


   def waitForRedundantJobs(self,
                            waitForJobsInfo,
                            progressReport,
                            abortGlobal):
      completeRemoteJobIndexes = []

      minimumDelay = 5       #  5 10 20 40 80 160 320
      maximumDelay = 320
      updateFrequency = 5
      maximumReportDelay = 320

      delayTime = 0
      sleepTime = minimumDelay
      nDelays = 0
      timeLastReported = delayTime

      showCurrentRunStatus = False
      nHeldJobs = 0
      nIncompleteJobs = 0
      previousRunProgressMessage = ""

      previous = {}
      for instance in waitForJobsInfo:
         if waitForJobsInfo[instance]['state'] == 'released':
            if waitForJobsInfo[instance]['recentJobStatus'] != 'D':
               previous[instance] = {}
               previous[instance]['status']           = ""
               previous[instance]['stage']            = ""
               previous[instance]['site']             = ""
               if 'instanceDirectory' in waitForJobsInfo[instance]:
                  previous[instance]['runStatusMessage'] = ""
               else:
                  previous[instance]['runStatusMessage'] = None
               previous[instance]['tailFiles']        = {}
               nIncompleteJobs += 1
               if waitForJobsInfo[instance]['isBatchJob']:
                  showCurrentRunStatus = True
         else:
            nHeldJobs += 1

      current,currentRunProgressMessage, \
              nIncompleteJobs,timeLastReported,sleepTime,nDelays = self.checkRedundantJobs(waitForJobsInfo,
                                                                                           progressReport,
                                                                                           completeRemoteJobIndexes,
                                                                                           minimumDelay,
                                                                                           sleepTime,
                                                                                           delayTime,
                                                                                           nDelays,
                                                                                           maximumReportDelay,
                                                                                           timeLastReported,
                                                                                           nIncompleteJobs,
                                                                                           showCurrentRunStatus,
                                                                                           previousRunProgressMessage,
                                                                                           previous)

      showCurrentRunStatus = True
      while (len(completeRemoteJobIndexes) == 0) and (nIncompleteJobs+nHeldJobs > 0) and not abortGlobal['abortAttempted']:
         nDelays += 1
         time.sleep(sleepTime)
         delayTime += sleepTime
         if nDelays == updateFrequency:
            nDelays = 0
            sleepTime *= 2
            if sleepTime > maximumDelay:
               sleepTime = maximumDelay

         previousRunProgressMessage = currentRunProgressMessage
         previous = current

         current,currentRunProgressMessage, \
                 nIncompleteJobs,timeLastReported,sleepTime,nDelays = self.checkRedundantJobs(waitForJobsInfo,
                                                                                              progressReport,
                                                                                              completeRemoteJobIndexes,
                                                                                              minimumDelay,
                                                                                              sleepTime,
                                                                                              delayTime,
                                                                                              nDelays,
                                                                                              maximumReportDelay,
                                                                                              timeLastReported,
                                                                                              nIncompleteJobs,
                                                                                              showCurrentRunStatus,
                                                                                              previousRunProgressMessage,
                                                                                              previous)
         if nHeldJobs > 0:
            break

      if nHeldJobs == 0 or abortGlobal['abortAttempted']:
         message = "waitForRedundantJobs: nCompleteJobs = %d, nIncompleteJobs = %d, nHeldJobs = %d, abortGlobal = %s" % \
                                  (len(completeRemoteJobIndexes),nIncompleteJobs,nHeldJobs,abortGlobal['abortAttempted'])
         self.logger.log(logging.INFO,getLogMessage(message))

      del previous

      return(completeRemoteJobIndexes)


   def checkSweepJobs(self,
                      waitForJobsInfo,
                      parameterCombinationsPath,
                      progressReport,
                      completeRemoteJobIndexes,
                      minimumDelay,
                      sleepTime,
                      delayTime,
                      nDelays,
                      maximumReportDelay,
                      timeLastReported,
                      nHeldJobs,
                      nIncompleteJobs,
                      previousJobStatuses,
                      previousJobStages,
                      previousJobSites,
                      previousRunProgressMessage,
                      showCurrentRunStatus):
      currentJobStatuses = copy.copy(previousJobStatuses)
      currentJobStages   = copy.copy(previousJobStages)
      currentJobSites    = copy.copy(previousJobSites)

      jobs = []
      for instance in waitForJobsInfo:
         if waitForJobsInfo[instance]['state'] == 'released':
            if waitForJobsInfo[instance]['recentJobStatus'] != 'D':
               if waitForJobsInfo[instance]['isBatchJob']:
                  siteMonitorDesignator = waitForJobsInfo[instance]['siteMonitorDesignator']
                  remoteJobId           = waitForJobsInfo[instance]['remoteJobId']
                  job = {'siteDesignator':siteMonitorDesignator,
                         'remoteJobId':remoteJobId}
                  jobs.append(job)
      jobsStatus = self.queryRemoteJobsStatus(jobs)
      del jobs

      for instance in waitForJobsInfo:
         if waitForJobsInfo[instance]['state'] == 'released':
            if waitForJobsInfo[instance]['recentJobStatus'] != 'D':
               if waitForJobsInfo[instance]['isBatchJob']:
                  siteMonitorDesignator = waitForJobsInfo[instance]['siteMonitorDesignator']
                  remoteJobId           = waitForJobsInfo[instance]['remoteJobId']
                  knownSite             = waitForJobsInfo[instance]['knownSite']
                  previousJobStatus = previousJobStatuses[instance]
                  previousJobStage  = previousJobStages[instance]
                  previousJobSite   = previousJobSites[instance]
                  currentJobStatus = jobsStatus[siteMonitorDesignator][remoteJobId]['status']
                  currentJobStage  = jobsStatus[siteMonitorDesignator][remoteJobId]['stage']
                  currentJobSite   = jobsStatus[siteMonitorDesignator][remoteJobId]['site']

                  if currentJobSite == "" or currentJobSite == '?':
                     if knownSite != "":
                        currentJobSite = knownSite
                  if currentJobSite != "" and currentJobSite != '?':
                     waitForJobsInfo[instance]['recentJobSite'] = currentJobSite
                  if currentJobStatus != previousJobStatus or \
                     currentJobStage != previousJobStage or \
                     currentJobSite != previousJobSite:
                     jobStatusMessage = self.__getJobStatusMessage(currentJobStatus)
                     if currentJobSite == "" or currentJobSite == '?':
                        message = "status:%s %s" % (currentJobStage,currentJobStatus)
                        self.logger.log(logging.INFO,getLogMessage(message))
                     else:
                        message = "status:%s %s %s" % (currentJobStage,currentJobStatus,currentJobSite)
                        self.logger.log(logging.INFO,getLogMessage(message))
                     if progressReport == 'text':
                        if currentJobSite == "" or currentJobSite == '?':
                           self.__writeToStdout("(%s) %s %s %s\n" % (remoteJobId, \
                                                                     currentJobStage,jobStatusMessage, \
                                                                     time.ctime()))
                        else:
                           self.__writeToStdout("(%s) %s %s at %s %s\n" % (remoteJobId, \
                                                                           currentJobStage,jobStatusMessage,currentJobSite, \
                                                                           time.ctime()))
                     waitForJobsInfo[instance]['recentJobStatus'] = currentJobStatus
                     if currentJobStatus == 'D':
                        completeRemoteJobIndexes.append(instance)
                        nIncompleteJobs -= 1
                     timeLastReported = delayTime
                     sleepTime = minimumDelay
                     nDelays = 0
                  else:
                     if delayTime >= (timeLastReported + maximumReportDelay):
                        jobStatusMessage = self.__getJobStatusMessage(currentJobStatus)
                        if progressReport == 'text':
                           if currentJobSite == "" or currentJobSite == '?':
                              self.__writeToStdout("(%s) %s %s %s\n" % (remoteJobId, \
                                                                        currentJobStage,jobStatusMessage, \
                                                                        time.ctime()))
                           else:
                              self.__writeToStdout("(%s) %s %s at %s %s\n" % (remoteJobId, \
                                                                              currentJobStage,jobStatusMessage,currentJobSite, \
                                                                              time.ctime()))
                        timeLastReported = delayTime
               else:
                  currentJobStatus = 'D'
                  currentJobStage  = 'Job'
                  currentJobSite   = ''
                  waitForJobsInfo[instance]['recentJobStatus'] = currentJobStatus
                  completeRemoteJobIndexes.append(instance)
                  nIncompleteJobs -= 1

               currentJobStatuses[instance] = currentJobStatus
               currentJobStages[instance]   = currentJobStage
               currentJobSites[instance]    = currentJobSite

      if nIncompleteJobs == 0 and nHeldJobs == 0:
         if not self.finishDate:
            self.finishDate = time.strftime("%a %b %e %X %Z %Y")

      currentRunProgressMessage = self.updateRunStatus('sweep',waitForJobsInfo,parameterCombinationsPath)
      if showCurrentRunStatus:
         if progressReport == 'submit':
            if currentRunProgressMessage and currentRunProgressMessage != previousRunProgressMessage:
               self.__writeToStdout("%s timestamp=%.1f\n" % (currentRunProgressMessage,time.time()))

      return(currentJobStatuses,currentJobStages,currentJobSites,currentRunProgressMessage,
             nIncompleteJobs,timeLastReported,sleepTime,nDelays)


   def waitForSweepJobs(self,
                        waitForJobsInfo,
                        parameterCombinationsPath,
                        progressReport,
                        abortGlobal):
      completeRemoteJobIndexes = []

      minimumDelay = 5       #  5 10 20 40 80 160 320
      maximumDelay = 320
      updateFrequency = 5
      maximumReportDelay = 320

      delayTime = 0
      sleepTime = minimumDelay
      nDelays = 0
      timeLastReported = delayTime

      previousJobStatuses = {}
      previousJobStages   = {}
      previousJobSites    = {}

      nHeldJobs = 0
      nIncompleteJobs = 0
      showCurrentRunStatus = False
      for instance in waitForJobsInfo:
         if waitForJobsInfo[instance]['state'] == 'released':
            if waitForJobsInfo[instance]['recentJobStatus'] != 'D':
               nIncompleteJobs += 1
               showCurrentRunStatus = True
               previousJobStatuses[instance] = ""
               previousJobStages[instance]   = ""
               previousJobSites[instance]    = ""
         else:
            nHeldJobs += 1
      previousRunProgressMessage = ""

      currentJobStatuses,currentJobStages,currentJobSites, \
                         currentRunProgressMessage,nIncompleteJobs, \
                         timeLastReported,sleepTime,nDelays = self.checkSweepJobs(waitForJobsInfo,
                                                                                  parameterCombinationsPath,
                                                                                  progressReport,
                                                                                  completeRemoteJobIndexes,
                                                                                  minimumDelay,
                                                                                  sleepTime,
                                                                                  delayTime,
                                                                                  nDelays,
                                                                                  maximumReportDelay,
                                                                                  timeLastReported,
                                                                                  nHeldJobs,
                                                                                  nIncompleteJobs,
                                                                                  previousJobStatuses,
                                                                                  previousJobStages,
                                                                                  previousJobSites,
                                                                                  previousRunProgressMessage,
                                                                                  showCurrentRunStatus)

      showCurrentRunStatus = True
      while (len(completeRemoteJobIndexes) == 0) and (nIncompleteJobs+nHeldJobs > 0) and not abortGlobal['abortAttempted']:
         nDelays += 1
         time.sleep(sleepTime)
         delayTime += sleepTime
         if nDelays == updateFrequency:
            nDelays = 0
            sleepTime *= 2
            if sleepTime > maximumDelay:
               sleepTime = maximumDelay

         previousJobStatuses        = currentJobStatuses
         previousJobStages          = currentJobStages
         previousJobSites           = currentJobSites
         previousRunProgressMessage = currentRunProgressMessage

         currentJobStatuses,currentJobStages,currentJobSites, \
                            currentRunProgressMessage,nIncompleteJobs, \
                            timeLastReported,sleepTime,nDelays = self.checkSweepJobs(waitForJobsInfo,
                                                                                     parameterCombinationsPath,
                                                                                     progressReport,
                                                                                     completeRemoteJobIndexes,
                                                                                     minimumDelay,
                                                                                     sleepTime,
                                                                                     delayTime,
                                                                                     nDelays,
                                                                                     maximumReportDelay,
                                                                                     timeLastReported,
                                                                                     nHeldJobs,
                                                                                     nIncompleteJobs,
                                                                                     previousJobStatuses,
                                                                                     previousJobStages,
                                                                                     previousJobSites,
                                                                                     previousRunProgressMessage,
                                                                                     showCurrentRunStatus)

         if nHeldJobs > 0:
            break

      if nHeldJobs == 0 or abortGlobal['abortAttempted']:
         message = "waitForSweepJobs: nCompleteJobs = %d, nIncompleteJobs = %d, nHeldJobs = %d, abortGlobal = %s" % \
                       (len(completeRemoteJobIndexes),nIncompleteJobs,nHeldJobs,abortGlobal['abortAttempted'])
         self.logger.log(logging.INFO,getLogMessage(message))

      del previousJobStatuses
      del previousJobStages
      del previousJobSites

      return(completeRemoteJobIndexes)


   def checkPegasusWorkflowJobs(self,
                                waitForJobsInfo,
                                nInstances,
                                parameterCombinationsPath,
                                progressReport,
                                executeInstance,
                                completeRemoteJobIndexes,
                                minimumDelay,
                                sleepTime,
                                delayTime,
                                nDelays,
                                maximumReportDelay,
                                timeLastReported,
                                nIncompleteJobs,
                                previousJobStatuses,
                                previousJobStages,
                                previousJobSites,
                                previousRunStatusMessages,
                                previousRunProgressMessages):
      currentJobStatuses         = copy.copy(previousJobStatuses)
      currentJobStages           = copy.copy(previousJobStages)
      currentJobSites            = copy.copy(previousJobSites)
      currentRunStatusMessages   = copy.copy(previousRunStatusMessages)
      currentRunProgressMessages = copy.copy(previousRunProgressMessages)

      if waitForJobsInfo[executeInstance]['state'] == 'released':
         if waitForJobsInfo[executeInstance]['recentJobStatus'] != 'D':
            if waitForJobsInfo[executeInstance]['isBatchJob']:
               siteMonitorDesignator = waitForJobsInfo[executeInstance]['siteMonitorDesignator']
               remoteJobId           = waitForJobsInfo[executeInstance]['remoteJobId']
               knownSite             = waitForJobsInfo[executeInstance]['knownSite']
               previousJobStatus          = previousJobStatuses[executeInstance]
               previousJobStage           = previousJobStages[executeInstance]
               previousJobSite            = previousJobSites[executeInstance]
               previousRunStatusMessage   = previousRunStatusMessages[executeInstance]
               previousRunProgressMessage = previousRunProgressMessages[executeInstance]
               currentJobStatus,currentJobStage,currentJobSite,wfInstances = \
                                                                             self.queryPegasusWorkflowStatus(siteMonitorDesignator,
                                                                                                             remoteJobId,nInstances)
               wfExitCodes = self.jobOutput.getInProgressPegasusJobExitCodes( \
                                                                            waitForJobsInfo[executeInstance]['instanceDirectory'], \
                                                                            waitForJobsInfo[executeInstance]['scratchDirectory'])
               for instance in wfExitCodes:
                  if wfExitCodes[instance] == 'EF':
                     if instance in wfInstances:
                        wfInstances[instance]['jobStatus'] = wfExitCodes[instance]
               del wfExitCodes

               instanceChangedStatus = False
               if currentJobStatus == 'D':
                  if not self.finishDate:
                     self.finishDate = time.strftime("%a %b %e %X %Z %Y")
                     instanceChangedStatus = True
               for wfInstance in wfInstances:
                  if wfInstances[wfInstance]['jobStatus'] != waitForJobsInfo[wfInstance]['recentJobStatus']:
                     waitForJobsInfo[wfInstance]['recentJobStatus'] = wfInstances[wfInstance]['jobStatus']
                     instanceChangedStatus = True
                  if wfInstances[wfInstance]['jobSite'] != '?':
                     wfCurrentJobSite = wfInstances[wfInstance]['jobSite']
                     if wfCurrentJobSite != "" and wfCurrentJobSite != '?':
                        waitForJobsInfo[wfInstance]['recentJobSite'] = wfCurrentJobSite
               if instanceChangedStatus:
                  currentRunProgressMessage = self.updateRunStatus('pegasus',wfInstances,parameterCombinationsPath)
                  self.jobOutput.getInProgressPegasusJobStdFiles(waitForJobsInfo[executeInstance]['instanceDirectory'],
                                                                 waitForJobsInfo[executeInstance]['scratchDirectory'])
               else:
                  currentRunProgressMessage = previousRunProgressMessage
               del wfInstances

               if   progressReport == 'pegasus':
                  runStatusPath = os.path.join(waitForJobsInfo[executeInstance]['instanceDirectory'],'work','pegasusstatus.txt')
                  currentRunStatusMessage = self.__getPegasusRunStatusMessage(runStatusPath)
                  if currentRunStatusMessage and currentRunStatusMessage != previousRunStatusMessage:
                     self.__writeToStdout("(%s) %s\n" % (remoteJobId,currentRunStatusMessage))
               elif progressReport == 'submit':
                  currentRunStatusMessage = ""
                  if currentRunProgressMessage and currentRunProgressMessage != previousRunProgressMessage:
                     self.__writeToStdout("%s timestamp=%.1f\n" % (currentRunProgressMessage,time.time()))
               else:
                  currentRunStatusMessage = ""

               if currentJobSite == "" or currentJobSite == '?':
                  if knownSite != "":
                     currentJobSite = knownSite
               if currentJobSite != "" and currentJobSite != '?':
                  waitForJobsInfo[executeInstance]['recentJobSite'] = currentJobSite
               if currentJobStatus != previousJobStatus or \
                  currentJobStage != previousJobStage or \
                  currentJobSite != previousJobSite:
                  jobStatusMessage = self.__getJobStatusMessage(currentJobStatus)
                  if currentJobSite == "" or currentJobSite == '?':
                     message = "status:%s %s" % (currentJobStage,currentJobStatus)
                     self.logger.log(logging.INFO,getLogMessage(message))
                  else:
                     message = "status:%s %s %s" % (currentJobStage,currentJobStatus,currentJobSite)
                     self.logger.log(logging.INFO,getLogMessage(message))
                  if progressReport == 'text':
                     if currentJobSite == "" or currentJobSite == '?':
                        self.__writeToStdout("(%s) %s %s %s\n" % (remoteJobId, \
                                                                  currentJobStage,jobStatusMessage, \
                                                                  time.ctime()))
                     else:
                        self.__writeToStdout("(%s) %s %s at %s %s\n" % (remoteJobId, \
                                                                        currentJobStage,jobStatusMessage,currentJobSite, \
                                                                        time.ctime()))
                  waitForJobsInfo[executeInstance]['recentJobStatus'] = currentJobStatus
                  if currentJobStatus == 'D':
                     completeRemoteJobIndexes.append(executeInstance)
                     nIncompleteJobs -= 1
                  timeLastReported = delayTime
                  sleepTime = minimumDelay
                  nDelays = 0
               else:
                  if delayTime >= (timeLastReported + maximumReportDelay):
                     jobStatusMessage = self.__getJobStatusMessage(currentJobStatus)
                     if progressReport == 'text':
                        if currentJobSite == "" or currentJobSite == '?':
                           self.__writeToStdout("(%s) %s %s %s\n" % (remoteJobId, \
                                                                     currentJobStage,jobStatusMessage, \
                                                                     time.ctime()))
                        else:
                           self.__writeToStdout("(%s) %s %s at %s %s\n" % (remoteJobId, \
                                                                           currentJobStage,jobStatusMessage,currentJobSite, \
                                                                           time.ctime()))
                     timeLastReported = delayTime
            else:
               currentJobStatus          = 'D'
               currentJobStage           = 'Job'
               currentJobSite            = ''
               currentRunStatusMessage   = ""
               currentRunProgressMessage = ""
               if not self.finishDate:
                  self.finishDate = time.strftime("%a %b %e %X %Z %Y")
               waitForJobsInfo[executeInstance]['recentJobStatus'] = currentJobStatus
               completeRemoteJobIndexes.append(executeInstance)
               nIncompleteJobs -= 1

            currentJobStatuses[executeInstance]         = currentJobStatus
            currentJobStages[executeInstance]           = currentJobStage
            currentJobSites[executeInstance]            = currentJobSite
            currentRunStatusMessages[executeInstance]   = currentRunStatusMessage
            currentRunProgressMessages[executeInstance] = currentRunProgressMessage

      return(currentJobStatuses,currentJobStages,currentJobSites,
             currentRunStatusMessages,currentRunProgressMessages,
             nIncompleteJobs,timeLastReported,sleepTime,nDelays)


   def waitForPegasusWorkflowJobs(self,
                                  waitForJobsInfo,
                                  nInstances,
                                  parameterCombinationsPath,
                                  progressReport,
                                  abortGlobal):
      completeRemoteJobIndexes = []

      minimumDelay = 5       #  5 10 20 40 80 160 320
      maximumDelay = 320
      updateFrequency = 5
      maximumReportDelay = 320

      delayTime = 0
      sleepTime = minimumDelay
      nDelays = 0
      timeLastReported = delayTime

      executeInstance = 0
      previousJobStatuses         = {executeInstance:""}
      previousJobStages           = {executeInstance:""}
      previousJobSites            = {executeInstance:""}
      previousRunStatusMessages   = {executeInstance:""}
      previousRunProgressMessages = {executeInstance:""}

      nHeldJobs = 0
      nIncompleteJobs = 0
      if waitForJobsInfo[executeInstance]['state'] == 'released':
         if waitForJobsInfo[executeInstance]['recentJobStatus'] != 'D':
            nIncompleteJobs += 1
      else:
         nHeldJobs += 1

      if nHeldJobs == 0:
         currentJobStatuses,currentJobStages,currentJobSites, \
                            currentRunStatusMessages,currentRunProgressMessages, \
                            nIncompleteJobs, \
                            timeLastReported,sleepTime,nDelays = self.checkPegasusWorkflowJobs(waitForJobsInfo,
                                                                                               nInstances,
                                                                                               parameterCombinationsPath,
                                                                                               progressReport,
                                                                                               executeInstance,
                                                                                               completeRemoteJobIndexes,
                                                                                               minimumDelay,
                                                                                               sleepTime,
                                                                                               delayTime,
                                                                                               nDelays,
                                                                                               maximumReportDelay,
                                                                                               timeLastReported,
                                                                                               nIncompleteJobs,
                                                                                               previousJobStatuses,
                                                                                               previousJobStages,
                                                                                               previousJobSites,
                                                                                               previousRunStatusMessages,
                                                                                               previousRunProgressMessages)

         while (len(completeRemoteJobIndexes) == 0) and (nIncompleteJobs > 0) and not abortGlobal['abortAttempted']:
            nDelays += 1
            time.sleep(sleepTime)
            delayTime += sleepTime
            if nDelays == updateFrequency:
               nDelays = 0
               sleepTime *= 2
               if sleepTime > maximumDelay:
                  sleepTime = maximumDelay

            previousJobStatuses         = currentJobStatuses
            previousJobStages           = currentJobStages
            previousJobSites            = currentJobSites
            previousRunStatusMessages   = currentRunStatusMessages
            previousRunProgressMessages = currentRunProgressMessages

            currentJobStatuses,currentJobStages,currentJobSites, \
                               currentRunStatusMessages,currentRunProgressMessages, \
                               nIncompleteJobs, \
                               timeLastReported,sleepTime,nDelays = self.checkPegasusWorkflowJobs(waitForJobsInfo,
                                                                                                  nInstances,
                                                                                                  parameterCombinationsPath,
                                                                                                  progressReport,
                                                                                                  executeInstance,
                                                                                                  completeRemoteJobIndexes,
                                                                                                  minimumDelay,
                                                                                                  sleepTime,
                                                                                                  delayTime,
                                                                                                  nDelays,
                                                                                                  maximumReportDelay,
                                                                                                  timeLastReported,
                                                                                                  nIncompleteJobs,
                                                                                                  previousJobStatuses,
                                                                                                  previousJobStages,
                                                                                                  previousJobSites,
                                                                                                  previousRunStatusMessages,
                                                                                                  previousRunProgressMessages)
      else:
         time.sleep(sleepTime)

      if nHeldJobs == 0 or abortGlobal['abortAttempted']:
         message = "waitForPegasusWorkflowJobs: nCompleteJobs = %d, nIncompleteJobs = %d, nHeldJobs = %d, abortGlobal = %s" % \
                                      (len(completeRemoteJobIndexes),nIncompleteJobs,nHeldJobs,abortGlobal['abortAttempted'])
         self.logger.log(logging.INFO,getLogMessage(message))

      del previousJobStatuses
      del previousJobStages
      del previousJobSites
      del previousRunStatusMessages
      del previousRunProgressMessages

      return(completeRemoteJobIndexes)


   def checkBoincWorkflowJobs(self,
                              waitForJobsInfo,
                              nInstances,
                              parameterCombinationsPath,
                              progressReport,
                              executeInstance,
                              completeRemoteJobIndexes,
                              minimumDelay,
                              sleepTime,
                              delayTime,
                              nDelays,
                              maximumReportDelay,
                              timeLastReported,
                              nIncompleteJobs,
                              previousJobStatuses,
                              previousJobStages,
                              previousJobSites,
                              previousRunStatusMessages,
                              previousRunProgressMessages):
      currentJobStatuses         = copy.copy(previousJobStatuses)
      currentJobStages           = copy.copy(previousJobStages)
      currentJobSites            = copy.copy(previousJobSites)
      currentRunStatusMessages   = copy.copy(previousRunStatusMessages)
      currentRunProgressMessages = copy.copy(previousRunProgressMessages)

      if waitForJobsInfo[executeInstance]['state'] == 'released':
         if waitForJobsInfo[executeInstance]['recentJobStatus'] != 'D':
            if waitForJobsInfo[executeInstance]['isBatchJob']:
               siteMonitorDesignator = waitForJobsInfo[executeInstance]['siteMonitorDesignator']
               remoteJobId           = waitForJobsInfo[executeInstance]['remoteJobId']
               knownSite             = waitForJobsInfo[executeInstance]['knownSite']
               previousJobStatus          = previousJobStatuses[executeInstance]
               previousJobStage           = previousJobStages[executeInstance]
               previousJobSite            = previousJobSites[executeInstance]
               previousRunStatusMessage   = previousRunStatusMessages[executeInstance]
               previousRunProgressMessage = previousRunProgressMessages[executeInstance]
               currentJobStatus,currentJobStage,currentJobSite,wfInstances = \
                                                                               self.queryBoincWorkflowStatus(siteMonitorDesignator,
                                                                                                             remoteJobId,nInstances)
               instanceChangedStatus = False
               if currentJobStatus == 'D':
                  if not self.finishDate:
                     self.finishDate = time.strftime("%a %b %e %X %Z %Y")
                     instanceChangedStatus = True
               for wfInstance in wfInstances:
                  if wfInstances[wfInstance]['jobStatus'] != waitForJobsInfo[wfInstance]['recentJobStatus']:
                     waitForJobsInfo[wfInstance]['recentJobStatus'] = wfInstances[wfInstance]['jobStatus']
                     instanceChangedStatus = True
                  if wfInstances[wfInstance]['jobSite'] != '?':
                     wfCurrentJobSite = wfInstances[wfInstance]['jobSite']
                     if wfCurrentJobSite != "" and wfCurrentJobSite != '?':
                        waitForJobsInfo[wfInstance]['recentJobSite'] = wfCurrentJobSite
               if instanceChangedStatus:
                  currentRunProgressMessage = self.updateRunStatus('boinc',wfInstances,parameterCombinationsPath)
               else:
                  currentRunProgressMessage = previousRunProgressMessage
               del wfInstances

               if progressReport == 'submit':
                  currentRunStatusMessage = ""
                  if currentRunProgressMessage and currentRunProgressMessage != previousRunProgressMessage:
                     self.__writeToStdout("%s timestamp=%.1f\n" % (currentRunProgressMessage,time.time()))
               else:
                  currentRunStatusMessage = ""

               if currentJobSite == "" or currentJobSite == '?':
                  if knownSite != "":
                     currentJobSite = knownSite
               if currentJobSite != "" and currentJobSite != '?':
                  waitForJobsInfo[executeInstance]['recentJobSite'] = currentJobSite
               if currentJobStatus != previousJobStatus or \
                  currentJobStage != previousJobStage or \
                  currentJobSite != previousJobSite:
                  jobStatusMessage = self.__getJobStatusMessage(currentJobStatus)
                  if currentJobSite == "" or currentJobSite == '?':
                     message = "status:%s %s" % (currentJobStage,currentJobStatus)
                     self.logger.log(logging.INFO,getLogMessage(message))
                  else:
                     message = "status:%s %s %s" % (currentJobStage,currentJobStatus,currentJobSite)
                     self.logger.log(logging.INFO,getLogMessage(message))
                  if progressReport == 'text':
                     if currentJobSite == "" or currentJobSite == '?':
                        self.__writeToStdout("(%s) %s %s %s\n" % (remoteJobId, \
                                                                  currentJobStage,jobStatusMessage, \
                                                                  time.ctime()))
                     else:
                        self.__writeToStdout("(%s) %s %s at %s %s\n" % (remoteJobId, \
                                                                        currentJobStage,jobStatusMessage,currentJobSite, \
                                                                        time.ctime()))
                  waitForJobsInfo[executeInstance]['recentJobStatus'] = currentJobStatus
                  if currentJobStatus == 'D':
                     completeRemoteJobIndexes.append(executeInstance)
                     nIncompleteJobs -= 1
                  timeLastReported = delayTime
                  sleepTime = minimumDelay
                  nDelays = 0
               else:
                  if delayTime >= (timeLastReported + maximumReportDelay):
                     jobStatusMessage = self.__getJobStatusMessage(currentJobStatus)
                     if progressReport == 'text':
                        if currentJobSite == "" or currentJobSite == '?':
                           self.__writeToStdout("(%s) %s %s %s\n" % (remoteJobId, \
                                                                     currentJobStage,jobStatusMessage, \
                                                                     time.ctime()))
                        else:
                           self.__writeToStdout("(%s) %s %s at %s %s\n" % (remoteJobId, \
                                                                           currentJobStage,jobStatusMessage,currentJobSite, \
                                                                           time.ctime()))
                     timeLastReported = delayTime
            else:
               currentJobStatus          = 'D'
               currentJobStage           = 'Job'
               currentJobSite            = ''
               currentRunStatusMessage   = ""
               currentRunProgressMessage = ""
               if not self.finishDate:
                  self.finishDate = time.strftime("%a %b %e %X %Z %Y")
               waitForJobsInfo[executeInstance]['recentJobStatus'] = currentJobStatus
               completeRemoteJobIndexes.append(executeInstance)
               nIncompleteJobs -= 1

            currentJobStatuses[executeInstance]         = currentJobStatus
            currentJobStages[executeInstance]           = currentJobStage
            currentJobSites[executeInstance]            = currentJobSite
            currentRunStatusMessages[executeInstance]   = currentRunStatusMessage
            currentRunProgressMessages[executeInstance] = currentRunProgressMessage

      return(currentJobStatuses,currentJobStages,currentJobSites,
             currentRunStatusMessages,currentRunProgressMessages,
             nIncompleteJobs,timeLastReported,sleepTime,nDelays)


   def updateBoincWorkflowStatus(self,
                                 siteMonitorDesignator,
                                 remoteJobId,
                                 nInstances,
                                 parameterCombinationsPath):
      currentJobStatus,currentJobStage,currentJobSite,wfInstances = self.queryBoincWorkflowStatus(siteMonitorDesignator,
                                                                                                  remoteJobId,nInstances)
      if currentJobStatus == 'D':
         if not self.finishDate:
            self.finishDate = time.strftime("%a %b %e %X %Z %Y")

      currentRunProgressMessage = self.updateRunStatus('boinc',wfInstances,parameterCombinationsPath)

      return(currentJobStatus,currentRunProgressMessage)


   def waitForBoincWorkflowJobs(self,
                                waitForJobsInfo,
                                nInstances,
                                parameterCombinationsPath,
                                progressReport,
                                abortGlobal):
      completeRemoteJobIndexes = []

      minimumDelay = 5       #  5 10 20 40 80 160 320
      maximumDelay = 320
      updateFrequency = 5
      maximumReportDelay = 320

      delayTime = 0
      sleepTime = minimumDelay
      nDelays = 0
      timeLastReported = delayTime

      executeInstance = 0
      previousJobStatuses         = {executeInstance:""}
      previousJobStages           = {executeInstance:""}
      previousJobSites            = {executeInstance:""}
      previousRunStatusMessages   = {executeInstance:""}
      previousRunProgressMessages = {executeInstance:""}

      nHeldJobs = 0
      nIncompleteJobs = 0
      if waitForJobsInfo[executeInstance]['state'] == 'released':
         if waitForJobsInfo[executeInstance]['recentJobStatus'] != 'D':
            nIncompleteJobs += 1
      else:
         nHeldJobs += 1

      if nHeldJobs == 0:
         currentJobStatuses,currentJobStages,currentJobSites, \
                            currentRunStatusMessages,currentRunProgressMessages, \
                            nIncompleteJobs, \
                            timeLastReported,sleepTime,nDelays = self.checkBoincWorkflowJobs(waitForJobsInfo,
                                                                                             nInstances,
                                                                                             parameterCombinationsPath,
                                                                                             progressReport,
                                                                                             executeInstance,
                                                                                             completeRemoteJobIndexes,
                                                                                             minimumDelay,
                                                                                             sleepTime,
                                                                                             delayTime,
                                                                                             nDelays,
                                                                                             maximumReportDelay,
                                                                                             timeLastReported,
                                                                                             nIncompleteJobs,
                                                                                             previousJobStatuses,
                                                                                             previousJobStages,
                                                                                             previousJobSites,
                                                                                             previousRunStatusMessages,
                                                                                             previousRunProgressMessages)

         while (len(completeRemoteJobIndexes) == 0) and (nIncompleteJobs > 0) and not abortGlobal['abortAttempted']:
            nDelays += 1
            time.sleep(sleepTime)
            delayTime += sleepTime
            if nDelays == updateFrequency:
               nDelays = 0
               sleepTime *= 2
               if sleepTime > maximumDelay:
                  sleepTime = maximumDelay

            previousJobStatuses         = currentJobStatuses
            previousJobStages           = currentJobStages
            previousJobSites            = currentJobSites
            previousRunStatusMessages   = currentRunStatusMessages
            previousRunProgressMessages = currentRunProgressMessages

            currentJobStatuses,currentJobStages,currentJobSites, \
                               currentRunStatusMessages,currentRunProgressMessages, \
                               nIncompleteJobs, \
                               timeLastReported,sleepTime,nDelays = self.checkBoincWorkflowJobs(waitForJobsInfo,
                                                                                                nInstances,
                                                                                                parameterCombinationsPath,
                                                                                                progressReport,
                                                                                                executeInstance,
                                                                                                completeRemoteJobIndexes,
                                                                                                minimumDelay,
                                                                                                sleepTime,
                                                                                                delayTime,
                                                                                                nDelays,
                                                                                                maximumReportDelay,
                                                                                                timeLastReported,
                                                                                                nIncompleteJobs,
                                                                                                previousJobStatuses,
                                                                                                previousJobStages,
                                                                                                previousJobSites,
                                                                                                previousRunStatusMessages,
                                                                                                previousRunProgressMessages)
      else:
         time.sleep(sleepTime)

      if nHeldJobs == 0 or abortGlobal['abortAttempted']:
         message = "waitForBoincWorkflowJobs: nCompleteJobs = %d, nIncompleteJobs = %d, nHeldJobs = %d, abortGlobal = %s" % \
                                    (len(completeRemoteJobIndexes),nIncompleteJobs,nHeldJobs,abortGlobal['abortAttempted'])
         self.logger.log(logging.INFO,getLogMessage(message))

      del previousJobStatuses
      del previousJobStages
      del previousJobSites
      del previousRunStatusMessages
      del previousRunProgressMessages

      return(completeRemoteJobIndexes)


   def waitForKilledBatchJobs(self,
                              runType,
                              waitForJobsInfo,
                              progressReport,
                              parameterCombinationsPath=None):
      minimumDelay = 5       #  5 10 20 40 80 160 320
      maximumDelay = 30
      updateFrequency = 5
      maximumReportDelay = 30

      delayTime = 0
      sleepTime = minimumDelay
      nDelays = 0
      timeLastReported = delayTime

      previousJobStatuses = {}
      previousJobStages   = {}
      previousJobSites    = {}

      showCurrentRunStatus = False
      nIncompleteJobs = 0
      for instance in waitForJobsInfo:
         if waitForJobsInfo[instance]['recentJobStatus'] == 'K':
            if waitForJobsInfo[instance]['isBatchJob']:
               siteMonitorDesignator = waitForJobsInfo[instance]['siteMonitorDesignator']
               remoteJobId           = waitForJobsInfo[instance]['remoteJobId']
               knownSite             = waitForJobsInfo[instance]['knownSite']
               currentInstance  = self.queryRemoteJobStatus(siteMonitorDesignator,remoteJobId)
               currentJobStatus = currentInstance['status']
               currentJobStage  = currentInstance['stage']
               currentJobSite   = currentInstance['site']

               showCurrentRunStatus = True

               if currentJobSite == "" or currentJobSite == '?':
                  if knownSite != "":
                     currentJobSite = knownSite
               if currentJobSite != "" and currentJobSite != '?':
                  waitForJobsInfo[instance]['recentJobSite'] = currentJobSite
               jobStatusMessage = self.__getJobStatusMessage(currentJobStatus)
               if jobStatusMessage == 'Unknown Status':
                  message = "waitForKilledBatchJobs: siteMonitor=%s,remoteJobId=%s,jobStatus=%s,jobStage=%s,jobSite=%s" % \
                                        (siteMonitorDesignator,remoteJobId,currentJobStatus,currentJobStage,currentJobSite)
                  self.logger.log(logging.INFO,getLogMessage(message))
               if currentJobSite == "" or currentJobSite == '?':
                  self.logger.log(logging.INFO,getLogMessage("status:%s %s" % (currentJobStage,currentJobStatus)))
                  if progressReport == 'text':
                     self.__writeToStdout("(%s) %s %s %s\n" % (remoteJobId,currentJobStage, \
                                                               jobStatusMessage,time.ctime()))
               else:
                  self.logger.log(logging.INFO,getLogMessage("status:%s %s %s" % (currentJobStage,currentJobStatus,currentJobSite)))
                  if progressReport == 'text':
                     self.__writeToStdout("(%s) %s %s at %s %s\n" % (remoteJobId,currentJobStage, \
                                                                     jobStatusMessage,currentJobSite,time.ctime()))
            else:
               currentJobStatus = 'D'
               currentJobStage  = 'Job'
               currentJobSite   = ''

            if currentJobStatus == 'D':
               waitForJobsInfo[instance]['recentJobStatus'] = 'KD'
            else:
               nIncompleteJobs += 1

            previousJobStatuses[instance] = currentJobStatus
            previousJobStages[instance]   = currentJobStage
            previousJobSites[instance]    = currentJobSite

      currentRunProgressMessage = self.updateRunStatus(runType,waitForJobsInfo,parameterCombinationsPath)
      if showCurrentRunStatus:
         if progressReport == 'submit':
            if currentRunProgressMessage:
               self.__writeToStdout("%s timestamp=%.1f\n" % (currentRunProgressMessage,time.time()))

      while nIncompleteJobs > 0:
         nDelays += 1
         time.sleep(sleepTime)
         delayTime += sleepTime
         if nDelays == updateFrequency:
            nDelays = 0
            sleepTime *= 2
            if sleepTime > maximumDelay:
               sleepTime = maximumDelay

         previousRunProgressMessage = currentRunProgressMessage
         for instance in waitForJobsInfo:
            if waitForJobsInfo[instance]['recentJobStatus'] == 'K':
               if waitForJobsInfo[instance]['isBatchJob']:
                  siteMonitorDesignator = waitForJobsInfo[instance]['siteMonitorDesignator']
                  remoteJobId           = waitForJobsInfo[instance]['remoteJobId']
                  knownSite             = waitForJobsInfo[instance]['knownSite']
                  previousJobStatus = previousJobStatuses[instance]
                  previousJobStage  = previousJobStages[instance]
                  previousJobSite   = previousJobSites[instance]
                  currentInstance  = self.queryRemoteJobStatus(siteMonitorDesignator,remoteJobId)
                  currentJobStatus = currentInstance['status']
                  currentJobStage  = currentInstance['stage']
                  currentJobSite   = currentInstance['site']

                  if currentJobSite == "" or currentJobSite == '?':
                     if knownSite != "":
                        currentJobSite = knownSite
                  if currentJobSite != "" and currentJobSite != '?':
                     waitForJobsInfo[instance]['recentJobSite'] = currentJobSite
                  if   currentJobStatus != previousJobStatus or \
                       currentJobStage != previousJobStage or \
                       currentJobSite != previousJobSite:
                     jobStatusMessage = self.__getJobStatusMessage(currentJobStatus)
                     if jobStatusMessage == 'Unknown Status':
                        message = "waitForKilledBatchJobs: siteMonitor=%s,remoteJobId=%s,jobStatus=%s,jobStage=%s,jobSite=%s" % \
                                              (siteMonitorDesignator,remoteJobId,currentJobStatus,currentJobStage,currentJobSite)
                        self.logger.log(logging.INFO,getLogMessage(message))
                     if currentJobSite == "" or currentJobSite == '?':
                        self.logger.log(logging.INFO,getLogMessage("status:%s %s" % (currentJobStage,currentJobStatus)))
                        if progressReport == 'text':
                           self.__writeToStdout("(%s) %s %s %s\n" % (remoteJobId,currentJobStage, \
                                                                     jobStatusMessage,time.ctime()))
                     else:
                        message = "status:%s %s %s" % (currentJobStage,currentJobStatus,currentJobSite)
                        self.logger.log(logging.INFO,getLogMessage(message))
                        if progressReport == 'text':
                           self.__writeToStdout("(%s) %s %s at %s %s\n" % (remoteJobId,currentJobStage, \
                                                                           jobStatusMessage,currentJobSite,time.ctime()))
                     if currentJobStatus == 'D':
                        waitForJobsInfo[instance]['recentJobStatus'] = 'KD'
                        nIncompleteJobs -= 1
                     timeLastReported = delayTime
                     sleepTime = minimumDelay
                     nDelays = 0
                  elif progressReport == 'text':
                     if delayTime >= (timeLastReported + maximumReportDelay):
                        jobStatusMessage = self.__getJobStatusMessage(currentJobStatus)
                        if currentJobSite == "" or currentJobSite == '?':
                           self.__writeToStdout("(%s) %s %s %s\n" % (remoteJobId,currentJobStage, \
                                                                     jobStatusMessage,time.ctime()))
                        else:
                           self.__writeToStdout("(%s) %s %s at %s %s\n" % (remoteJobId,currentJobStage, \
                                                                           jobStatusMessage,currentJobSite,time.ctime()))
                        timeLastReported = delayTime

                  previousJobStatuses[instance] = currentJobStatus
                  previousJobStages[instance]   = currentJobStage
                  previousJobSites[instance]    = currentJobSite

         currentRunProgressMessage = self.updateRunStatus(runType,waitForJobsInfo,parameterCombinationsPath)
         if progressReport == 'submit':
            if currentRunProgressMessage and currentRunProgressMessage != previousRunProgressMessage:
               self.__writeToStdout("%s timestamp=%.1f\n" % (currentRunProgressMessage,time.time()))

      del previousJobStatuses
      del previousJobStages
      del previousJobSites


