# @package      hubzero-submit-distributor
# @file         RemoteInstantSCRIPT.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import re
import random
import logging

from hubzero.submit.LogMessage        import getLogJobIdMessage as getLogMessage 
from hubzero.submit.ParameterTemplate import ParameterTemplate

try:
   iterRange = xrange
except NameError as e:
   iterRange = range

class RemoteInstantSCRIPT:
   SUBMISSIONSCRIPTCOMMANDPREFIX = ''

   def __init__(self,
                hubUserName,
                hubUserId,
                submitterClass,
                session,
                instanceToken,
                wsJobId,
                runName,
                localJobId,
                instanceId,
                workingDirectory,
                appScriptName,
                isMultiCoreRequest,
                siteInfo,
                toolFilesInfo,
                dockerImageInfo,
                submissionScriptsInfo,
                managerInfo,
                nNodes,
                ppn,
                timeHistoryLogs):
      self.logger                   = logging.getLogger(__name__)
      self.hubUserName              = hubUserName
      self.hubUserId                = hubUserId
      self.submitterClass           = submitterClass
      self.session                  = session
      self.instanceToken            = instanceToken
      self.wsJobId                  = wsJobId
      self.runName                  = runName
      self.localJobId               = localJobId
      self.instanceId               = instanceId
      self.workingDirectory         = workingDirectory
      self.appScriptName            = appScriptName
      self.isMultiCoreRequest       = isMultiCoreRequest
      self.computationMode          = managerInfo['computationMode']
      self.preManagerCommands       = managerInfo['preManagerCommands']
      self.managerCommand           = managerInfo['managerCommand']
      self.postManagerCommands      = managerInfo['postManagerCommands']
      self.nNodes                   = nNodes
      self.ppn                      = ppn
      self.venue                    = siteInfo['venue']
      self.venues                   = siteInfo['venues']
      self.venueIndex               = siteInfo['venueIndex']
      self.submissionScriptCommands = siteInfo['submissionScriptCommands']
      self.toolFilesInfo            = toolFilesInfo
      self.dockerImageInfo          = dockerImageInfo
      self.submissionScriptsInfo    = submissionScriptsInfo
      self.timestampStart           = timeHistoryLogs['timestampStart']
      self.timestampFinish          = timeHistoryLogs['timestampFinish']
      self.timestampTransferred     = timeHistoryLogs['timestampTransferred']
      self.timeResults              = timeHistoryLogs['timeResults']

      self.nodeFileName = ""
      self.nodeList     = []

      self.toolInputTemplateFileName  = ""
      self.toolInputTemplate          = ""
      self.toolOutputTemplateFileName = ""
      self.toolOutputTemplate         = ""


   def __buildMPIFile(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Instant','SCRIPT','mpi')

      commandsSCRIPT = ""
      if self.submissionScriptCommands:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            commandSeparator = "\n%s " % self.SUBMISSIONSCRIPTCOMMANDPREFIX
            commandsSCRIPT = self.SUBMISSIONSCRIPTCOMMANDPREFIX + " " + commandSeparator.join(self.submissionScriptCommands)
         else:
            commandSeparator = "\n"
            commandsSCRIPT = commandSeparator.join(self.submissionScriptCommands)

      substitutions = {}
      substitutions["PREMANAGERCOMMANDS"]       = "\n".join(self.preManagerCommands)
      substitutions["MANAGERCOMMAND"]           = self.managerCommand
      substitutions["POSTMANAGERCOMMANDS"]      = "\n".join(self.postManagerCommands)
      substitutions["NNODES"]                   = self.nNodes
      nProcessors = str(int(self.nNodes)*int(self.ppn))
      substitutions["NPROCESSORS"]              = nProcessors
      substitutions["APPSCRIPTNAME"]            = self.appScriptName
      substitutions["TS_TRANSFERRED"]           = self.timestampTransferred
      substitutions["TS_START"]                 = self.timestampStart
      substitutions["TS_FINISH"]                = self.timestampFinish
      substitutions["TIME_RESULTS"]             = self.timeResults
      substitutions["RUNNAME"]                  = self.runName
      substitutions["JOBID"]                    = self.localJobId
      substitutions["INSTANCEID"]               = self.instanceId
      substitutions["WORKINGDIRECTORY"]         = self.workingDirectory
      substitutions["NODEFILE"]                 = self.nodeFileName
      substitutions["HUBUSERNAME"]              = self.hubUserName
      substitutions["HUBUSERID"]                = str(self.hubUserId)
      substitutions["SUBMISSIONSCRIPTCOMMANDS"] = commandsSCRIPT

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError as e:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (e.args[0])))
      except TypeError:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))
      else:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            submissionScript = re.sub("(\n)*\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),
                                           "\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),submissionScript)

      return(submissionScript)


   def __buildNodeList(self):
      del self.nodeList
      self.nodeList = []
      self.nodeFileName = "%s_%s.machinelist" % (self.localJobId,self.instanceId)

      for core in iterRange(int(self.ppn)):
         self.nodeList.append(self.venue)
      eligible = list(range(len(self.venues)))
      eligible.remove(self.venueIndex)
      for node in iterRange(int(self.nNodes) - 1):
         index = eligible[random.randint(0,len(eligible)-1)]
         enclosure = self.venues[index]
         for core in iterRange(int(self.ppn)):
            self.nodeList.append(enclosure)
         eligible.remove(index)
         if len(eligible) == 0:
            eligible = list(range(len(self.venues)))


   def __buildMatlabPCTFile(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Instant','SCRIPT','matlabpct')

      commandsSCRIPT = ""
      if self.submissionScriptCommands:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            commandSeparator = "\n%s " % self.SUBMISSIONSCRIPTCOMMANDPREFIX
            commandsSCRIPT = self.SUBMISSIONSCRIPTCOMMANDPREFIX + " " + commandSeparator.join(self.submissionScriptCommands)
         else:
            commandSeparator = "\n"
            commandsSCRIPT = commandSeparator.join(self.submissionScriptCommands)

      substitutions = {}
      substitutions["PREMANAGERCOMMANDS"]       = "\n".join(self.preManagerCommands)
      substitutions["POSTMANAGERCOMMANDS"]      = "\n".join(self.postManagerCommands)
      substitutions["APPSCRIPTNAME"]            = self.appScriptName
      substitutions["TS_TRANSFERRED"]           = self.timestampTransferred
      substitutions["TS_START"]                 = self.timestampStart
      substitutions["TS_FINISH"]                = self.timestampFinish
      substitutions["TIME_RESULTS"]             = self.timeResults
      substitutions["WORKINGDIRECTORY"]         = self.workingDirectory
      substitutions["HUBUSERNAME"]              = self.hubUserName
      substitutions["HUBUSERID"]                = str(self.hubUserId)
      substitutions["SUBMISSIONSCRIPTCOMMANDS"] = commandsSCRIPT

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError as e:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (e.args[0])))
      except TypeError:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))
      else:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            submissionScript = re.sub("(\n)*\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),
                                           "\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),submissionScript)

      return(submissionScript)


   def getBatchScript(self):
      batchScriptName = ""
      if self.isMultiCoreRequest:
         if   self.computationMode == 'mpi':
            self.__buildNodeList()
            batchScript = self.__buildMPIFile()
            batchScriptName = "%s_%s.cl" % (self.localJobId,self.instanceId)
         elif self.computationMode == 'parallel':
            batchScript = ""
         elif self.computationMode == 'matlabmpi':
            batchScript = ""
         elif self.computationMode == 'matlabpct':
            batchScript = self.__buildMatlabPCTFile()
            batchScriptName = "%s_%s.cl" % (self.localJobId,self.instanceId)
      else:
         batchScript = ""
      batchScriptExecutable = True

      return(batchScriptName,batchScript,batchScriptExecutable)


   def getBatchLog(self):
      batchLogName = ""

      return(batchLogName)


   def getBatchNodeList(self):
      return(self.nodeFileName,self.nodeList)


   def getBatchToolInputTemplate(self):
      return(self.toolInputTemplateFileName,self.toolInputTemplate)


   def getBatchToolOutputTemplate(self):
      return(self.toolOutputTemplateFileName,self.toolOutputTemplate)


   def getUserDaxExecutables(self):
      daxExecutables = []

      return(daxExecutables)


   def getRemoteJobIdNumber(self,
                            remoteJobId):
      try:
         remoteJobIdNumber = remoteJobId
      except:
         remoteJobIdNumber = "-1"

      return(remoteJobIdNumber)


