# @package      hubzero-submit-distributor
# @file         RemoteBatchTAPIS3.py
# @copyright    Copyright (c) 2020-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2020-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import os.path
import re
import math
import logging
import socket

from hubzero.submit.LogMessage        import getLogJobIdMessage as getLogMessage
from hubzero.submit.ParameterTemplate import ParameterTemplate
from hubzero.submit.TimeConversion    import minTohhmmss

class RemoteBatchTAPIS3:
   def __init__(self,
                session,
                instanceToken,
                wsJobId,
                runName,
                localJobId,
                instanceId,
                tapisSiteInfo,
                submissionScriptsInfo,
                nNodes,
                ppn,
                nGpus,
                gpn,
                memoryMB,
                wallTime):
      self.logger                = logging.getLogger(__name__)
      self.session               = session
      self.instanceToken         = instanceToken
      self.wsJobId               = wsJobId
      self.runName               = runName
      self.localJobId            = localJobId
      self.instanceId            = instanceId
      self.tapisAppId            = tapisSiteInfo['appId']
      self.tapisAppVersion       = tapisSiteInfo['appVersion']
      self.tapisQueue            = tapisSiteInfo['queue']
      if tapisSiteInfo['partition']:
         self.tapisPartition     = "--partition " + tapisSiteInfo['partition']
      else:
         self.tapisPartition     = "--comment PartitionNotSpecified"
      if tapisSiteInfo['account']:
         self.tapisAccount       = "--account " + tapisSiteInfo['account']
      else:
         self.tapisAccount       = "--comment AccountNotSpecified"
      if tapisSiteInfo['qos']:
         self.tapisQOS           = "--qos " + tapisSiteInfo['qos']
      else:
         self.tapisQOS           = "--comment QosNotSpecified"
      if tapisSiteInfo['constraint']:
         self.tapisConstraint    = "--constraint " + tapisSiteInfo['constraint']
      else:
         self.tapisConstraint    = "--comment ConstraintNotSpecified"
      if   nGpus > 0 and gpn == 0:
         tapisGpn = int(math.ceil(float(nGpus) / float(nNodes)))
      elif gpn > 0:
         tapisGpn = gpn
      else:
         tapisGpn = 0
      if tapisGpn > 0:
         self.tapisGpusPerNode   = "--gpus-per-node " + str(tapisGpn)
      else:
         self.tapisGpusPerNode   = "--comment GpusPerNodeNotSpecified"
      if memoryMB > 0:
         self.tapisExecSystem    = tapisSiteInfo['execMemorySystem']
      else:
         self.tapisExecSystem    = tapisSiteInfo['execSystem']
      self.tapisStageSystem      = tapisSiteInfo['stageSystem']
      self.tapisArchiveSystem    = tapisSiteInfo['archiveSystem']
      self.submissionScriptsInfo = submissionScriptsInfo
      self.nNodes                = nNodes
      self.ppn                   = ppn
      self.nGpus                 = nGpus
      self.gpn                   = gpn
      self.memoryMB              = max(100,memoryMB)
      self.wallTime              = math.ceil(wallTime)


   def __buildArchiveFile(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','TAPIS3','archive')

      substitutions = {}
      substitutions["SESSION"]            = self.session
      substitutions["INSTANCETOKEN"]      = self.instanceToken
      substitutions["WSJOBID"]            = self.wsJobId
      substitutions["RUNNAME"]            = self.runName
      substitutions["JOBID"]              = self.localJobId
      substitutions["INSTANCEID"]         = self.instanceId
      substitutions["NNODES"]             = self.nNodes
      substitutions["TAPISMEMORY"]        = self.memoryMB
      substitutions["PPN"]                = self.ppn
      substitutions["WALLTIME"]           = self.wallTime
      substitutions["TAPISAPPID"]         = self.tapisAppId
      substitutions["TAPISAPPVERSION"]    = self.tapisAppVersion
      substitutions["TAPISQUEUE"]         = self.tapisQueue
      substitutions["TAPISPARTITION"]     = self.tapisPartition
      substitutions["TAPISACCOUNT"]       = self.tapisAccount
      substitutions["TAPISQOS"]           = self.tapisQOS
      substitutions["TAPISCONSTRAINT"]    = self.tapisConstraint
      substitutions["TAPISGPUSPERNODE"]   = self.tapisGpusPerNode
      substitutions["TAPISEXECSYSTEM"]    = self.tapisExecSystem
      substitutions["TAPISARCHIVESYSTEM"] = self.tapisArchiveSystem
      substitutions["TAPISSTAGESYSTEM"]   = self.tapisStageSystem
      substitutions["SUBMITHOSTFQDN"]     = socket.getfqdn()

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError as e:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@{%s}\n" % (e.args[0])))
         self.logger.log(logging.ERROR,getLogMessage("in SubmissionScripts/Distributor/Batch/TAPIS3/archive\n"))
      except TypeError:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))

      return(submissionScript)


   def __buildNoArchiveFile(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','TAPIS3','noarchive')

      substitutions = {}
      substitutions["SESSION"]          = self.session
      substitutions["INSTANCETOKEN"]    = self.instanceToken
      substitutions["WSJOBID"]          = self.wsJobId
      substitutions["RUNNAME"]          = self.runName
      substitutions["JOBID"]            = self.localJobId
      substitutions["INSTANCEID"]       = self.instanceId
      substitutions["NNODES"]           = self.nNodes
      substitutions["PPN"]              = self.ppn
      substitutions["TAPISMEMORY"]      = self.memoryMB
      substitutions["WALLTIME"]         = self.wallTime
      substitutions["TAPISAPPID"]       = self.tapisAppId
      substitutions["TAPISAPPVERSION"]  = self.tapisAppVersion
      substitutions["TAPISQUEUE"]       = self.tapisQueue
      substitutions["TAPISPARTITION"]   = self.tapisPartition
      substitutions["TAPISACCOUNT"]     = self.tapisAccount
      substitutions["TAPISQOS"]         = self.tapisQOS
      substitutions["TAPISCONSTRAINT"]  = self.tapisConstraint
      substitutions["TAPISGPUSPERNODE"] = self.tapisGpusPerNode
      substitutions["TAPISEXECSYSTEM"]  = self.tapisExecSystem
      substitutions["TAPISSTAGESYSTEM"] = self.tapisStageSystem
      substitutions["SUBMITHOSTFQDN"]   = socket.getfqdn()

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError as e:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@{%s}\n" % (e.args[0])))
         self.logger.log(logging.ERROR,getLogMessage("in SubmissionScripts/Distributor/Batch/TAPIS3/noarchive\n"))
      except TypeError:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))

      return(submissionScript)


   def getBatchTapisFile(self):
      batchTapisFileName = "%s_%s.tapis3" % (self.localJobId,self.instanceId)
      if self.tapisArchiveSystem:
         tapisJson = self.__buildArchiveFile()
      else:
         tapisJson = self.__buildNoArchiveFile()

      return(batchTapisFileName,tapisJson)


   def getRemoteJobIdNumber(self,
                            remoteJobId):
      remoteJobIdNumber = -1
      try:
         label,jobId = remoteJobId.split(':')
         if label == 'jobId':
            remoteJobIdNumber = jobId.strip()
      except:
         pass
 
      return(remoteJobIdNumber)


