#!/usr/bin/env python
#
# @package      hubzero-submit-monitors
# @file         monitorJobR.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2004-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2004-2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
# ----------------------------------------------------------------------
#  monitorJobR.py
#
#  script which queries monitorJob.py for active job(s) at site
#
import os
import sys
import time
import re
import logging

from hubzero.submit.DaemonsInfo      import DaemonsInfo
from hubzero.submit.RemoteJobMonitor import RemoteJobMonitor

CONFIGURATIONDIRECTORY   = os.path.join(os.sep,'etc','submit')
DAEMONSCONFIGURATIONFILE = 'daemons.conf'

APPLICATIONLOGGER = logging.getLogger('') 

def openLogger():
   class EmptyFilter(logging.Filter):
      """
      This is a filter which rejects empty messages

      """

      def filter(self,record):
         if record.getMessage() == "":
            emptyRecord = True
         else:
            emptyRecord = False

         return(not emptyRecord)

   APPLICATIONLOGGER.setLevel(logging.DEBUG)

   logHandler = logging.StreamHandler()

   emptyFilter = EmptyFilter()
   logHandler.addFilter(emptyFilter)

   logFormatter = logging.Formatter('%(asctime)s %(message)s','[%a %b %d %H:%M:%S %Y]')
   logHandler.setFormatter(logFormatter)
   APPLICATIONLOGGER.addHandler(logHandler)


def natsort(listValues):
   def tryInt(stringValue):
      try:
         return int(stringValue)
      except:
         return stringValue.lower()

   def alphanumKey(stringValue):
      """ Turn a string into a list of string and number chunks.
          "z23a" -> ["z", 23, "a"]
      """
      return [ tryInt(chunk) for chunk in re.split('([0-9]+)', stringValue) ]

   """ Sort the given list in the way that humans expect.
   """
   listValues.sort(key=alphanumKey)


def monitorJobR(listenURI):
   """Report active jobs"""
   if   len(sys.argv) == 3:
      siteDesignator = sys.argv[1]
      localJobId     = sys.argv[2]
   elif len(sys.argv) == 2:
      siteDesignator = sys.argv[1]
      localJobId     = '*'
   else:
      siteDesignator = '*'
      localJobId     = '*'

   remoteJobMonitor = RemoteJobMonitor(listenURI)

   report = remoteJobMonitor.queryRemoteActiveJobStatus(siteDesignator,localJobId)

   ignoreStatus = ['D','Dr']

   whenHeaderPrinted = False
   if len(report['siteDesignators']) > 0:
      siteDesignators = report['siteDesignators']
      for siteDesignator in siteDesignators:
         siteDesignatorHeaderPrinted = False

         queues = siteDesignators[siteDesignator]['queues']
         for queue in queues:
            queueHeaderPrinted = False

            jobIds = queues[queue]
            reportedJobs = {}
            for jobId in jobIds:
               status = queues[queue][jobId]['status']
               stage  = queues[queue][jobId]['stage']
               if not status in reportedJobs:
                  reportedJobs[status] = []
               reportedJobs[status].append(jobId)

            for status in reportedJobs.keys():
               if not status in ignoreStatus:
                  statusHeaderPrinted = False

                  jobIds = reportedJobs[status]
                  if len(jobIds) > 0:
                     if not whenHeaderPrinted:
                        print "Current time              %s" % (time.ctime())
                        print "Last time status reported %s" % (time.ctime(report['maxLastReportTime']))
                        whenHeaderPrinted = True
                     if not siteDesignatorHeaderPrinted:
                        lastReportTime = siteDesignators[siteDesignator]['lastReportTime']
                        print "\nReport for monitor %s" % (siteDesignator)
                        print "   Last time status reported %s" % (time.ctime(float(lastReportTime)))
                        siteDesignatorHeaderPrinted = True
                     if not queueHeaderPrinted:
                        print "   Queue %s" % (queue)
                        queueHeaderPrinted = True
                     if not statusHeaderPrinted:
                        print "      Jobs in status: %s" % (status)
                        statusHeaderPrinted = True

                     natsort(jobIds)
                     for jobId in jobIds:
                        print "         %s" % (jobId)

            del reportedJobs

   if not whenHeaderPrinted:
      print "Current time %s" % (time.ctime())
      print "No jobs currently being processed"
      whenHeaderPrinted = True


if __name__ == '__main__':
   openLogger()

   configFilePath = os.path.join(CONFIGURATIONDIRECTORY,DAEMONSCONFIGURATIONFILE)
   daemonsInfo    = DaemonsInfo(configFilePath)
   listenURI      = daemonsInfo.getDaemonListenURI('jobMonitor','tcp')

   monitorJobR(listenURI)


