<?php  
            require('/srv/new-pegasus.isi.edu/includes/common.php'); 
            pegasus_header("16.2. DAX Generator API");
        ?><div class="breadcrumbs">
<span class="breadcrumb-link"><a href="index.php">Pegasus 4.8.0 User Guide</a></span> &gt; <span class="breadcrumb-link"><a href="api.php">API Reference</a></span> &gt; <span class="breadcrumb-node">DAX Generator API</span>
</div><hr><div class="section">
<div class="titlepage"><div><div><h2 class="title" style="clear: both">
<a name="dax_generator_api"></a>16.2. DAX Generator API</h2></div></div></div>
<div class="toc"><dl class="toc">
<dt><span class="section"><a href="dax_generator_api.php#api-java">16.2.1. The Java DAX Generator API</a></span></dt>
<dt><span class="section"><a href="dax_generator_api.php#api-python">16.2.2. The Python DAX Generator API</a></span></dt>
<dt><span class="section"><a href="dax_generator_api.php#api-perl">16.2.3. The Perl DAX Generator</a></span></dt>
<dt><span class="section"><a href="dax_generator_api.php#api-r">16.2.4. The R DAX Generator API</a></span></dt>
</dl></div>
<p>The DAX generating APIs support Java, Perl, Python, and R. This 
    section will show in each language the necessary code, using Pegasus-provided
    libraries, to generate the diamond DAX example above. There may be minor
    differences in details, e.g. to show-case certain features, but
    effectively all generate the same basic diamond.</p>
<div class="section">
<div class="titlepage"><div><div><h3 class="title">
<a name="api-java"></a>16.2.1. The Java DAX Generator API</h3></div></div></div>
<p>The Java DAX API provided with the Pegasus distribution allows
      easy creation of complex and huge workflows. This API is used by several
      applications to generate their abstract DAX. SCEC, which is Southern
      California Earthquake Center, uses this API in their CyberShake workflow
      generator to generate huge DAX containing 10&amp;rsquor;s of thousands
      of tasks with 100&amp;rsquor;s of thousands of input and output files.
      The <a class="ulink" href="javadoc/index.html" target="_top">Java API</a> is well documented
      using <a class="ulink" href="javadoc/edu/isi/pegasus/planner/dax/ADAG.html" target="_top">Javadoc
      for ADAGs</a> .</p>
<p>The steps involved in creating a DAX using the API are</p>
<div class="orderedlist"><ol class="orderedlist" type="1">
<li class="listitem"><p>Create a new <span class="emphasis"><em>ADAG</em></span> object</p></li>
<li class="listitem"><p>Add any metadata attributes associated with the whole
          workflow.</p></li>
<li class="listitem"><p>Add any Workflow notification elements</p></li>
<li class="listitem"><p>Create <span class="emphasis"><em>File</em></span> objects as necessary. You can
          augment the files with physical information, if you want to include
          them into your DAX. Otherwise, the physical information is
          determined from the replica catalog.</p></li>
<li class="listitem"><p>(Optional) Create <span class="emphasis"><em>Executable</em></span> objects, if
          you want to include your transformation catalog into your DAX.
          Otherwise, the translation of a job/task into executable location
          happens with the transformation catalog.</p></li>
<li class="listitem"><p>Create a new <span class="emphasis"><em>Job</em></span> object.</p></li>
<li class="listitem"><p>Add arguments, files, profiles, notifications and other
          information to the <span class="emphasis"><em>Job</em></span> object</p></li>
<li class="listitem"><p>Add the job object to the <span class="emphasis"><em>ADAG</em></span>
          object</p></li>
<li class="listitem"><p>Repeat step 4-6 as necessary.</p></li>
<li class="listitem"><p>Add all dependencies to the <span class="emphasis"><em>ADAG</em></span>
          object.</p></li>
<li class="listitem"><p>Call the <span class="emphasis"><em>writeToFile()</em></span> method on the
          <span class="emphasis"><em>ADAG</em></span> object to render the XML DAX file.</p></li>
</ol></div>
<p>An example Java code that generates the diamond dax show above is
      listed below. This same code can be found in the Pegasus distribution in
      the <code class="filename">examples/grid-blackdiamond-java</code> directory
      as <code class="filename">BlackDiamonDAX.java</code>:</p>
<a name="api-java-example"></a><pre class="programlisting">/**
 *  Copyright 2007-2008 University Of Southern California
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

import edu.isi.pegasus.planner.dax.*;


/**
 * An example class to highlight how to use the JAVA DAX API to generate a diamond
 * DAX.
 *
 */
public class Diamond {



    public class Diamond {



    public ADAG generate(String site_handle, String pegasus_location) throws Exception {

        java.io.File cwdFile = new java.io.File (".");
        String cwd = cwdFile.getCanonicalPath();

        ADAG dax = new ADAG("diamond");
        dax.addNotification(Invoke.WHEN.start,"/pegasus/libexec/notification/email -t notify@example.com");
        dax.addNotification(Invoke.WHEN.at_end,"/pegasus/libexec/notification/email -t notify@example.com");
        dax.addMetadata( "name", "diamond");
        dax.addMetadata( "createdBy", "Karan Vahi");

        File fa = new File("f.a");
        fa.addPhysicalFile("file://" + cwd + "/f.a", "local");
        fa.addMetaData( "size", "1024" );
        dax.addFile(fa);

        File fb1 = new File("f.b1");
        File fb2 = new File("f.b2");
        File fc1 = new File("f.c1");
        File fc2 = new File("f.c2");
        File fd = new File("f.d");
        fd.setRegister(true);

        Executable preprocess = new Executable("pegasus", "preprocess", "4.0");
        preprocess.setArchitecture(Executable.ARCH.X86).setOS(Executable.OS.LINUX);
        preprocess.setInstalled(true);
        preprocess.addPhysicalFile("file://" + pegasus_location + "/bin/keg", site_handle);
        preprocess.addMetaData( "size", "2048" );

        Executable findrange = new Executable("pegasus", "findrange", "4.0");
        findrange.setArchitecture(Executable.ARCH.X86).setOS(Executable.OS.LINUX);
        findrange.setInstalled(true);
        findrange.addPhysicalFile("file://" + pegasus_location + "/bin/keg", site_handle);

        Executable analyze = new Executable("pegasus", "analyze", "4.0");
        analyze.setArchitecture(Executable.ARCH.X86).setOS(Executable.OS.LINUX);
        analyze.setInstalled(true);
        analyze.addPhysicalFile("file://" + pegasus_location + "/bin/keg", site_handle);

        dax.addExecutable(preprocess).addExecutable(findrange).addExecutable(analyze);

        // Add a preprocess job
        Job j1 = new Job("j1", "pegasus", "preprocess", "4.0");
        j1.addArgument("-a preprocess -T 60 -i ").addArgument(fa);
        j1.addArgument("-o ").addArgument(fb1);
        j1.addArgument(" ").addArgument(fb2);
        j1.addMetadata( "time", "60" );
        j1.uses(fa, File.LINK.INPUT);
        j1.uses(fb1, File.LINK.OUTPUT);
        j1.uses(fb2, File.LINK.OUTPUT);
        j1.addNotification(Invoke.WHEN.start,"/pegasus/libexec/notification/email -t notify@example.com");
        j1.addNotification(Invoke.WHEN.at_end,"/pegasus/libexec/notification/email -t notify@example.com");
        dax.addJob(j1);

        // Add left Findrange job
        Job j2 = new Job("j2", "pegasus", "findrange", "4.0");
        j2.addArgument("-a findrange -T 60 -i ").addArgument(fb1);
        j2.addArgument("-o ").addArgument(fc1);
        j2.addMetadata( "time", "60" );
        j2.uses(fb1, File.LINK.INPUT);
        j2.uses(fc1, File.LINK.OUTPUT);
        j2.addNotification(Invoke.WHEN.start,"/pegasus/libexec/notification/email -t notify@example.com");
        j2.addNotification(Invoke.WHEN.at_end,"/pegasus/libexec/notification/email -t notify@example.com");
        dax.addJob(j2);

        // Add right Findrange job
        Job j3 = new Job("j3", "pegasus", "findrange", "4.0");
        j3.addArgument("-a findrange -T 60 -i ").addArgument(fb2);
        j3.addArgument("-o ").addArgument(fc2);
        j3.addMetadata( "time", "60" );
        j3.uses(fb2, File.LINK.INPUT);
        j3.uses(fc2, File.LINK.OUTPUT);
        j3.addNotification(Invoke.WHEN.start,"/pegasus/libexec/notification/email -t notify@example.com");
        j3.addNotification(Invoke.WHEN.at_end,"/pegasus/libexec/notification/email -t notify@example.com");
        dax.addJob(j3);

        // Add analyze job
        Job j4 = new Job("j4", "pegasus", "analyze", "4.0");
        j4.addArgument("-a analyze -T 60 -i ").addArgument(fc1);
        j4.addArgument(" ").addArgument(fc2);
        j4.addArgument("-o ").addArgument(fd);
        j4.addMetadata( "time", "60" );
        j4.uses(fc1, File.LINK.INPUT);
        j4.uses(fc2, File.LINK.INPUT);
        j4.uses(fd, File.LINK.OUTPUT);
        j4.addNotification(Invoke.WHEN.start,"/pegasus/libexec/notification/email -t notify@example.com");
        j4.addNotification(Invoke.WHEN.at_end,"/pegasus/libexec/notification/email -t notify@example.com");
        dax.addJob(j4);

        dax.addDependency("j1", "j2");
        dax.addDependency("j1", "j3");
        dax.addDependency("j2", "j4");
        dax.addDependency("j3", "j4");
        return dax;
    }

    /**
     * Create an example DIAMOND DAX
     * @param args
     */
    public static void main(String[] args) {
        if (args.length != 1) {
            System.out.println("Usage: java GenerateDiamondDAX  &lt;pegasus_location&gt; ");
            System.exit(1);
        }

        try {
            Diamond diamond = new Diamond();
            String pegasusHome = args[0];
            String site = "TestCluster";
            ADAG dag = diamond.generate( site, pegasusHome );
            dag.writeToSTDOUT();
            //generate(args[0], args[1]).writeToFile(args[2]);
        }
        catch (Exception e) {
            e.printStackTrace();
        }

    }
}

</pre>
<p>Of course, you will have to set up some catalogs and properties to
      run this example. The details are catpured in the examples directory
      <code class="filename">examples/grid-blackdiamond-java</code>.</p>
</div>
<div class="section">
<div class="titlepage"><div><div><h3 class="title">
<a name="api-python"></a>16.2.2. The Python DAX Generator API</h3></div></div></div>
<p>Refer to the <a class="ulink" href="python/" target="_top">auto-generated python
      documentation</a> explaining this API.</p>
<a name="api-python-example"></a><pre class="programlisting">#!/usr/bin/env python

from Pegasus.DAX3 import *

# Create a DAX
diamond = ADAG("diamond")

# Add some metadata
diamond.metadata("name", "diamond")
diamond.metadata("createdby", "Gideon Juve")

# Add input file to the DAX-level replica catalog
a = File("f.a")
a.addPFN(PFN("gsiftp://site.com/inputs/f.a","site"))
a.metadata("size", "1024")
diamond.addFile(a)

# Add executables to the DAX-level replica catalog
e_preprocess = Executable(namespace="diamond", name="preprocess", version="4.0", os="linux", arch="x86_64")
e_preprocess.metadata("size", "2048")
e_preprocess.addPFN(PFN("gsiftp://site.com/bin/preprocess","site"))
diamond.addExecutable(e_preprocess)

e_findrange = Executable(namespace="diamond", name="findrange", version="4.0", os="linux", arch="x86_64")
e_findrange.addPFN(PFN("gsiftp://site.com/bin/findrange","site"))
diamond.addExecutable(e_findrange)

e_analyze = Executable(namespace="diamond", name="analyze", version="4.0", os="linux", arch="x86_64")
e_analyze.addPFN(PFN("gsiftp://site.com/bin/analyze","site"))
diamond.addExecutable(e_analyze)

# Add a preprocess job
preprocess = Job(e_preprocess)
preprocess.metadata("time", "60")
b1 = File("f.b1")
b2 = File("f.b2")
preprocess.addArguments("-a preprocess","-T60","-i",a,"-o",b1,b2)
preprocess.uses(a, link=Link.INPUT)
preprocess.uses(b1, link=Link.OUTPUT, transfer=True)
preprocess.uses(b2, link=Link.OUTPUT, transfer=True)
diamond.addJob(preprocess)

# Add left Findrange job
frl = Job(e_findrange)
frl.metadata("time", "60")
c1 = File("f.c1")
frl.addArguments("-a findrange","-T60","-i",b1,"-o",c1)
frl.uses(b1, link=Link.INPUT)
frl.uses(c1, link=Link.OUTPUT, transfer=True)
diamond.addJob(frl)

# Add right Findrange job
frr = Job(e_findrange)
frr.metadata("time", "60")
c2 = File("f.c2")
frr.addArguments("-a findrange","-T60","-i",b2,"-o",c2)
frr.uses(b2, link=Link.INPUT)
frr.uses(c2, link=Link.OUTPUT, transfer=True)
diamond.addJob(frr)

# Add Analyze job
analyze = Job(e_analyze)
analyze.metadata("time", "60")
d = File("f.d")
analyze.addArguments("-a analyze","-T60","-i",c1,c2,"-o",d)
analyze.uses(c1, link=Link.INPUT)
analyze.uses(c2, link=Link.INPUT)
analyze.uses(d, link=Link.OUTPUT, transfer=True, register=True)
diamond.addJob(analyze)

# Add dependencies
diamond.depends(parent=preprocess, child=frl)
diamond.depends(parent=preprocess, child=frr)
diamond.depends(parent=frl, child=analyze)
diamond.depends(parent=frr, child=analyze)

# Write the DAX to stdout
import sys
diamond.writeXML(sys.stdout)

# Write the DAX to a file
f = open("diamond.dax","w")
diamond.writeXML(f)
f.close()</pre>
</div>
<div class="section">
<div class="titlepage"><div><div><h3 class="title">
<a name="api-perl"></a>16.2.3. The Perl DAX Generator</h3></div></div></div>
<p>The Perl API example below can be found in file
      <code class="filename">blackdiamond.pl</code> in directory <code class="filename">examples/grid-blackdiamond-perl</code>. It
      requires that you set the environment variable
      <code class="envar">PEGASUS_HOME</code> to the installation directory of Pegasus,
      and include into <code class="envar">PERL5LIB</code> the path to the directory
      <code class="filename">lib/perl</code> of the Pegasus
      installation. The actual code is longer, and will not require these
      settings, only the example below does. The Perl API is documented using
      <a class="ulink" href="perl/" target="_top">perldoc</a>.
      For each of the modules you can invoke
      <span class="application">perldoc</span>, if your <code class="envar">PERL5LIB</code>
      variable is set.</p>
<p>The steps to generate a DAX from Perl are similar to the Java
      steps. However, since most methods to the classes are deeply within the
      Perl class modules, the convenience module
      <code class="code">Perl::DAX::Factory</code> makes most constructors accessible
      without you needing to type your fingers raw:</p>
<div class="orderedlist"><ol class="orderedlist" type="1">
<li class="listitem"><p>Create a new <span class="emphasis"><em>ADAG</em></span> object.</p></li>
<li class="listitem"><p>Create <span class="emphasis"><em>Job</em></span> objects as necessary.</p></li>
<li class="listitem"><p>As example, the required input file "f.a" is declared as
          <span class="emphasis"><em>File</em></span> object and linked to the
          <span class="emphasis"><em>ADAG</em></span> object.</p></li>
<li class="listitem"><p>The first job arguments and files are filled into the job, and
          the job is added to the <span class="emphasis"><em>ADAG</em></span> object.</p></li>
<li class="listitem"><p>Repeat step 4 for the remaining jobs.</p></li>
<li class="listitem"><p>Add dependencies for all jobs. You have the option of
          assigning label text to edges, though these are not used
          (yet).</p></li>
<li class="listitem"><p>To generate the DAX file, invoke the
          <span class="emphasis"><em>toXML()</em></span> method on the <span class="emphasis"><em>ADAG</em></span>
          object. The first argument is an opened file handle or
          <code class="code">IO::Handle</code> descriptor scalar to write to, the second
          the default indentation for the root element, and the third the XML
          namespace to use for elements and attributes. The latter is
          typically unused unless you want to include your output into another
          XML document.</p></li>
</ol></div>
<pre class="programlisting">#!/usr/bin/env perl
#
use 5.006;
use strict;
use IO::Handle;
use Cwd;
use File::Spec;
use File::Basename;
use Sys::Hostname;
use POSIX ();

BEGIN { $ENV{'PEGASUS_HOME'} ||= `pegasus-config --nocrlf --home` }
use lib File::Spec-&gt;catdir( $ENV{'PEGASUS_HOME'}, 'lib', 'perl' );

use Pegasus::DAX::Factory qw(:all);
use constant NS =&gt; 'diamond';

my $adag = newADAG( name =&gt; NS );

# Workflow MetaData
my $meta = newMetaData('name', 'diamond');
$adag-&gt;addMetaData( $meta );
$adag-&gt;metaData( 'createdBy', 'Rajiv Mayani' );

my $job1 = newJob( namespace =&gt; NS, name =&gt; 'preprocess', version =&gt; '2.0' );
my $job2 = newJob( namespace =&gt; NS, name =&gt; 'findrange', version =&gt; '2.0' );
my $job3 = newJob( namespace =&gt; NS, name =&gt; 'findrange', version =&gt; '2.0' );
my $job4 = newJob( namespace =&gt; NS, name =&gt; 'analyze', version =&gt; '2.0' );

# create "f.a" locally
my $fn = "f.a";
open( F, "&gt;$fn" ) || die "FATAL: Unable to open $fn: $!\n";
my @now = gmtime();
printf F "%04u-%02u-%02u %02u:%02u:%02uZ\n",
        $now[5]+1900, $now[4]+1, @now[3,2,1,0];
close F;

my $file = newFile( name =&gt; 'f.a' );
$file-&gt;addPFN( newPFN( url =&gt; 'file://' . Cwd::abs_path($fn),
                       site =&gt; 'local' ) );
$file-&gt;metaData( 'size', '1024' );
$adag-&gt;addFile($file);

# follow this path, if the PEGASUS_HOME was determined
if ( exists $ENV{'PEGASUS_HOME'} ) {
    my $keg = File::Spec-&gt;catfile( $ENV{'PEGASUS_HOME'}, 'bin', 'keg' );
    my @os = POSIX::uname();
    # $os[2] =~ s/^(\d+(\.\d+(\.\d+)?)?).*/$1/;  ## create a proper osversion
    $os[4] =~ s/i.86/x86/;

    # add Executable instances to DAX-included TC. This will only work,
    # if we know how to access the keg executable. HOWEVER, for a grid
    # workflow, these entries are not used, and you need to
    # [1] install the work tools remotely
    # [2] create a TC with the proper entries
    if ( -x $keg ) {
        for my $j ( $job1, $job2, $job4 ) {
            my $app = newExecutable( namespace =&gt; $j-&gt;namespace,
                                     name =&gt; $j-&gt;name,
                                     version =&gt; $j-&gt;version,
                                     installed =&gt; 'false',
                                     arch =&gt; $os[4],
                                     os =&gt; lc($^O) );
            $app-&gt;addProfile( 'globus', 'maxtime', '2' );
            $app-&gt;addProfile( 'dagman', 'RETRY', '3' );
            $app-&gt;addPFN( newPFN( url =&gt; "file://$keg", site =&gt; 'local' ) );
            $app1-&gt;metaData( 'size', '2048' );
            $adag-&gt;addExecutable($app);
        }
    }
}

my %hash = ( link =&gt; LINK_OUT, register =&gt; 'false', transfer =&gt; 'true' );
my $fna = newFilename( name =&gt; $file-&gt;name, link =&gt; LINK_IN );
my $fnb1 = newFilename( name =&gt; 'f.b1', %hash );
my $fnb2 = newFilename( name =&gt; 'f.b2', %hash );
$job1-&gt;addArgument( '-a', $job1-&gt;name, '-T60', '-i', $fna,
                    '-o', $fnb1, $fnb2 );
$job1-&gt;metaData( 'time', '60' );
$adag-&gt;addJob($job1);

my $fnc1 = newFilename( name =&gt; 'f.c1', %hash );
$fnb1-&gt;link( LINK_IN );
$job2-&gt;addArgument( '-a', $job2-&gt;name, '-T60', '-i', $fnb1,
                    '-o', $fnc1 );
$job2-&gt;metaData( 'time', '60' );
$adag-&gt;addJob($job2);

my $fnc2 = newFilename( name =&gt; 'f.c2', %hash );
$fnb2-&gt;link( LINK_IN );
$job3-&gt;addArgument( '-a', $job3-&gt;name, '-T60', '-i', $fnb2,
                    '-o', $fnc2 );
$job3-&gt;metaData( 'time', '60' );
$adag-&gt;addJob($job3);
# a convenience function -- you can specify multiple dependents
$adag-&gt;addDependency( $job1, $job2, $job3 );

my $fnd = newFilename( name =&gt; 'f.d', %hash );
$fnc1-&gt;link( LINK_IN );
$fnc2-&gt;link( LINK_IN );
$job4-&gt;separator('');                # just to show the difference wrt default
$job4-&gt;addArgument( '-a ', $job4-&gt;name, ' -T60 -i ', $fnc1, ' ', $fnc2,
                    ' -o ', $fnd );
$job4-&gt;metaData( 'time', '60' );
$adag-&gt;addJob($job4);
# this is a convenience function adding parents to a child.
# it is clearer than overloading addDependency
$adag-&gt;addInverse( $job4, $job2, $job3 );

# workflow level notification in case of failure
# refer to Pegasus::DAX::Invoke for details
my $user = $ENV{USER} || $ENV{LOGNAME} || scalar getpwuid($&gt;);
$adag-&gt;invoke( INVOKE_ON_ERROR,
               "/bin/mailx -s 'blackdiamond failed' $user" );

my $xmlns = shift;
$adag-&gt;toXML( \*STDOUT, '', $xmlns );</pre>
</div>
<div class="section">
<div class="titlepage"><div><div><h3 class="title">
<a name="api-r"></a>16.2.4. The R DAX Generator API</h3></div></div></div>
<p>The R DAX API provided with the Pegasus distribution allows easy 
      creation of complex and large workflows in R environments. The API follows 
      the <a class="ulink" href="http://google.github.io/styleguide/Rguide.xml" target="_top">Google' R 
      style guide</a>, and all objects and methods are defined using the 
      <span class="emphasis"><em>S3</em></span> OOP system.</p>
<p>The API can be installed as follows:</p>
<div class="orderedlist"><ol class="orderedlist" type="1"><li class="listitem">
<p>Installing from source package (.tar.gz) in an R 
               environment:</p>
<pre class="programlisting">install.packages("/path/to/source/package.tar.gz", repo=NULL)</pre>
<p>The source package can be obtained using 
              <code class="code">pegasus-config --r</code> or from the <a class="ulink" href="http://pegasus.isi.edu/downloads" target="_top">Pegasus' downloads</a> 
              page.</p>
</li></ol></div>
<p>The R API is well documented using <a class="ulink" href="http://http://roxygen.org" target="_top">Roxygen</a>. In an R environment, it
      can be accessed using <code class="code">help(package=dax3)</code>. A <a class="ulink" href="r/dax3-manual.pdf" target="_top">PDF manual</a> is also available.</p>
<p>The steps involved in creating a DAX using the API are</p>
<div class="orderedlist"><ol class="orderedlist" type="1">
<li class="listitem"><p>Create a new <span class="emphasis"><em>ADAG</em></span> object</p></li>
<li class="listitem"><p>Add any metadata attributes associated with the whole
          workflow.</p></li>
<li class="listitem"><p>Add any Workflow notification elements.</p></li>
<li class="listitem"><p>Create <span class="emphasis"><em>File</em></span> objects as necessary. You can
          augment the files with physical information, if you want to include
          them into your DAX. Otherwise, the physical information is
          determined from the replica catalog.</p></li>
<li class="listitem"><p>(Optional) Create <span class="emphasis"><em>Executable</em></span> objects, if
          you want to include your transformation catalog into your DAX.
          Otherwise, the translation of a job/task into executable location
          happens with the transformation catalog.</p></li>
<li class="listitem"><p>Create a new <span class="emphasis"><em>Job</em></span> object.</p></li>
<li class="listitem"><p>Add arguments, files, profiles, notifications and other
          information to the <span class="emphasis"><em>Job</em></span> object</p></li>
<li class="listitem"><p>Add the job object to the <span class="emphasis"><em>ADAG</em></span>
          object</p></li>
<li class="listitem"><p>Repeat step 4-6 as necessary.</p></li>
<li class="listitem"><p>Add all dependencies to the <span class="emphasis"><em>ADAG</em></span>
          object.</p></li>
<li class="listitem"><p>Call the <code class="code">WriteXML()</code> method on the
          <span class="emphasis"><em>ADAG</em></span> object to render the XML DAX file.</p></li>
</ol></div>
<p>An example R code that generates the diamond dax show previously is
      listed below. A workflow example code can be found in the Pegasus 
      distribution in the <code class="filename">examples/grid-blackdiamond-r</code> directory
      as <code class="filename">blackdiamond.R</code>:</p>
<a name="api-r-example"></a><pre class="programlisting">#!/usr/bin/Rscript
#
library(dax3)

# Create a DAX
diamond &lt;- ADAG("diamond")

# Add some metadata
diamond &lt;- Metadata(diamond, "name", "diamond")
diamond &lt;- Metadata(diamond, "createdby", "Rafael Ferreira da Silva")

# Add input file to the DAX-level replica catalog
a &lt;- File("f.a")
a &lt;- AddPFN(a, PFN("gsiftp://site.com/inputs/f.a","site"))
a &lt;- Metadata(a, "size", "1024")
diamond &lt;- AddFile(diamond, a)

# Add executables to the DAX-level replica catalog
e_preprocess &lt;- Executable(namespace="diamond", name="preprocess", version="4.0", os="linux", arch="x86_64")
e_preprocess &lt;- Metadata(e_preprocess, "size", "2048")
e_preprocess &lt;- AddPFN(e_preprocess, PFN("gsiftp://site.com/bin/preprocess","site"))
diamond &lt;- AddExecutable(diamond, e_preprocess)

e_findrange &lt;- Executable(namespace="diamond", name="findrange", version="4.0", os="linux", arch="x86_64")
e_findrange &lt;- AddPFN(e_findrange, PFN("gsiftp://site.com/bin/findrange","site"))
diamond &lt;- AddExecutable(diamond, e_findrange)

e_analyze &lt;- Executable(namespace="diamond", name="analyze", version="4.0", os="linux", arch="x86_64")
e_analyze &lt;- AddPFN(e_analyze, PFN("gsiftp://site.com/bin/analyze","site"))
diamond &lt;- AddExecutable(diamond, e_analyze)

# Add a preprocess job
preprocess &lt;- Job(e_preprocess)
preprocess &lt;- Metadata(preprocess, "time", "60")
b1 &lt;- File("f.b1")
b2 &lt;- File("f.b2")
preprocess &lt;- AddArguments(preprocess, list("-a preprocess","-T60","-i",a,"-o",b1,b2))
preprocess &lt;- Uses(preprocess, a, link=DAX3.Link$INPUT)
preprocess &lt;- Uses(preprocess, b1, link=DAX3.Link$OUTPUT, transfer=TRUE)
preprocess &lt;- Uses(preprocess, b2, link=DAX3.Link$OUTPUT, transfer=TRUE)
diamond &lt;- AddJob(diamond, preprocess)

# Add left Findrange job
frl &lt;- Job(e_findrange)
frl &lt;- Metadata(frl, "time", "60")
c1 &lt;- File("f.c1")
frl &lt;- AddArguments(frl, list("-a findrange","-T60","-i",b1,"-o",c1))
frl &lt;- Uses(frl, b1, link=DAX3.Link$INPUT)
frl &lt;- Uses(frl, c1, link=DAX3.Link$OUTPUT, transfer=TRUE)
diamond &lt;- AddJob(diamond, frl)

# Add right Findrange job
frr &lt;- Job(e_findrange)
frr &lt;- Metadata(frr, "time", "60")
c2 &lt;- File("f.c2")
frr &lt;- AddArguments(frr, list("-a findrange","-T60","-i",b2,"-o",c2))
frr &lt;- Uses(frr, b2, link=DAX3.Link$INPUT)
frr &lt;- Uses(frr, c2, link=DAX3.Link$OUTPUT, transfer=TRUE)
diamond &lt;- AddJob(diamond, frr)

# Add Analyze job
analyze &lt;- Job(e_analyze)
analyze &lt;- Metadata(analyze, "time", "60")
d &lt;- File("f.d")
analyze &lt;- AddArguments(analyze, list("-a analyze","-T60","-i",c1,c2,"-o",d))
analyze &lt;- Uses(analyze, c1, link=DAX3.Link$INPUT)
analyze &lt;- Uses(analyze, c2, link=DAX3.Link$INPUT)
analyze &lt;- Uses(analyze, d, link=DAX3.Link$OUTPUT, transfer=TRUE)
diamond &lt;- AddJob(diamond, analyze)

# Add dependencies
diamond &lt;- Depends(diamond, parent=preprocess, child=frl)
diamond &lt;- Depends(diamond, parent=preprocess, child=frr)
diamond &lt;- Depends(diamond, parent=frl, child=analyze)
diamond &lt;- Depends(diamond, parent=frr, child=analyze)

# Get generated diamond dax
WriteXML(diamond, stdout())

</pre>
</div>
</div><div class="navfooter">
<hr>
<table width="100%" summary="Navigation footer">
<tr>
<td width="40%" align="left">
<a accesskey="p" href="api.php">Prev</a> </td>
<td width="20%" align="center"><a accesskey="u" href="api.php">Up</a></td>
<td width="40%" align="right"> <a accesskey="n" href="dax_no_api.php">Next</a>
</td>
</tr>
<tr>
<td width="40%" align="left" valign="top">Chapter 16. API Reference </td>
<td width="20%" align="center"><a accesskey="h" href="index.php">Table of Contents</a></td>
<td width="40%" align="right" valign="top"> 16.3. DAX Generator without a Pegasus DAX API</td>
</tr>
</table>
</div><?php  
            pegasus_footer();
        ?>
