# @package      hubzero-mw2-common
# @file         win_user_account.py
# @author       Pascal Meunier <pmeunier@purdue.edu>
# @copyright    Copyright (c) 2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Based on previous work by Brian Rohler, Richard L. Kennell and Nicholas Kisseberth
#
# Copyright (c) 2015-2017 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

"""
For containers running under Virtuozzo, on Windows execution hosts.
This file was created for the NEES project (nees.org) and has not been 
maintained since NEES stopped using HUBzero middleware (circa 2015).  
It is provided for historical interest and in case someone is interested
in supporting Windows execution hosts.  Use at your own risk.
"""

import os
import time
from log import log
from errors import MaxwellError
from mw_common.constants import MYSQL_CONN_ATTEMPTS, VERBOSE
from mw_common.mw_db import MW_DB
  
class User_account():
  """ Get, manipulate and format account information for a given user.
    Allow root to act as that user by relinquishing privileges.
  """
  def __init__(self, user):
    self.user = user
    try:
      db = MW_DB(mysql_host, mysql_user, mysql_password, mysql_db, MYSQL_CONN_ATTEMPTS)
      row = db.getrow("""SELECT uidNumber,gidNumber,homeDirectory,loginShell
      FROM jos_xprofiles 
      WHERE username=%s""", (self.user))
      log("mysql_host = %s" % mysql_host)
      log("mysql_user = %s" % mysql_user)
      log("mysql_password = %s" % mysql_password)
      log("mysql_db = %s" % mysql_db)
      log("MYSQL_CONN_ATTEMPTS = %s" % MYSQL_CONN_ATTEMPTS)
    except StandardError:
      raise MaxwellError("Giving up: account info for '%s'." % self.user)

    db.db.close()

    if row is None:
      log("No database entry found for user %s." % self.user)
    else:
      self.uid = int(row[0])
      self.gid = int(row[1])
      self.homedir = "c:" + row[2]
      self.shell = row[3]
    if VERBOSE:
      log("Got account information for: '%s', uid: %d, gid: %d" % (self.user, self.uid, self.gid))

  def env(self, session_id, timeout):
    return [
      "HOME=\"%s\"" % self.homedir,
      "LOGNAME=\"%s\"" % self.user,
      "PATH=\"%s\"" % "c:/bin;c:/usr/bin;c:/usr/bin/X11;c:/sbin;c:/usr/sbin",
      "SESSION=\"%s\"" % session_id,
      "SESSIONDIR=\"%s\"" % self.session_dir(session_id),
      "TIMEOUT=\"%s\"" % str(timeout),
      "USER=\"%s\"" % self.user
    ]

  def passwd_entry(self):
    return "%s:!:%d:%d::%s:%s" % (self.user, self.uid, self.gid, self.homedir, self.shell)

  def groups(self, db):
    groups = []
    rows = db.getall("""
    SELECT jos_xgroups.cn FROM jos_xgroups_members JOIN jos_xgroups
    ON jos_xgroups_members.gidNumber = jos_xgroups.gidNumber
    WHERE jos_xgroups_members.uidNumber = %s""", (self.uid))
    return [ x[0] for x in rows ]
