# @package      hubzero-mw2-common
# @file         session.py
# @author       Pascal Meunier <pmeunier@purdue.edu>
# @copyright    Copyright (c) 2016-2017 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Based on previous work by Richard L. Kennell and Nicholas Kisseberth
#
# Copyright (c) 2016-2017 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

"""
Support for the middleware client to manipulate tool sessions and the MySQL table of the same name.
"""
import re
import time
from errors import SessionError, PrivateError
from constants import GEOM_REGEXP, SESSION_K, VERBOSE, ALPHANUM
from log import log, ttyprint
from host import Host
from display import Display
from support import genpasswd
from app import App
from zone import Zone

class Session:
  """Model a tool session.  Manipulate the associated SQL tables and guarantee their integrity,
  as well as system rules."""

  def __init__(self, sessnum, session_suffix="", overrides={}):
    self.session_suffix = session_suffix
    self.sessnum = int(sessnum)
    self.disp = None
    # Host object for execution host
    self.host = None
    self.app = None
    self.readonly = 'No'
    self.viewtoken = None
    self.sesstoken = None
    self.user = None
    # vncpass is a property of the display
    self.vncpass = None
    self.zone = None
    self.K = SESSION_K
    self.K.update(overrides)

  def get_zone(self, db):
    if self.zone is not None and self.zone !=0:
      return self.zone
    zone_id = db.getsingle("""
      SELECT zone_id FROM session
      WHERE sessnum=%s""", self.sessnum
      )
    if zone_id is None:
      zone_id = db.getsingle("""
        SELECT zone_id FROM sessionlog
        WHERE sessnum=%s""", self.sessnum
        )
      if zone_id is None:
        raise SessionError("No zone for session %d" % self.sessnum)
    self.zone = Zone.get_zone_by_id(db, zone_id)
    return self.zone

  def get_app(self, db, mysql_prefix):
    """Find application running in an existing session"""
    appname = db.getsingle("""
      SELECT appname FROM session
      WHERE sessnum=%s""", self.sessnum
      )
    if appname is None:
      raise SessionError("Unable to find session %d" % self.sessnum)
    self.app = App(db, appname, self.K["APP_K"], mysql_prefix)

  def get_viewperms(self, db, user):
    row = db.getrow("""
      SELECT vncpass, readonly, viewtoken FROM viewperm
      WHERE sessnum=%s
      AND viewuser=%s""",
      (self.sessnum, user))
    if row is None:
      raise SessionError("No viewperm entry for session %d and user %s" % (self.sessnum, user))
    self.vncpass = row[0]
    self.readonly = row[1]
    self.viewtoken = row[2]
    self.user = user

  def get_vncpass(self, db):
    """For remote sessions, retrieve vncpass from viewperm (same for all users)"""
    self.vncpass = db.getsingle("""
      SELECT vncpass FROM viewperm WHERE sessnum=%s LIMIT 1""",
      (self.sessnum))
    if self.vncpass is None:
      raise SessionError("No viewperm entry for session %d" % self.sessnum)

  def del_session(self, db, reason):
    """Used to recover from aborted session start.  Display is managed separately
    because no display may be associated with the session yet.  This is also why
    calling stop_session() is not appropriate.
    Get a row from session table and delete it.
    get lock to prevent race conditions and several instances trying to
    stop a session due to a user frantically clicking
    If the session has been deleted already, that signals that another
    instance is taking care of it.  First we must save the info we'll need.
    the lock is to prevent two instances from getting the row and so thinking
    that "they're it".
    """
    db.lock("stop_session")
    if self.K["ZONE_SUPPORT"]:
      row = db.getrow("""
        SELECT username,remoteip,exechost,dispnum,timeout,start,appname,
          TIME_TO_SEC(TIMEDIFF(NOW(),start)) AS walltime, zone_id
        FROM session WHERE sessnum=%s""", self.sessnum
      )
    else:
      row = db.getrow("""
        SELECT username,remoteip,exechost,dispnum,timeout,start,appname,
          TIME_TO_SEC(TIMEDIFF(NOW(),start)) AS walltime
        FROM session WHERE sessnum=%s""", self.sessnum
      )
    db.c.execute("""DELETE FROM session WHERE sessnum=%s""", self.sessnum)
    db.unlock("stop_session")
    if row is None:
      return None# we've lost the race, session is already being deleted or has been

    log("Stopping %d for reason: %s" % (self.sessnum, reason))
    # finish deleting dependent tables
    # Note: added "on delete cascade" in schema to guarantee database integrity
    # so this code could be deleted once we're sure all db schemas have been updated.
    db.c.execute("""DELETE FROM viewperm WHERE sessnum=%s""", self.sessnum)
    db.c.execute("""DELETE FROM fileperm WHERE sessnum=%s""", self.sessnum)
    return row

  def get_job_count(self, db):
    """Return the number of running jobs in this session"""
    count = db.getsingle("""select count(*) FROM job WHERE sessnum=%s
      AND heartbeat > DATE_ADD(NOW(), INTERVAL -1 DAY)""", self.sessnum)
    if count is None:
      return 0
    return count

  def stop_session(self, db, reason):
    """Stop the session.
    This may be called after a "notify" message from execution host, by the CMS (reason "user")
    or by the expiration daemon (reason "timeout").  The execution host always sends a notify message
    at the end (reason "exited").  Only set status "absent" when getting the reason "exited".

    Matching display must exist.  Session truly ends only when display is put in "absent" state.
    If display is in "starting" state, wait for a while
    move session information to sessionlog if not already done -- this means session is stopping
    - if display status is "used":
      - set display status to "stopping" if not already stopping
    When called with any reason other than "exited":
      - set display status to "stopping"
      - send display stop command to execution host
    When called with reason "exited":
      - set display status to "absent"
    """
    tries = 0
    while True:
      db.lock("stop_session")
      disp = Display.from_sessnum(db, self.sessnum)
      if disp is None:
        # no matching display...  Perhaps this is a remote venue session?
        break
      if disp.status != 'starting':
        break
      db.unlock("stop_session")
      log("Display for session: %d is still starting, can't stop it yet" % (self.sessnum))
      time.sleep(5)
      tries += 1
      if tries > 99:
        # starting failed -- program was killed, etc.
        disp.broken(db)
        log("Display %d on host %s stuck in starting state, trying to stop it" % \
                           (disp.dispnum, disp.host.hostname))
        db.lock("stop_session")
        if reason != 'exited':
          disp.tell_stop(db) # does not change display state
          disp = None
        break

    # get session information
    if self.K["ZONE_SUPPORT"]:
      row = db.getrow("""
        SELECT username,remoteip,exechost,dispnum,timeout,start,appname,
          TIMESTAMPDIFF(SECOND, start, NOW()) AS walltime, zone_id
        FROM session WHERE sessnum=%s""", self.sessnum
      )
    else:
      row = db.getrow("""
        SELECT username,remoteip,exechost,dispnum,timeout,start,appname,
          TIMESTAMPDIFF(SECOND, start, NOW()) AS walltime
        FROM session WHERE sessnum=%s""", self.sessnum
      )
    if row is None:
      db.unlock("stop_session")
      # session is already stopping or stopped.
      count = db.getsingle("""select count(*) FROM sessionlog WHERE sessnum=%s""", self.sessnum)
      if count is None:
        raise SessionError("No such session, and no sessionlog entry for %d." % self.sessnum)
      if reason != 'exited':
        raise PrivateError("sessionlog entry for session %d has already been processed." % self.sessnum)
      # notify call, container has stopped
      # what if a "renotify" command is issued, and the display has been reused for another session?
      # Check state
      if disp is not None:
        disp.absent(db)
      return

    user = row[0]
    remoteip = row[1]
    exechost = row[2]
    dispnum = int(row[3])
    timeout = row[4]
    start = row[5]
    appname = row[6]
    walltime = row[7]
    if self.K["ZONE_SUPPORT"]:
      zone_id = int(row[8])
      z = Zone.get_zone_by_id(db, zone_id)
      if not z.is_local(db):
        # tell remote site that we want to stop this session
        z.tell(db, ["stop", str(self.sessnum)] )

    # move session info to sessionlog
    if disp is not None:
      disp.stopping(db)
      if dispnum != disp.dispnum:
        log("mismatched display number in session and display tables for session %d." % self.sessnum)

    db.c.execute("""DELETE FROM session WHERE sessnum=%s""", self.sessnum)

    status = 0
    if reason == 'timeout':
      status = 65535
      walltime -= timeout

    if self.K["ZONE_SUPPORT"]:
      db.c.execute("""
        INSERT INTO sessionlog
          (sessnum,username,remoteip,exechost,dispnum,start,appname,status,walltime, zone_id)
        VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s, %s)""" ,
          (self.sessnum, user, remoteip, exechost, dispnum, start, appname, status, "%f" %walltime, zone_id)
      )
    else:
      db.c.execute("""
        INSERT INTO sessionlog
          (sessnum,username,remoteip,exechost,dispnum,start,appname,status,walltime)
        VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s)""" ,
        (self.sessnum, user, remoteip, exechost, dispnum, start, appname, status, "%f" %walltime)
      )

    db.unlock("stop_session")

    log("Stopped %d for reason: %s" % (self.sessnum, reason))
    # finish deleting dependent tables
    # Note: added "on delete cascade" in schema to guarantee database integrity
    # so this code could be deleted once we're sure all db schemas have been updated.
    db.c.execute("""DELETE FROM viewperm WHERE sessnum=%s""", self.sessnum)
    db.c.execute("""DELETE FROM fileperm WHERE sessnum=%s""", self.sessnum)

    if disp is None:
      db.c.execute("""DELETE FROM view WHERE sessnum=%s""", self.sessnum)
    else:
      if reason == 'exited':
        # notify call, container has stopped already
        # check that no views are left before making the display "absent" and available for restarting
        # this matters especially for Windows execution hosts
        wait_total = 0
        printed_notice = 0
        views_remaining = 1
        # 3600 = 1 hour
        # if database is restarted or we lose connection, we're in trouble, display will never be set to absent...
        while wait_total < 3600 and views_remaining > 0:
          views_remaining = db.getsingle("""SELECT COUNT(*) FROM view WHERE sessnum=%s""", self.sessnum)
          if views_remaining is None:
            # if we lost the database connection, don't wait anymore
            views_remaining = 0
          if views_remaining > 0:
            if printed_notice == 0:
              log("Continue %d checking remaining view." % self.sessnum)
              printed_notice = 1
            time.sleep(5)
            wait_total += 5
        if wait_total >= 3600:
          db.c.execute("""DELETE FROM view WHERE sessnum=%s""", self.sessnum)
        disp.absent(db)
      else:
        # wait for the notify to put display in absent state
        disp.tell_stop(db) # does not change display state

  @staticmethod
  def create_indb_zone(db, user, ip, app, session_suffix, zone, SESSION_CONF):
    """Create a session row in the database.  Return the autoincremented number."""
    if zone.zone_id == 0:
      raise SessionError("zone_id is zero")
    timeout = app.timeout
    if timeout == 0:
      timeout = self.K["APP_K"]["TIMEOUT_DEFAULT"]
    sesstoken = genpasswd(32, ALPHANUM)
    db.c.execute("""
      INSERT INTO session
      (username,remoteip,exechost,dispnum,start,accesstime,appname,sessname,sesstoken,timeout, zone_id)
      VALUES(%s, %s, %s, 0, now(),now(), %s, %s, %s, %s, %s)""" ,
      (user, ip, zone.get_host(db).hostname, app.appname, app.appfullname, sesstoken, timeout, zone.zone_id)
    )
    sess_str = db.getsingle("""SELECT last_insert_id()""", ())
    if sess_str is None:
      raise SessionError("Unable to get session id after insert")
    sess = Session(sess_str, session_suffix, SESSION_CONF)
    assert(sess.sessnum != 0)
    sess.sesstoken = sesstoken
    sess.user = user
    sess.app = app
    sess.zone = zone
    return sess

  @staticmethod
  def create_indb(db, user, ip, app, timeout, session_suffix, SESSION_CONF):
    """Create a session row in the database.  Return the autoincremented number."""
    timeout = app.timeout
    if timeout == 0:
      timeout = self.K["APP_K"]["TIMEOUT_DEFAULT"]
    sesstoken = genpasswd(32, ALPHANUM)
    db.c.execute("""
      INSERT INTO session
      (username,remoteip,exechost,dispnum,start,accesstime,appname,sessname,sesstoken,timeout)
      VALUES(%s, %s, '', 0, now(),now(), %s, %s, %s, %s)""" ,
      (user, ip, app.appname, app.appfullname, sesstoken, timeout)
    )
    sess_str = db.getsingle("""SELECT last_insert_id()""", ())
    if sess_str is None:
      raise SessionError("Unable to get session id after insert")
    sess = Session(sess_str, session_suffix, SESSION_CONF)
    assert(sess.sessnum != 0)
    sess.sesstoken = sesstoken
    sess.user = user
    sess.app = app
    return sess

  @staticmethod
  def integrity_check(db):
    """Look for rows without user names, the CMS creates those rows when there are
    problems with maxwell"""

    db.c.execute("""
      DELETE FROM session WHERE username = ""
      """)
    nrows = db.getsingle("""SELECT ROW_COUNT()""", ())
    if nrows > 0:
      log("cleaned up %d rows in session table with empty user names" % nrows)

    rows = db.getall("""
      SELECT fileperm.sessnum
      FROM fileperm LEFT OUTER JOIN session ON fileperm.sessnum = session.sessnum
      WHERE session.sessnum is NULL""", ())
    for row in rows:
      db.c.execute("""
        DELETE FROM fileperm WHERE sessnum = %s""", row[0]
      )
    nrows = db.getsingle("""SELECT ROW_COUNT()""", ())
    if nrows > 0:
      log("cleaned up %d rows in fileperm table without matching sessions" % nrows)

    rows = db.getall("""
      SELECT viewperm.sessnum
      FROM viewperm LEFT OUTER JOIN session ON viewperm.sessnum = session.sessnum
      WHERE session.sessnum is NULL""", ())
    for row in rows:
      db.c.execute("""
        DELETE FROM viewperm WHERE sessnum = %s""", row[0]
      )
    nrows = db.getsingle("""SELECT ROW_COUNT()""", ())
    if nrows > 0:
      log("cleaned up %d rows in viewperm table without matching sessions" % nrows)

  def set_exec(self, db, disp):
    """ disp is a Display object"""
    self.disp = disp
    self.host = disp.host
    db.c.execute("""
      UPDATE session SET exechost=%s,dispnum=%s
      WHERE sessnum=%s""",
      (self.host.hostname, disp.dispnum, self.sessnum))

  def get_disp(self,db):
    self.disp = disp
    
  def set_accesstime(self, db):
    """Set session accesstime to NOW"""
    db.c.execute("""
      UPDATE session SET accesstime=NOW()
      WHERE sessnum=%s""",
      (self.sessnum))

  def unshare(self, db, viewuser):
    db.c.execute("""DELETE FROM viewperm WHERE sessnum = %s and viewuser = %s""", (self.sessnum, viewuser))
    db.c.execute("""DELETE FROM fileperm WHERE sessnum = %s and fileuser = %s""", (self.sessnum, viewuser))

  def set_viewperm(self, db, viewtoken, cookie, vncpass=None):
    self.viewtoken = viewtoken
    # remote venue: display is remote so vncpass is given to us as parameter
    if vncpass is not None:
      self.vncpass = vncpass
    elif self.vncpass is None:
      if self.disp is None:
        raise SessionError("self.disp is None and no vncpassword given")
      else:
        self.vncpass = self.disp.vncpass
    if self.app is None:
      geo = db.getsingle("""SELECT geometry from viewperm where sessnum=%s""", self.sessnum)
      if geo is None:
        geo="800x600"    
    else:
      geo = self.app.geometry
    db.c.execute("""
      INSERT INTO viewperm(sessnum,viewuser,viewtoken,geometry,vncpass,readonly)
      VALUES(%s, %s, %s, %s, %s, %s) ON DUPLICATE KEY UPDATE viewtoken=%s,readonly=%s""",
      (self.sessnum, self.user, viewtoken, geo, self.vncpass, self.readonly, viewtoken, self.readonly))
    db.c.execute("""
      INSERT INTO fileperm(sessnum,fileuser,cookie)
      VALUES(%s,%s,%s) ON DUPLICATE KEY UPDATE cookie=%s
      """ , (self.sessnum, self.user, cookie, cookie))
    if self.K["NOVNC"]:
      # tell the execution host what the viewtoken is
      status = 1
      count = 0
      while status != 0 and count < 5:
        count += 1
        try:
          status = self.host.service("set_viewtoken", [self.disp.dispnum, viewtoken])
        except StandardError:
          continue
      if count >= 5:
        raise SessionError("unable to invoke command for session %s (app = %s, user= %s)",
                           self.sessnum, self.app.appname, self.user)

  def applet_html(self, force_readonly, hub_url, db):
    """Send JSON to the web server.  The CMS will issue HTML.  
    This is an internal function called from create_session and view_applet."""

    # we provide a different applet path depending on whether the execution host is Windows or Linux
    if self.host is None:
      self.get_sessionhost(db)
    if self.host is None:
      applet_path = ""
    else:
      if self.host.is_windows: # Windows execution host
        applet_path = self.K["JAVA_APPLET_WINPATH"]
      else:
        applet_path = self.K["JAVA_APPLET_PATH"]

    if force_readonly:
      # this is vulnerable, easily overridden
      self.readonly = 'Yes'

    if re.match(GEOM_REGEXP, self.app.geometry) is None:
      raise SessionError("Bad geometry for appname='%s'" % self.app.appname)
    (width, height) = self.app.geometry.split('x')

    # 2 different proxy servers, for Java and for "NoVNC" aka WebSocket (wsproxy)
    # Java proxy runs on the web server
    if self.K["ZONE_SUPPORT"]:
      # get information on proxy server from MySQL
      javaproxy = self.get_zone(db).get_host(db).hostname
      # missing support for this in zone table
      # wsproxy = self.get_zone(db).get_proxy(db)
      javaport = self.K["VNC_PROXY_PORT"]
      wsproxy = "vncproxy." + javaproxy
      wsport = self.K["WS_PROXY_PORT"]
    else:
      # get information on proxy server from middleware configuration
      javaproxy = self.K["VNC_PROXY_SERVER"]
      javaport = self.K["VNC_PROXY_PORT"]
      wsproxy = self.K["WS_PROXY_SERVER"]
      wsport = self.K["WS_PROXY_PORT"]

    # Web proxy for applications like Jupyter, etc.
    # Check the config to see if there is a value for the params field.
    # If the params field contains 'weber=' then redirect to the URL.
    redirect_url = ''
    weber_auth = ''
    tooltable = self.app.mysql_prefix + self.app.K["TOOL_TABLE"]
    params = db.getsingle('SELECT params FROM ' + tooltable +
                          ' WHERE instance=%s AND state IS NOT NULL',
                          self.app.appname)
    # params as JSON, change discussed with DSK October 20, 2016
    # import json
    # p = json.loads(params)
    # if p.has_key('weber'):
    #   if p['weber'] is True:
    if "weber=" in str(params):
      import os
      # construct redirect_url from /etc/mw-proxy/front-proxy.conf
      FRONT_PROXY_CONFIG_FILE = '/etc/mw-proxy/front-proxy.conf'
      # FRONT_LISTEN_PORT = 443
      # FRONT_LISTEN_SSL = True
      # FRONT_LISTEN_HOST = ''
      proxy_vars = {}
      if os.path.isfile(FRONT_PROXY_CONFIG_FILE):
        # read values into proxy_vars, otherwise "global name 'FRONT_...' is not defined"
        # because they go into locals but can't access them
        execfile(FRONT_PROXY_CONFIG_FILE, proxy_vars)
      else:
        log("front proxy config file is not readable")
      try:
        if proxy_vars['FRONT_LISTEN_SSL']:
          prefix = "https://"
        else:
          prefix = "http://"
      except KeyError:
        prefix = "https://"
      try:
        if proxy_vars['FRONT_LISTEN_HOST'] == '':
          # guess it from hostname of hub_url and prepend https://proxy.
          host = hub_url
          idx = host.find('://')
          if idx != -1:
            host = host[idx+3:]
          idx = host.find('/')
          if idx != -1:
            host = host[0:idx]
          prefix = prefix + "proxy." + host
        else:
          # log("front listen host is '%s'" % str(proxy_vars['FRONT_LISTEN_HOST']))
          prefix = prefix + proxy_vars['FRONT_LISTEN_HOST']
      except KeyError:
        # duplicated code...
        log("FRONT_LISTEN_HOST not found, guessing from hub_url")
        # guess it from hostname of hub_url and prepend https://proxy.
        host = hub_url
        idx = host.find('://')
        if idx != -1:
          host = host[idx+3:]
        idx = host.find('/')
        if idx != -1:
          host = host[0:idx]
        prefix = prefix + "proxy." + host
      try:
        if proxy_vars['FRONT_LISTEN_PORT'] == 443:
          redirect_url = prefix
        else:
          redirect_url = prefix + ":%d" % proxy_vars['FRONT_LISTEN_PORT']
      except KeyError:
          redirect_url = prefix
      log("redirect_url is '%s'" % str(redirect_url ))
      row = db.getrow(
        """SELECT fileperm.cookie,display.dispnum,fileperm.fileuser
           FROM fileperm,display
           WHERE fileperm.sessnum = display.sessnum
           AND display.sessnum=%s""", self.sessnum)
      cookie = row[0]
      dispnum = int(row[1])
      user = row[2]
      redirect_url = redirect_url + "/weber/%d/%s/%d/" % (self.sessnum,cookie,dispnum)

      rows = db.getall("""SELECT sessnum,vncpass FROM viewperm
                          WHERE viewuser=%s""", user)
      for s,v in rows:
        pair = str(s) + ':' + str(v)
        if weber_auth == '':
          weber_auth = pair
        else:
          weber_auth = weber_auth + ',' + pair

    import json
    appvars = {}
    appvars["debug"] = 0
    appvars["connect"] = """vncsession:%s""" % self.viewtoken
    appvars["encoding"] = "ZRLE"
    appvars["width"] = int(width)
    appvars["height"] = int(height)
    appvars["encpassword"] = self.vncpass
    appvars["port"] = javaport
    appvars["show_controls"] = False
    if self.host.is_windows: # Windows execution host
      appvars["show_local_cursor"] = True
    else:
      appvars["show_local_cursor"] = False
    # Note input "readonly=1" is still a string, not a proper Python integer so ("1" == 1) is false
    appvars["readonly"] = (force_readonly == "1")
    if applet_path != "":
      appvars["archive"] = "%s/%s" % (hub_url, applet_path)
    appvars["host"] = javaproxy
    appvars["wsproxy_port"] = wsport
    appvars["wsproxy_encrypt"] = True
    appvars["wsproxy_host"] = wsproxy
    if redirect_url != '':
      appvars["redirect_url"] = redirect_url
    if weber_auth != '':
      appvars["weber_auth"] = weber_auth
    ttyprint(json.dumps(appvars))
    self.set_accesstime(db)
    return

  def invoke_command(self, appopts, params):
    """Invoke the application command for the session creation."""

    # timeout is now handled by the expire-sessions.py daemon. Essentially this is now lint
    timeout = 0

    comm = self.app.command
    if comm.split() == []:
      raise SessionError("Command is empty")


    # there used to be also a "stream" mode that wrote log files locally
    # startxapp <user> <sessionid> <timeout> <dispnum> <command>
    args = [self.user, self.sessname(), str(timeout), str(self.disp.dispnum), "notify"]

    if params is not None and params != "":
      args += ['params']

    args.append(comm)

    # pass state from web server to application, e.g., file name
    if appopts != "" and appopts is not None: # "-A"
      args += ['-A', appopts]
      if VERBOSE:
        log("appopts is '%s'" % str(appopts))
    status = 1
    count = 0
    while status != 0 and count < 5:
      count += 1
      # service(self, service_cmd, args, feed="")
      try:
        status = self.host.service("startxapp", args)
      except StandardError:
        continue
    if count >= 5:
      raise SessionError("unable to invoke command for session %s (app = %s, user= %s)",
                         self.sessnum, self.app.appname, self.user)

  def sessname(self):
    return "%d%s" % (self.sessnum, self.session_suffix)

  def get_exechost(self, db):
    if self.K["ZONE_SUPPORT"]:
      if self.get_zone(db).is_local(db):
        ex = db.getsingle("""SELECT exechost FROM sessionlog WHERE sessnum=%s""", self.sessnum)
        if ex is None or ex == "":
          raise SessionError("local exechost not present in sessionlog '%d', zone %s" % (self.sessnum, self.get_zone(db).zone))
        self.host = Host(db, ex)
      else:
        log("zone %s is not local, getting master host name" % self.get_zone(db).zone)
        # zone has a basichost, we're getting a full host here.
        self.host = Host(db, self.zone.get_host(db).hostname, {"KEY_PATH": self.zone.get_host(db).key_path})
    else:
      ex = db.getsingle("""SELECT exechost FROM sessionlog WHERE sessnum=%s""", self.sessnum)
      if ex is None:
        # return None to support jersey_start procedure, instead of raising an exception
        return None
      self.host = Host(db, ex)

  def get_sessionhost(self, db):
    """Find the execution host or master host name for a session"""
    ex = db.getsingle("""SELECT exechost FROM session WHERE sessnum=%s""", self.sessnum)
    if ex is None or ex == "":
      self.get_exechost(db)
      return
    self.host = Host(db, ex)

  def get_stats(self, db, get_stats_retries, get_stats_delay):
    """Get session stats (.err and .out files) after asynchronous notifier."""
    #
    # If the session was terminated by the user, stop_session has already been
    # called and the session table doesn't have this entry anymore.
    # Otherwise we can get it from session.  Either source should have the
    # same exechost.
    #
    import os
    self.get_exechost(db)
    if VERBOSE:
      log("key path is '%s' for host '%s'" % (self.host.key_path,self.host.hostname))

    # SCP session .err and .out files
    if self.K["ZONE_SUPPORT"]:
      if self.zone.is_local(db):
        # get it from /var/log/mw-service/open-sessions
        log("zone %s is type %s, tests as local" % (self.get_zone(db).zone, self.zone.type))
        if self.host.is_windows:
          src_out = self.host.scp_format('%s/%d.out' % (self.K["WIN_S_WEBSERVER_LOG_PATH"], self.sessnum))
          src_err = self.host.scp_format('%s/%d.err' % (self.K["WIN_S_WEBSERVER_LOG_PATH"], self.sessnum))
        else:
          src_out = self.host.scp_format("%s/%d.out" % (self.K["SERVICE_LOG_PATH"], self.sessnum))
          src_err = self.host.scp_format("%s/%d.err" % (self.K["SERVICE_LOG_PATH"], self.sessnum))
      else:
        # get it from /var/log/mw/sessions
        log("zone %s is not local, scp from master path" % self.get_zone(db).zone)
        src_out = self.host.scp_format("%s/%d.out" % (self.K["LOG_PATH"], self.sessnum))
        src_err = self.host.scp_format("%s/%d.err" % (self.K["LOG_PATH"], self.sessnum))
    else:
      if self.host.is_windows:
        src_out = self.host.scp_format('%s/%d.out' % (self.K["WIN_S_WEBSERVER_LOG_PATH"], self.sessnum))
        src_err = self.host.scp_format('%s/%d.err' % (self.K["WIN_S_WEBSERVER_LOG_PATH"], self.sessnum))
      else:
        src_out = self.host.scp_format("%s/%d.out" % (self.K["SERVICE_LOG_PATH"], self.sessnum))
        src_err = self.host.scp_format("%s/%d.err" % (self.K["SERVICE_LOG_PATH"], self.sessnum))
    if VERBOSE:
      log("scp from %s to %s" % (src_err, self.K["LOG_PATH"]))
      log("scp from %s to %s" % (src_out, self.K["LOG_PATH"]))
    retry = 0
    got_them = False
    while retry < get_stats_retries and not got_them:
      # scp(self, src, dest)
      try:
        self.host.scp(src_err, self.K["LOG_PATH"])
        self.host.scp(src_out, self.K["LOG_PATH"])
      except PrivateError, pe:
        retry += 1
        log("%s" % pe)
        time.sleep(get_stats_delay)
      else:
        got_them = True
        os.chmod(self.K["LOG_PATH"] + "/%d.out" % self.sessnum, 0640)
        os.chmod(self.K["LOG_PATH"] + "/%d.err" % self.sessnum, 0640)

    # Call maxwell_service to "purgeoutputs"
    # this deletes the files we just scped.
    # don't delete if scp failed
    if got_them:
      if self.K["ZONE_SUPPORT"]:
        if self.get_zone(db).is_local(db):
          deleted_them = False
        else:
          deleted_them = True
      else:
        deleted_them = False
      while retry < get_stats_retries and not deleted_them:
        status = self.host.service("purgeoutputs", ["%d" % self.sessnum])
        if status != 0:
          retry += 1
          log("Purgelogs status for %d was %s." % (self.sessnum, status))
          time.sleep(get_stats_delay)
        else:
          deleted_them = True
    if retry == get_stats_retries:
      log("get_stats: can't purge '%s':'%d'" % (self.host.hostname, self.sessnum))


  def parse_time(self, db, line, zone_id=0):
    """Parse a time specification for a subtask."""
    #log("parse_time")

    event = ""
    job = 0
    start = 0.0
    wall = 0.0
    cpu = 0.0
    ncpus = 1
    venue = ""
    status = 0

    for item in line.split():
      try:
        if item == 'MiddlewareTime:':
          continue

        (k, v) = item.split('=')
        #log("parsing %s=%s" % (k,v))

        if k == 'job':
          job = int(v)
        elif k == 'event':
          event = v
        elif k == 'start':
          start = float(v)
        elif k == 'walltime':
          wall = float(v)
        elif k == 'cputime':
          cpu = float(v)
        elif k == 'ncpus':
          ncpus = int(v)
        elif k == 'venue':
          venue = v
        elif k == 'status':
          status = int(v)
        else:
          log("Unknown tag in parse_time: %s" % k)
        continue

      except StandardError:
        log("Bad item in parse_time: %s in %s" % (item, line))
        return

    if job == 0 or event == '':
      log("parse_time: bad format: %s" % line)
      return

    # start is a datetime in database
    # we're getting the value from the sessionlog table and adding to it
    start = "date_add(start, interval %f second_microsecond)" % start
    wall = "%f" %  wall
    cpu = "%f" % cpu
    row = db.getrow("""
      SELECT start, sessnum FROM joblog
      WHERE sessnum=%s AND job=%s AND event=%s AND venue=%s""",
        (self.sessnum, job, event, venue))

    if row is None:
      db.c.execute("""
        INSERT INTO
        joblog(sessnum,job,event,start,walltime,cputime,ncpus,status,venue)
        SELECT sessnum, %s, %s, """ + start + """, %s, %s, %s, %s, %s
        FROM sessionlog WHERE sessnum=%s""",
        (job, event, wall, cpu, ncpus, status, venue, self.sessnum))
    else:
      if row[0] == '0000-00-00 00:00:00' or row[0] is None:
        print "fixing start for a job in session %d" % row[1]
        db.c.execute("""
          UPDATE joblog SET start = (SELECT """ + start + """
          FROM sessionlog WHERE sessnum=%s) WHERE sessnum=%s AND job=%s AND event=%s AND venue=%s""",
          (self.sessnum, self.sessnum, job, event, venue))
      else:
        print row


  def sum_viewtime(self, db):
    """Sum the view time for a session."""
    debug_viewtime = 0
    views_remaining = 1
    printed_notice = 0
    wait_total = 0
    # 5000 sec is more than an hour and a half
    # if database is restarted or we lose connection, we're in trouble
    while wait_total < 5000 and views_remaining > 0:
      views_remaining = db.getsingle("""SELECT COUNT(*) FROM view WHERE sessnum=%s""", self.sessnum)
      if views_remaining is None:
        raise SessionError("Can't get count from view")
      if views_remaining > 0:
        if printed_notice == 0:
          log("Continue %d checking remaining view." % self.sessnum)
          printed_notice = 1
        time.sleep(5)
        wait_total += 5

    if printed_notice:
      log("WaitedOn %d a total of %d seconds for remaining view." % (self.sessnum, wait_total))

    if wait_total == 5000:
      log("Views never terminated for %d, returning 0 (sum_viewtime)" % self.sessnum)
      return 0

    arr = db.getall("""SELECT unix_timestamp(time),duration,remoteip
                     FROM viewlog WHERE sessnum=%s ORDER BY time""", self.sessnum)

    total_duration = 0
    prevfinish = 0
    for row in arr:
      start = int(row[0])
      duration = float(row[1])
      finish = start+duration
      remoteip = row[2]

      if duration <= 0:
        continue

      if prevfinish < start:
        prevfinish = finish
        total_duration += duration
      else:
        if finish < prevfinish:
          if debug_viewtime:
            log("Skipping next record because it is contained in previous.")
        else:
          overlap = prevfinish - start
          if debug_viewtime:
            log("Next record overlaps with previous by %f" % overlap)
          tmp_duration = duration - overlap
          total_duration += tmp_duration
          prevfinish = finish

      if debug_viewtime:
        log("%d: %f %f %s" % (start, duration, finish, remoteip))

    return total_duration


  def process_stats(self, db, zone_id=0):
    """Get the statistics from the command."""
    import os
    try:
      sess_file = open("%s/%d.err" % (self.K["LOG_PATH"], self.sessnum))
    except IOError:
      log("Session log file is missing!")
      return
    #
    # Get fields that are already in the session log...
    #
    row = db.getrow("""SELECT walltime,status FROM sessionlog
                      WHERE sessnum=%s""", self.sessnum)
    rtime = float(row[0])
    status = int(row[1])

    utime = 0.0
    stime = 0.0
    reported_timeout = 0
    while 1:
      line = sess_file.readline()
      if line == '':
        break

      if line.startswith("MiddlewareTime: "):
        #  This inserts (job, event, start, wall, cpu, ncpus, status, venue, self.sessnum))
        self.parse_time(db, line, zone_id)
        continue

      # possibly lint since timeouts aren't handled here anymore
      if line.startswith("Timeout_Value: "):
        #log("Timeout status for %s: %s" % (sess,line))
        try:
          reported_timeout = int(line.split(' ')[1])
          #log("Timeout is %d" % reported_timeout)
        except StandardError:
          continue

      if status == 0:
        if line.startswith("_Status: "): # possibly lint
          #log("Exit status for %s: %s" % (sess,line))
          try:
            status = int(line.split(' ')[1])
            #log("Status is %d" % status)
          except StandardError:
            continue

      if status != 65535:
        if line.startswith("real "):
          try:
            rtime = float(line.split(' ')[1])
          except StandardError:
            rtime = 0
            continue

      if line.startswith("user "):
        try:
          utime = float(line.split(' ')[1])
        except StandardError:
          utime = 0
          continue

      if line.startswith("sys "):
        try:
          stime = float(line.split(' ')[1])
        except StandardError:
          stime = 0
          continue

    # If session timed out, subtract the timeout value from the
    # wall time.  This gives a realistic value for how long the
    # user was actually looking at the application.
    #
    # Also set the status value to something that indicates timeout.
    # 65535 is an impossible exit value.
    if reported_timeout > 0:
      rtime -= reported_timeout
      status = 65535

    if rtime < 0:
      log("Session %d rtime(%f) < 0" % (self.sessnum, rtime))
      rtime = 0

    if utime + stime < 0.01:
      log("ERROR: Session %d consumed no CPU." % self.sessnum)

    vtime = self.sum_viewtime(db)

    cmd = """UPDATE sessionlog
          SET walltime=%f,viewtime=%f,cputime=%f,status=%d
            WHERE sessnum=""" % (rtime, vtime, utime+stime, status)
    db.c.execute(cmd + "%s", self.sessnum)

    row = db.getrow("""SELECT walltime,cputime,status FROM joblog
      WHERE sessnum=%s AND job=0 AND event='application'""", (self.sessnum))
    if row is None:
	    cmd = """
	      INSERT INTO
	      joblog(sessnum,job,event,start,walltime,cputime,ncpus,status,venue)
	      SELECT sessnum, 0, "application", start, %f, %f, 1,    %d, ""
	      FROM sessionlog WHERE sessnum=""" % (rtime, utime+stime, status)
	    db.c.execute(cmd + "%s", self.sessnum)
    #else:
    #  log("joblog already inserted")
    #  if str(rtime) != str(row[0]):
    #    log("walltime already logged as %s but now recalculated at %s" %(str(row[0]), str(rtime) ))
    #  if str(utime+stime) != str(row[1]):
    #    log("cputime already logged as %s but recalculated as %s" %(str(row[1]), str(utime+stime) ))
    #  if str(status) != str(row[2]):
    #    log("status already logged as %s but now recalculated at %s" %(str(row[2]), str(status) ))

    # check the size of the log file and delete it if larger than 1 MB
    if os.path.getsize("%s/%d.out" % (self.K["LOG_PATH"], self.sessnum)) > 1000000:
      os.unlink("%s/%d.out" % (self.K["LOG_PATH"], self.sessnum))
      log("deleted %d.out because it was too large" % self.sessnum)
    if os.path.getsize("%s/%d.err" % (self.K["LOG_PATH"], self.sessnum)) > 1000000:
      os.unlink("%s/%d.err" % (self.K["LOG_PATH"], self.sessnum))
      log("deleted %d.err because it was too large" % self.sessnum)



  @staticmethod
  def make(db, user, ip, app, session_suffix, sessnum, sesstoken, SESSION_CONF):
    """Create a session row in the database, using the provided session id.
    """
    timeout = app.timeout
    # sesstoken = genpasswd(32, ALPHANUM)
    db.c.execute("""
      INSERT INTO session
      (username,remoteip,exechost,dispnum,start,accesstime,appname,sessname,sesstoken,timeout,sessnum)
      VALUES(%s, %s, '', 0, now(),now(), %s, %s, %s, %s, %s)""" ,
      (user, ip, app.appname, app.appfullname, sesstoken, timeout, sessnum)
    )
    sess = Session(sessnum, session_suffix, SESSION_CONF)
    sess.sesstoken = sesstoken
    sess.user = user
    sess.app = app
    return sess

